<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\PaymentModule;
use Proxim\Util\ArrayUtils;

class Authorize extends PaymentModule
{
    /**
    * Authorize configuration
    */
    const AUTHORIZE_ENABLED = 'AUTHORIZE_ENABLED';
    const AUTHORIZE_MODE = 'AUTHORIZE_MODE';
    const AUTHORIZE_API_LOGIN_ID = 'AUTHORIZE_API_LOGIN_ID';
    const AUTHORIZE_TRANSACTION_KEY = 'AUTHORIZE_TRANSACTION_KEY';

    public function __construct()
    {
        $this->name = 'authorize';
        $this->icon = 'fa fa-credit-card';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Authorize.net';
        $this->description = 'Set up Authorize.net as a payment method and enable users to pay without leaving your website';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    {
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $authorizeConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::AUTHORIZE_ENABLED,
                self::AUTHORIZE_MODE,
                self::AUTHORIZE_API_LOGIN_ID,
                self::AUTHORIZE_TRANSACTION_KEY,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'authorizeConfiguration' => $authorizeConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateAuthorizeSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        Configuration::updateValue('PAYMENT_METHOD_DEFAULT', ArrayUtils::has($payload, 'default_method_enabled') ? $this->name : '' , false, $site_id);
        Configuration::updateValue(self::AUTHORIZE_ENABLED, ArrayUtils::has($payload, 'authorize_enabled') ? true : false , false, $site_id);
        Configuration::updateValue(self::AUTHORIZE_MODE, ArrayUtils::get($payload, 'authorize_mode'), false, $site_id);
        Configuration::updateValue(self::AUTHORIZE_API_LOGIN_ID, ArrayUtils::get($payload, 'authorize_api_login_id'), false, $site_id);
        Configuration::updateValue(self::AUTHORIZE_TRANSACTION_KEY, ArrayUtils::get($payload, 'authorize_transaction_key'), false, $site_id);
        
        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}