<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Currency;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Mail;
use Proxim\MailTemplate;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class CartReminder extends Module
{   
    const NEXT_CARTREMINDER_CHECK_CRON = 'NEXT_CARTREMINDER_CHECK_CRON';
    const CARTREMINDER_NOTIFY_TIME = 'CARTREMINDER_NOTIFY_TIME';

    public function __construct()
    {
        $this->name = 'cartreminder';
        $this->icon = 'fa fa-shopping-basket';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Abandoned Cart Reminder';
        $this->description = 'Improve your conversion rate by 15% (and more) with abandoned carts reminder emails!';
    } 

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        Db::getInstance()->Execute("
            CREATE TABLE IF NOT EXISTS " . Db::prefix("cartreminder") . " ( 
                `cartreminder_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `order_id` BIGINT(20) UNSIGNED NOT NULL,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`cartreminder_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ");

        $this->registerHook(['actionBeforeCron', 'displayOrderActions']);
    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayOrderActions($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $order_id = ArrayUtils::get($params, 'order_id');

        $order = new Order( (int) $order_id );
        if(Validate::isLoadedObject($order) && $order->status_id == Order::FAKE) {
            $sql = new DbQuery();
            $sql->select('cr.order_id');
            $sql->from('cartreminder', 'cr');
            $sql->where('cr.order_id = ' . $order->id );
            $reminder_sent = Db::getInstance()->getValue($sql);

            $smarty->assign([
                'reminder_sent' => $reminder_sent
            ]);
            
            return $this->showTemplate('actions.sendreminder');
        }
    }

    public function sendCartReminder( int $order_id ) {
        $order = new Order( (int) $order_id );

        if(Validate::isLoadedObject($order)) {
            $customer = $order->getCustomer(); 
            $order_url = Configuration::get('SITE_DOMAIN', $order->site_id) . '/dashboard/order/' . $order->id;

            if(Validate::isEmail($customer->email)) {
                try {
                    Mail::send(
                        MailTemplate::TEMPLATE_ORDER_PAY_REMINDER,
                        'Order '.$order->id.': Please complete your payment',
                        array(
                            'order_url' => $order_url,
                            'order_domain' => Configuration::get('SITE_DOMAIN', $order->site_id),
                            'order_id' => $order->id,
                            'order_title' => $order->title,
                            'username' => $customer->name, 
                        ),
                        $customer->email,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        $customer->site_id
                    );

                    Db::getInstance()->insert('cartreminder', array(
                        'order_id' => $order->id,
                        'date_add' => DateUtils::now()
                    ));
                } catch(Exception $e) {
                }
            }
        }

        return true;
    }

    public function hookActionBeforeCron() {
        $app = $this->application;

        $next_cron_date = Configuration::get(self::NEXT_CARTREMINDER_CHECK_CRON);
        if ( !$app->request->isAjax() && $next_cron_date && !DateUtils::hasPassed($next_cron_date)) {
            return;
        }

        $sql = new DbQuery();
        $sql->select('o.order_id');
        $sql->from('order', 'o');
        $sql->leftJoin('cartreminder', 'cr', 'o.order_id = cr.order_id');
        $sql->where('status_id = ' . Order::FAKE);
        $orderNotifyTime = (int) Configuration::get( self::CARTREMINDER_NOTIFY_TIME , PROX_SITE_ID, 1);
        $sql->where( sprintf('NOW() >= DATE_ADD(o.date_add, INTERVAL %s HOUR)', $orderNotifyTime));
        $sql->where( 'NOW() <= DATE_ADD(o.date_add, INTERVAL 2 DAY)');
        $sql->where('cr.order_id IS NULL');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        foreach($result as $order) {
            $this->sendCartReminder( (int) $order['order_id'] );
        }

        Configuration::updateValue(self::NEXT_CARTREMINDER_CHECK_CRON, DateUtils::date(strtotime('+1 hour')) );
    }

    public function sendReminder() {
        $app = $this->application;
        $order_id = $app->request->post('order_id');

        if($this->sendCartReminder( $order_id )) {
            return $app->controller->modal("SUCCESS", "Email sent successfully", "Reminder email sent successfully");
        } else {
            return $app->controller->modal("ERROR", "Email not sent", "Reminder email not sent successfully");
        }
    }

    public function getContent() {
        $cartreminderConfiguration = Configuration::getMultiple([
            self::CARTREMINDER_NOTIFY_TIME
        ]);

        $this->smarty->assign([
            'cartreminderConfiguration' => $cartreminderConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        Configuration::updateValue(self::CARTREMINDER_NOTIFY_TIME, (int) ArrayUtils::get($payload, 'notify_time'));

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }

}