<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Pager;
use Proxim\Tools;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class CreditFine extends Module
{   
    const NEXT_CREDITFINE_CHECK_CRON = 'NEXT_CREDITFINE_CHECK_CRON';
    const CREDITFINE_NOTIFY_TIME = 'CREDITFINE_NOTIFY_TIME';

    const CREDIT_FINE_PASSED_1_HRS = 'CREDIT_FINE_PASSED_1_HRS';
    const CREDIT_FINE_PASSED_2_HRS = 'CREDIT_FINE_PASSED_2_HRS';

    public function __construct()
    {
        $this->name = 'creditfine';
        $this->icon = 'fas fa-money-bill-wave';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Fines and Credit';
        $this->description = 'Charge fines and credit your employees on their next payouts';
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        Db::getInstance()->Execute("
            CREATE TABLE IF NOT EXISTS " . Db::prefix("credit_fine") . " ( 
                `credit_fine_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `type` enum('credit', 'fine') NOT NULL,
                `employee_id` BIGINT(20) UNSIGNED NOT NULL,
                `amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `description` LONGTEXT DEFAULT NULL,
                `resolved` TINYINT(1) UNSIGNED NOT NULL DEFAULT '0',
                `resolved_at` DATETIME DEFAULT NULL,
                `requested` TINYINT(1) UNSIGNED NOT NULL DEFAULT '0',
                `requested_at` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`credit_fine_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ");

        try {
            Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix('credit_fine') . " ADD (
                    `order_id` BIGINT(20) UNSIGNED DEFAULT NULL,
                    `date_upd` DATETIME DEFAULT NULL
                );
            ");
        } catch(Exception $exception) {

        }

        Db::getInstance()->Execute("
            CREATE TABLE IF NOT EXISTS " . Db::prefix("fine_type") . " ( 
                `fine_type_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `name` VARCHAR(155) NOT NULL,
                `type` enum('percentage', 'amount') NOT NULL,
                `value` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`fine_type_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ");

        $this->registerHook([
            'actionBeforeCron',
            'displayEmployeeFinancesStats', 
            'displayFinancialOverviewNavbar',
            'displayFinancialOverviewNavContent',
            'actionUpdateRequestedAdditionalPayments',
            'actionGetEmployeeAdditionalPayments',
            'displayOrderAdditionalInfo',
            'actionBeforeAdminDelete',
            'loadMoreBefore'
        ]);

    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template 
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent() {
        $sql = new DbQuery();
        $sql->select('ft.*');
        $sql->from('fine_type', 'ft');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $fines = array();
        foreach($result as $fine) {
            $fine['is_amount'] = ($fine['type'] == 'amount') ? 1 : 0;
            $fine['is_percentage'] = ($fine['type'] == 'percentage') ? 1 : 0;

            $fines[] = $fine;
        }

        $creditfineConfiguration = Configuration::getMultiple([
            self::CREDIT_FINE_PASSED_1_HRS,
            self::CREDIT_FINE_PASSED_2_HRS
        ]);

        $this->smarty->assign([
            'fines' => $fines,
            'creditfineConfiguration' => $creditfineConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookDisplayOrderAdditionalInfo( $params ) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        $order_id = ArrayUtils::get($params, 'order_id');

        $sql = new DbQuery();
        $sql->select('ft.*');
        $sql->from('fine_type', 'ft');
        $fines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $smarty->assign([
            'fines' => $fines
        ]); 


        if($user->is_admin || $user->is_sub_admin || $user->is_writer) {
            $order = new Order( (int) $order_id );
            if(Validate::isLoadedObject($order)) {

                if($user->is_writer && $user->id != $order->writer_id ) {
                    return;
                }

                if(($user->is_admin || $user->is_sub_admin) && !$order->writer_id ) {
                    return;
                }

                $charged_fines = Db::getInstance()->executeS('SELECT * FROM ' . Db::prefix('credit_fine') . ' WHERE type = "fine" AND order_id = ' . (int) $order->id . ' AND employee_id = ' . (int) $order->writer_id);
                
                $fine_amount = 0;
                foreach($charged_fines as $fine) {
                    $fine_amount = $fine_amount+$fine['amount'];
                }

                $smarty->assign([
                    'order_id' => $order->id,
                    'charged_fines' => $charged_fines,
                    'fine_amount' => $fine_amount,
                    'late_penalty_formatted' => formatPrice($fine_amount)
                ]); 
                
                return $this->showTemplate('order.penalty');
            }
        }
    }

    public function hookDisplayEmployeeFinancesStats( $params ) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $insights = array();

        $employee_id = ArrayUtils::get($params, 'employee_id');
        $employee = new Employee( (int) $employee_id );
        if(Validate::isLoadedObject($employee)) {
            $insights['credit_fine_outstanding_credit'] = (float) Db::getInstance()->getValue('SELECT SUM(amount) FROM ' . Db::prefix('credit_fine') . ' WHERE type = "credit" AND resolved = 0 AND employee_id = ' . (int) $employee->id );
            $insights['credit_fine_outstanding_fines'] = (float) Db::getInstance()->getValue('SELECT SUM(amount) FROM ' . Db::prefix('credit_fine') . ' WHERE type = "fine" AND resolved = 0 AND employee_id = ' . (int) $employee->id );
            $insights['credit_fine_total_fines'] = (float) Db::getInstance()->getValue('SELECT SUM(amount) FROM ' . Db::prefix('credit_fine') . ' WHERE type = "fine" AND resolved = 1 AND employee_id = ' . (int) $employee->id );

            $smarty->assign([
                'employee_id' => $employee->id,
                'insights' => $insights,
            ]);

            return $this->showTemplate('employee.stats');
        } 
    }

    public function hookActionGetEmployeeAdditionalPayments($params) {
        $additionalPayments = array();

        $employee_id = ArrayUtils::get($params, 'employee_id');
        $requested = ArrayUtils::has($params, 'requested');
        $resolved = ArrayUtils::has($params, 'resolved');

        $employee = new Employee( (int) $employee_id );
        if(!Validate::isLoadedObject($employee)) {
            return $additionalPayments;
        }

        $sql = new DbQuery();
        $sql->select('SUM(amount)');
        $sql->from('credit_fine');
        $sql->where('type = "credit"');
        $sql->where('employee_id = ' . (int) $employee->id );
        if($requested) {
            $sql->where('requested = 0');
        }

        if($resolved) {
            $sql->where('resolved = 0');
        }

        $credit_fine_outstanding_credit = (float) Db::getInstance()->getValue($sql);
        if($credit_fine_outstanding_credit) {
            $additionalPayments[] = array(
                'type' => 'credit',
                'amount' => (float) $credit_fine_outstanding_credit
            );
        }

        $sql = new DbQuery();
        $sql->select('SUM(amount)');
        $sql->from('credit_fine');
        $sql->where('type = "fine"');
        $sql->where('employee_id = ' . (int) $employee->id );
        if($requested) {
            $sql->where('requested = 0');
        }

        if($resolved) {
            $sql->where('resolved = 0');
        }

        $credit_fine_outstanding_fines = (float) Db::getInstance()->getValue($sql);
        if($credit_fine_outstanding_fines) {
            $additionalPayments[] = array(
                'type' => 'fine',
                'amount' => (float) $credit_fine_outstanding_fines
            );
        }

        return $additionalPayments;
    }

    public function hookActionUpdateRequestedAdditionalPayments($params) {
        $employee_id = ArrayUtils::get($params, 'employee_id');
        $requested = ArrayUtils::has($params, 'requested');
        $resolved = ArrayUtils::has($params, 'resolved');

        $employee = new Employee( (int) $employee_id );
        if(!Validate::isLoadedObject($employee)) {
            return false;
        }

        if($requested) {
            return Db::getInstance()->update(
                'credit_fine',
                array(
                    'requested' => true,
                    'requested_at' => DateUtils::now()
                ),
                'employee_id = ' . (int) $employee->id
            );
        } elseif($resolved) {
            return Db::getInstance()->update(
                'credit_fine',
                array(
                    'resolved' => true,
                    'resolved_at' => DateUtils::now()
                ),
                'employee_id = ' . (int) $employee->id
            );
        }
    }

    public function hookDisplayFinancialOverviewNavbar() {
        return $this->showTemplate('finacialoverview.nav');
    }

    public function hookDisplayFinancialOverviewNavContent() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user; 
        $params = $app->request->get();

        $selected_page = (int) ArrayUtils::get($params, 'page', 1);

        $sql = new DbQuery();
        $sql->select('*');
        $sql->from('credit_fine');
        $sql->where('employee_id = ' . (int) $user->id );
        $sql->orderBy('credit_fine_id DESC');
        $total_results = Db::getInstance(PROX_USE_SQL_SLAVE)->numRows($sql);

        $rows = array();

        if( $total_results > 0) {
            $params['total_items'] = $total_results;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = 20;
            $params['url'] = "/creditfine/viewCharges?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $rows = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'sub_view' => 'viewCharges',
            'total_rows' => $total_results,
            'rows' => $rows
        ]);

        return $this->showTemplate('finacialoverview.content');
    }

    public function viewCredits() {
        $app = $this->application;
        $smarty = $this->smarty;
        $params = $app->request->get();

        $selected_page = (int) ArrayUtils::get($params, 'page', 1);
        $employee_id = (int) ArrayUtils::get($params, 'eid');

        $employee = new Employee( (int) $employee_id );
        if(!Validate::isLoadedObject($employee)) {
            $app->setTemplate('404');
            $app->display();
        }

        $sql = new DbQuery();
        $sql->select('cf.*');
        $sql->from('credit_fine', 'cf');
        $sql->where('cf.employee_id = ' . (int) $employee->id );
        $sql->where('cf.type = "credit"');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_results = count($result);

        $rows = array();

        if( $total_results > 0) {
            $params['total_items'] = $total_results;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = 20;
            $params['url'] = "/creditfine/viewCredits?eid=" . $employee->id . "&page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $rows = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'total_rows' => $total_results,
            'rows' => $rows
        ]);

        return $this->showTemplate('employee.credits');
    }

    public function viewFines() {
        $app = $this->application;
        $smarty = $this->smarty;
        $params = $app->request->get();

        $selected_page = (int) ArrayUtils::get($params, 'page', 1);
        $employee_id = (int) ArrayUtils::get($params, 'eid');
        $is_resolved = ArrayUtils::has($params, 'resolved');

        $employee = new Employee( (int) $employee_id );
        if(!Validate::isLoadedObject($employee)) {
            $app->setTemplate('404');
            $app->display();
        }

        $sql = new DbQuery();
        $sql->select('*');
        $sql->from('credit_fine');
        $sql->where('employee_id = ' . (int) $employee->id );
        $sql->where('type = "fine"');
        if($is_resolved) {
            $sql->where('resolved = 1');
        } else {
            $sql->where('resolved = 0');
        }

        $sql->orderBy('credit_fine_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_results = count($result);

        $rows = array();

        if( $total_results > 0) {
            $params['total_items'] = $total_results;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = 20;
            $params['url'] = "/creditfine/viewFines?eid=" . $employee->id . "&page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $rows = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'total_rows' => $total_results,
            'rows' => $rows
        ]);

        return $this->showTemplate('employee.fines');
    }

    public function createCharge() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $payload = $app->request->post();

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $controller->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $handle = ArrayUtils::get($payload, 'handle');
        $amount = ArrayUtils::get($payload, 'amount');
        $description = ArrayUtils::get($payload, 'description');

        if(!$amount || !Validate::isPrice($amount)) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Please enter the amount"
            ]);
        }

        if(ArrayUtils::has($payload, 'node_id')) {
            $node_id = (int) ArrayUtils::get($payload, 'node_id');
            
            Db::getInstance()->update(
                'credit_fine',
                array(
                    'amount' => (float) $amount,
                    'description' => $description,
                ),
                'credit_fine_id = ' . (int) $node_id
            );

            return $app->sendResponse([
                "success" => true,
                "message" => ucfirst($handle) . " updated successfully"
            ]);
        } else {
            $employee_id = (int) ArrayUtils::get($payload, 'employee_id');

            $employee = new Employee( (int) $employee_id );
            if(!Validate::isLoadedObject($employee)) {
                return $controller->modal("ERROR", "Error", "This user does not exists");
            }

            Db::getInstance()->insert(
                'credit_fine',
                array(
                    'type' => $handle,
                    'employee_id' => $employee->id,
                    'amount' => (float) $amount,
                    'description' => $description,
                    'date_add' => DateUtils::now()
                )
            );

            return $app->sendResponse([
                'callback' => 'window.location.reload();'
            ]);
        }
    }

    public function resolveCharge() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $payload = $app->request->post();

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $controller->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $id = (int) ArrayUtils::get($payload, 'id');

        Db::getInstance()->update(
            'credit_fine',
            array(
                'resolved' => true,
                'resolved_at' => DateUtils::now(),
            ),
            'credit_fine_id = ' . (int) $id
        );

        return $controller->modal("SUCCESS", "Success", "Charge resolved successfully");
    }

    public function chargeFine() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $payload = $app->request->post();

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $controller->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $order_id = ArrayUtils::get($payload, 'order_id');

        $order = new Order((int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            return $controller->modal("ERROR", "Error", "This order does not exist");
        }

        $fine_id = (int) ArrayUtils::get($payload, 'fine_id');
        $fine_type = Db::getInstance()->getRow('SELECT * FROM ' . Db::prefix('fine_type') . ' WHERE fine_type_id = ' . (int) $fine_id );
        if(!$fine_type) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Select a valid fine type"
            ]);
        }

        $currentWriter = $order->getCurrentWriterFull();

        if($fine_type['type'] == 'percentage') {
            $amount = ($currentWriter['employee_pay'] * $fine_type['value']) / 100;
        } else {
            $amount = $fine_type['value'];
        }

        Db::getInstance()->insert(
            'credit_fine',
            array(
                'type' => 'fine',
                'employee_id' => $order->writer_id,
                'order_id' => $order->id,
                'amount' => (float) $amount,
                'description' => $fine_type['name'],
                'date_add' => DateUtils::now()
            )
        );

        return $controller->modal("SUCCESS", "Successful", "Fine charged successfully - " . $fine_type['name']);
    }

    public function postFine() { 
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $payload = $app->request->post();

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $controller->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $name = ArrayUtils::get($payload, 'name');
        $type = ArrayUtils::get($payload, 'fine_type');
        $amount = (float) ArrayUtils::get($payload, 'amount');
        $percentage = (float) ArrayUtils::get($payload, 'percentage');

        if(!$name) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Please enter the name"
            ]);
        }

        if($type == 'amount') {
            if(!$amount || !Validate::isPrice($amount)) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Please enter the amount"
                ]);
            }
        }

        if($type == 'percentage') {
            if(!$percentage || !Validate::isInt($percentage)) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Please enter the percentage"
                ]);
            }
        }

        if(ArrayUtils::has($payload, 'fine_type_id')) {
            $fine_type_id = (int) ArrayUtils::get($payload, 'fine_type_id');

            if($type == 'amount') {
                $value = $amount;
            } else {
                $value = $percentage;
            }
            
            Db::getInstance()->update(
                'fine_type',
                array(
                    'name' => $name,
                    'type' => $type,
                    'value' => (float) $value,
                    'date_upd' => DateUtils::now()
                ),
                'fine_type_id = ' . (int) $fine_type_id
            );

            return $app->sendResponse([
                "success" => true,
                "message" => "Fine updated successfully"
            ]);
        } else {

            if($type == 'amount') {
                $value = $amount;
            } else {
                $value = $percentage;
            }

            Db::getInstance()->insert(
                'fine_type',
                array(
                    'type' => $type,
                    'name' => $name,
                    'value' => (float) $value,
                    'date_add' => DateUtils::now()
                )
            );

            return $app->sendResponse([
                'callback' => 'window.location.reload();'
            ]);
        }
    }

    public function hookActionBeforeAdminDelete( $params ) {
        $id = ArrayUtils::get($params, 'id');
        $handle = ArrayUtils::get($params, 'handle');

        switch($handle) {
            case "creditfine":
                Db::getInstance()->delete('credit_fine', 'credit_fine_id = ' . (int) $id );
                break;

            case "fine_type":
                Db::getInstance()->delete('fine_type', 'fine_type_id = ' . (int) $id );
                break;
                
        }
    }

    public function hookActionBeforeCron() {
        $app = $this->application;

        $next_cron_date = Configuration::get(self::NEXT_CREDITFINE_CHECK_CRON);
        if ( $next_cron_date && !DateUtils::hasPassed($next_cron_date)) {
            return;
        }

        $sql = new DbQuery();
        $sql->select('o.order_id');
        $sql->from('order', 'o');
        $sql->leftJoin('credit_fine', 'cf', 'o.order_id = cf.order_id');
        $sql->where('o.status_id = ' . Order::WRITER_ASSIGNED );
        $sql->where( 'o.is_writer_confirmed = 1' );
        $sql->where('(cf.order_id IS NULL) OR ' . sprintf('(cf.date_upd <= DATE_SUB(NOW(), INTERVAL %s MINUTE))', 30 ));
        $sql->where( 'o.writer_deadline < NOW()' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        foreach($result as $order) {
            $order = new Order( (int) $order['order_id'] );
            if(Validate::isLoadedObject($order)) {
                $writer_deadline = DateUtils::timeLeft($order->writer_deadline, true);
                $passed_minutes = ArrayUtils::get($writer_deadline, 'minutes', 0)/60;
                $passed_hours = ArrayUtils::get($writer_deadline, 'hours', 0);
                $passed_hours = $passed_hours+$passed_minutes;

                $percentage = 0;
                if($passed_hours >= 1 && Configuration::get(self::CREDIT_FINE_PASSED_2_HRS) ) {
                    $percentage = (int) Configuration::get(self::CREDIT_FINE_PASSED_2_HRS);
                } else {
                    $percentage = (int) Configuration::get(self::CREDIT_FINE_PASSED_1_HRS);
                }

                if($percentage) {
                    $penalty_amount = ($order->writer_pay * $percentage) / 100;
                    $credit_fine_id = Db::getInstance()->getValue('SELECT credit_fine_id FROM ' . Db::prefix('credit_fine') . ' WHERE order_id = ' . (int) $order->id . ' AND employee_id = ' . (int) $order->writer_id);
                    if($credit_fine_id) {
                        Db::getInstance()->update(
                            'credit_fine',
                            array(
                                'amount' => $penalty_amount,
                                'description' => sprintf('Order #%s - %s Penalty', $order->id, formatPrice($penalty_amount) ),
                                'date_upd' => DateUtils::now()
                            ),
                            'credit_fine_id = ' . (int) $credit_fine_id
                        );
                    } else {
                        Db::getInstance()->insert(
                            'credit_fine',
                            array(
                                'type' => 'fine',
                                'employee_id' => $order->writer_id,
                                'order_id' => $order->id,
                                'amount' => (float) $penalty_amount,
                                'description' => sprintf('Order #%s - %s Penalty', $order->id, formatPrice($penalty_amount) ),
                                'date_upd' => DateUtils::now(),
                                'date_add' => DateUtils::now()
                            )
                        );
                    }
                }
            } 
        }

        Configuration::updateValue(self::NEXT_CREDITFINE_CHECK_CRON, DateUtils::date(null, '+10 minute') );
    }

    public function updateOrderPenalty() {
        $app = $this->application;
        $user = $app->user;
        $payload = $app->request->post();
        $controller = $app->controller;

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $controller->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $credit_fine_id = (int) ArrayUtils::get($payload, 'credit_fine_id');
        $amount = (float) ArrayUtils::get($payload, 'amount');

        $credit_fine = Db::getInstance()->getRow('SELECT * FROM ' . Db::prefix('credit_fine') . ' WHERE credit_fine_id = ' . (int) $credit_fine_id );
        if($credit_fine) {
            Db::getInstance()->update(
                'credit_fine',
                array(
                    'amount' => $amount,
                    'date_upd' => DateUtils::now()
                ),
                'credit_fine_id = ' . (int) $credit_fine['credit_fine_id']
            );
        }
        
        
        return $controller->modal("SUCCESS", "Successful", "Penalty Fee updated successfully");
    }

    public function hookLoadMoreBefore($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        $payload = ArrayUtils::get($params, 'payload');
        $get = ArrayUtils::get($payload, 'get');
        $offset = ArrayUtils::get($payload, 'offset');

        // initialize the attach type
        $append = true;

        $charged_fines = Db::getInstance()->executeS('SELECT * FROM ' . Db::prefix('credit_fine') . ' WHERE type = "fine" AND employee_id = ' . (int) $user->id . ' ORDER BY credit_fine_id DESC LIMIT 8');

        $return = array();
        $max_results = Configuration::get('MAX_RESULTS', null, 10);
        $offset *= $max_results;

        /* get quality controls */
        if ($get == "quality_controls") {
            $sql = new DbQuery();
            $sql->select('cf.*');
            $sql->from('credit_fine', 'cf');
            $sql->where('cf.type = "fine"');
            $sql->where('cf.employee_id = ' . (int) $user->id );
            $sql->where('cf.order_id IS NOT NULL');
            $sql->orderBy('cf.credit_fine_id DESC');
            $sql->limit($max_results, $offset);
            $result = Db::getInstance()->executeS($sql);

            $quality_controls = array();
            foreach($result as $res) {
                $res['desc_trucnate'] = Tools::truncateString($res['description'], 30);
                $quality_controls[] = $res;
            }

            $smarty->assign([
                'quality_controls' => $quality_controls
            ]);
          
            $return['data'] = $this->getTemplateContent("ajax.live.quality_control");
        }

        /* assign variables */
        $smarty->assign('offset', $offset );
        $smarty->assign('get', $get );
        /* return */
        $return['append'] = $append;

        // return & exit
        return $return;
    }
    
    public function updatePenaltySettings() {
        $app = $this->application;
        $payload = $app->request->post();

        foreach($payload as $key => $value ) {
            Configuration::updateValue($key, (int) $value );
        }

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}