<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Currency;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class CurrencyConverter extends Module
{
    /**
    * CurrencyConverter configuration
    */
    const NEXT_CURRENCY_CHECK_CRON = 'NEXT_CURRENCY_CHECK_CRON';

    public function __construct()
    {
        $this->name = 'currencyconverter';
        $this->icon = 'fa fa-dolllar-sign';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Currency Converter';
        $this->description = 'Automatic daily currency converter';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_sub_admin ? false : true;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook(['actionBeforeCron']);
    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getExchangeRateData() {
        $url = "https://api.exchangeratesapi.io/latest?base=USD";
        $ch = curl_init();
        curl_setopt ($ch, CURLOPT_URL, $url);
        curl_setopt ($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch,  CURLOPT_USERAGENT , "Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1)");
        curl_setopt ($ch, CURLOPT_CONNECTTIMEOUT, 0);
        $data = curl_exec($ch);
        curl_close($ch);

        return Tools::jsonDecode($data, true);
    }

    public function hookActionBeforeCron() {
        $next_cron_date = Configuration::get(self::NEXT_CURRENCY_CHECK_CRON);
        if ($next_cron_date && !DateUtils::hasPassed($next_cron_date)) {
            return;
        }

        #update conversion rates
        $currencies = Currency::getCurrencies();
        $currency_codes = array();
		foreach ( $currencies as $currency ) {
            // should not add USD to the list
			if ($currency['iso_code'] != "USD" ) {
                $currency_codes[] = $currency['iso_code'];
            }
        }

        if(!empty($currency_codes)) {
            $conversion_rates = $this->getExchangeRateData( $currency_codes );

            if ( $conversion_rates && 
                ArrayUtils::has($conversion_rates, 'rates') 
            ) {
                $rates = ArrayUtils::get($conversion_rates, 'rates');

                if (is_array($rates)) {
                    foreach ( $currencies as $currency ) {
                        $currency = new Currency( (int) $currency['currency_id'] );
                        if (Validate::isLoadedObject($currency) ) {
                            if ( ArrayUtils::has($rates, Tools::strtoupper($currency->iso_code) ) ) {
                                $conversion_rate = ArrayUtils::get($rates, Tools::strtoupper($currency->iso_code));
                                $currency->conversion_rate = (float) $conversion_rate;
                                $currency->last_checked = DateUtils::now();
                                $currency->update();
                            }
                        }
                    }
                }
            }
        }

        Configuration::updateValue(self::NEXT_CURRENCY_CHECK_CRON, DateUtils::date(null, '+1 hour') );
    }
}