<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Order\Rating;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class Editor_Review extends Module
{
    public function __construct()
    {
        $this->name = 'editor_review';
        $this->icon = 'fas fa-star-half-alt';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Editor Review';
        $this->description = 'Allow editors to comment and rate a writer\'s final paper';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->isLogged() ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        try {
            Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix("order_rating") . " ADD (
                    `user_type` ENUM('customer', 'employee') NOT NULL default 'customer',
                    `user_id` BIGINT(20) UNSIGNED DEFAULT NULL
                );
            ");
        } catch(Exception $e) {

        }
        
        $this->registerHook([
            'displayOrderActions',
            'actionControllerSetMedia'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookActionControllerSetMedia() {
        $app = Application::getInstance();
        $user = $app->user;

        $app->controller->addCss($this->_path . 'css/star-rating.min.css');
        $app->controller->addCss($this->_path . 'themes/krajee-svg/theme.min.css');

        $app->controller->addJs($this->_path . 'js/star-rating.min.js');
        $app->controller->addJs($this->_path . 'themes/krajee-svg/theme.min.js');
    }

    public function hookDisplayOrderActions($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $order_id = ArrayUtils::get($params, 'order_id');

        $order = new Order( (int) $order_id );
        if( Validate::isLoadedObject($order) && 
            ($order->status_id == Order::FINISHED ||
            $order->status_id == Order::DELIVERED ||
            $order->status_id == Order::DONE ||
            $order->status_id == Order::DISPUTE ||
            $order->status_id == Order::REVISION) &&
            ($user->is_editor && $order->editor_id == $user->id)
        ) {
            $sql = new DbQuery(); 
            $sql->select('rating.*');
            $sql->from('order_rating', 'rating');
            $sql->where('rating.user_type = "' . Rating::USER_EMPLOYEE . '"' );
            $sql->where('rating.order_id = ' . $order->id );
            $editorReview = Db::getInstance()->getRow($sql);

            $smarty->assign([
                'editorReview' => $editorReview
            ]);
            
            return $this->showTemplate('actions.rating');
        }
    }

    public function postReview() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $payload = $app->request->post();

        $order_id = ArrayUtils::get($payload, 'order_id');
        $order_rating = (int) ArrayUtils::get($payload, 'rating');
        $comment = ArrayUtils::get($payload, 'comment');
        $paper_on_time = ArrayUtils::get($payload, 'paper_on_time');

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'This order does not exist'
            ]);
        }

        $update = false;

        if(ArrayUtils::has($payload, 'rating_id')) {
            $rating_id = ArrayUtils::get($payload, 'rating_id');
            $rating = new Rating( (int) $rating_id );
            if(!Validate::isLoadedObject($rating)) {
                return $app->sendResponse([
                    'error' => true,
                    'message' => 'This review does not exist'
                ]);
            }
            $update = true;
        } else {
            $rating = new Rating();
            $rating->user_type = Rating::USER_EMPLOYEE;
            $rating->user_id = $user->id;

            $update = false;
        }

        $rating->order_id = $order->id;
        $rating->paper_rating = $order_rating;
        $rating->service_rating = $rating->service_rating ? $rating->service_rating : $order_rating;
        $rating->comment = $comment;
        $rating->paper_on_time = ($paper_on_time == 'yes') ? true : false;
        
        if($update) {
            $rating->update();
            return $controller->modal("SUCCESS", "Successful", "Review has been updated successfully");
        } else {
            $rating->add();
            return $controller->modal("SUCCESS", "Successful", "Review has been published successfully");
        }

    }
}
