<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\PaymentModule;
use Proxim\Util\ArrayUtils;

class FasterPay extends PaymentModule
{
    /**
    * FasterPay configuration
    */
    const FASTERPAY_ENABLED = 'FASTERPAY_ENABLED';
    const FASTERPAY_MODE = 'FASTERPAY_MODE';
    const FASTERPAY_API_PUBLIC_KEY = 'FASTERPAY_API_PUBLIC_KEY';
    const FASTERPAY_API_PRIVATE_KEY = 'FASTERPAY_API_PRIVATE_KEY';

    public function __construct()
    {
        $this->name = 'fasterpay';
        $this->icon = 'fas fa-credit-card';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'FasterPay';
        $this->description = 'FasterPay allows you to accept payment on your site via credit card';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    { 
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $fasterPayConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::FASTERPAY_ENABLED,
                self::FASTERPAY_MODE,
                self::FASTERPAY_API_PUBLIC_KEY,
                self::FASTERPAY_API_PRIVATE_KEY,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'fasterPayConfiguration' => $fasterPayConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateFasterPaySettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        Configuration::updateValue('PAYMENT_METHOD_DEFAULT', ArrayUtils::has($payload, 'default_method_enabled') ? $this->name : '' , false, $site_id);
        Configuration::updateValue(self::FASTERPAY_ENABLED, (ArrayUtils::get($payload, 'fasterpay_enabled') == 'on') ? true : false , false, $site_id);
        Configuration::updateValue(self::FASTERPAY_MODE, ArrayUtils::get($payload, 'fasterpay_mode'), false, $site_id);
        Configuration::updateValue(self::FASTERPAY_API_PUBLIC_KEY, ArrayUtils::get($payload, 'fasterpay_public_key'), false, $site_id);
        Configuration::updateValue(self::FASTERPAY_API_PRIVATE_KEY, ArrayUtils::get($payload, 'fasterpay_private_key'), false, $site_id);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}