<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\PaymentModule;
use Proxim\Util\ArrayUtils;

class Magento_PayPal extends PaymentModule
{
    /**
    * PayPal configuration
    */
    const MAGENTO_PAYPAL_ENABLED = 'MAGENTO_PAYPAL_ENABLED';
    const MAGENTO_PAYPAL_CHECKOUT_URL = 'MAGENTO_PAYPAL_CHECKOUT_URL';

    public function __construct()
    {
        $this->name = 'magento_paypal';
        $this->icon = 'fas fa-paypal';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Magento PayPal';
        $this->description = 'Magento PayPal allows your customers to checkout PayPal on a different website.';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    { 
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $magentoConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::MAGENTO_PAYPAL_ENABLED,
                self::MAGENTO_PAYPAL_CHECKOUT_URL,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'magentoConfiguration' => $magentoConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateMagentoSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        Configuration::updateValue('PAYMENT_METHOD_DEFAULT', ArrayUtils::has($payload, 'default_method_enabled') ? $this->name : '' , false, $site_id);
        Configuration::updateValue(self::MAGENTO_PAYPAL_ENABLED, (ArrayUtils::has($payload, 'magento_enabled')) ? true : false , false, $site_id);
        Configuration::updateValue(self::MAGENTO_PAYPAL_CHECKOUT_URL, ArrayUtils::get($payload, 'magento_checkout_url'), false, $site_id);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}