<?php
use Mailgun\Mailgun;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class MailgunMailer extends Module
{
    const MAILGUN_ENABLED = 'MAILGUN_ENABLED';
    const MAILGUN_DOMAIN = 'MAILGUN_DOMAIN';
    const PRIVATE_API_KEY = 'PRIVATE_API_KEY';
    const MAILGUN_API_HOSTNAME = 'MAILGUN_API_HOSTNAME';

    public function __construct()
    {
        $this->name = 'mailgunmailer';
        $this->icon = 'fa fa-envelope';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Mailgun';
        $this->description = 'Send Email Newsletters, Password Resets, Promotional Emails, Shipping Notifications With Confidence';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin || $user->is_sub_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook([
            'displayAdminEditSiteGeneralTabContent',
            'actionAfterSiteSettingsGeneralUpdate',
            'actionEmailSendBefore',
            'actionMailAlterMessageBeforeSend'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $mailgunConfiguration = Configuration::getMultiple([
                self::PRIVATE_API_KEY,
                self::MAILGUN_DOMAIN,
                self::MAILGUN_API_HOSTNAME,
                self::MAILGUN_ENABLED
            ]
        );

        $this->smarty->assign([
            'mailgunConfiguration' => $mailgunConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookDisplayAdminEditSiteGeneralTabContent() {
        $application = $this->application;
        $smarty = $this->smarty;

        $mailgunConfiguration = Configuration::getMultiple([
                self::MAILGUN_ENABLED,
                self::PRIVATE_API_KEY,
                self::MAILGUN_DOMAIN,
                self::MAILGUN_API_HOSTNAME
            ],
            $application->site->id
        );
        $smarty->assign('mailgunConfiguration', $mailgunConfiguration);
        
        return $this->showTemplate('tab_content');
    }

    public function hookActionMailAlterMessageBeforeSend($payload) {
        $message = ArrayUtils::get($payload, 'message');
        $fileAttachment = ArrayUtils::get($payload, 'fileAttachment');
        $htmlMessage = ArrayUtils::get($payload, 'htmlMessage');
        $siteId = (int) ArrayUtils::get($payload, 'siteId');

        if(!Configuration::get(self::MAILGUN_ENABLED, $siteId)) {
            return true;
        }

        $from = $message->getFrom();
        $to = $message->getTo();

        $fromAddress = $toAddress = $attachment = array();
        if (is_array($from)) {
            foreach ($from as $addr => $addrName) {
                $addr = trim($addr);
                $fromAddress[] = $addr;
            }
        }

        if (is_array($to)) {
            foreach ($to as $addr => $addrName) {
                $addr = trim($addr);
                $toAddress[] = $addr;
            }
        }

        if ($fileAttachment && !empty($fileAttachment)) {
            // Multiple attachments?
            if (!is_array(current($fileAttachment))) {
                $fileAttachment = array($fileAttachment);
            }

            foreach ($fileAttachment as $att) {
                $attachment[] = array(
                    'filePath' => $att['file_path'],
                    'filename' => $att['name'] 
                );
            }
        }

        $params = array(
            'from'    => $fromAddress,
            'to'      => $toAddress,
            'subject' => $message->getSubject(),
            'html'    => $htmlMessage,
            'attachment' => $attachment
        );

        try {
            $domain = Configuration::get(self::MAILGUN_DOMAIN, $siteId);
            $api_hostname = Configuration::get(self::MAILGUN_API_HOSTNAME, $siteId);
            $mailgun = Mailgun::create( Configuration::get(self::PRIVATE_API_KEY, $siteId), $api_hostname );
            $mailgun->messages()->send($domain, $params);
        } catch (Exception $e) {
        }

        return false;
    }

    public function updateMailgunSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        Configuration::updateValue(self::MAILGUN_ENABLED, ArrayUtils::has($payload, 'mailgun_enabled') ? true : false);
        Configuration::updateValue(self::PRIVATE_API_KEY, ArrayUtils::get($payload, 'mailgun_api_key'));
        Configuration::updateValue(self::MAILGUN_API_HOSTNAME, ArrayUtils::get($payload, 'mailgun_api_hostname'));
        Configuration::updateValue(self::MAILGUN_DOMAIN, ArrayUtils::get($payload, 'mailgun_domain'));

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    } 

    public function hookActionAfterSiteSettingsGeneralUpdate( $params ) {
        $site_id = ArrayUtils::get($params, 'site_id');
        $payload = ArrayUtils::get($params, 'payload');

        $site = new Site( (int) $site_id );
        if(Validate::isLoadedObject($site)) {
            Configuration::updateValue(self::MAILGUN_ENABLED, ArrayUtils::has($payload, 'mailgun_enabled') ? true : false, false, $site->id);
            Configuration::updateValue(self::PRIVATE_API_KEY, ArrayUtils::get($payload, 'mailgun_api_key'), false, $site->id);
            Configuration::updateValue(self::MAILGUN_API_HOSTNAME, ArrayUtils::get($payload, 'mailgun_api_hostname'), false, $site->id);
            Configuration::updateValue(self::MAILGUN_DOMAIN, ArrayUtils::get($payload, 'mailgun_domain'), false, $site->id);
        }
    }
}
