<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use Proxim\Tools;
use Proxim\Validate;

class Mpesa_Payout extends Module
{
    const MPESA_SHORTCODE = 'MPESA_SHORTCODE';
    const MPESA_PAYOUT_ENABLED = 'MPESA_PAYOUT_ENABLED';
    const MPESA_CONSUMER_KEY = 'MPESA_CONSUMER_KEY';
    const MPESA_CONSUMER_SECRET = 'MPESA_CONSUMER_SECRET';
    const MPESA_PAYBILL_ACC = 'MPESA_PAYBILL_ACC';
    const MPESA_INITIATOR_NAME = 'MPESA_INITIATOR_NAME';
    const MPESA_PASSWORD = 'MPESA_PASSWORD';
    const MPESA_USD_KES_RATE = 'MPESA_USD_KES_RATE';
    const MPESA_DEFAULT_RATE = 100;

    public function __construct()
    {
        $this->name = 'mpesa_payout';
        $this->icon = 'fa fa-tag';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Mpesa Payout';
        $this->description = 'Make payments to your employees to their mpesa with a single click!';

        $this->prox_themes_compliancy = array(
            'dashlite'
        );

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        return true;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        Db::getInstance()->Execute("
            CREATE TABLE IF NOT EXISTS " . Db::prefix("mpesa_payout") . " ( 
                `mpesa_payout_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `payment_request_id` BIGINT(20) UNSIGNED NOT NULL,
                `amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `phone` VARCHAR(50) NOT NULL,
                `transaction_id` TEXT DEFAULT NULL,
                `is_paid` TINYINT(1) UNSIGNED NOT NULL DEFAULT '0',
                `paid_at` DATETIME DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`mpesa_payout_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ");

        Configuration::updateValue(self::MPESA_USD_KES_RATE, self::MPESA_DEFAULT_RATE );
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent() {
        $app = $this->application; 
        $user = $app->user;

        if(!$user->is_admin) {
            $app->setTemplate('404');
            $app->display();
        }

        $mpesaConfiguration = Configuration::getMultiple([
            self::MPESA_SHORTCODE,
            self::MPESA_PAYOUT_ENABLED,
            self::MPESA_CONSUMER_KEY,
            self::MPESA_CONSUMER_SECRET,
            self::MPESA_PAYBILL_ACC,
            self::MPESA_INITIATOR_NAME,
            self::MPESA_PASSWORD,
            self::MPESA_USD_KES_RATE
        ]);

        $this->smarty->assign([
            'mpesaConfiguration' => $mpesaConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    /**
     * Compute security credential
     * 
     */
    public function computeSecurityCredential($initiatorPass){
        $pubKeyFile =  dirname(__FILE__) . '/config/mpesa.cer';
        $pubKey = '';
        if(\is_file($pubKeyFile)){
            $pubKey = file_get_contents($pubKeyFile);
        }else{
            throw new \Exception("Please provide a valid public key file");
        }
        openssl_public_encrypt($initiatorPass, $encrypted, $pubKey, OPENSSL_PKCS1_PADDING);
        return base64_encode($encrypted);
    }
    
    public function makePayment() {
        $app = $this->application;
        $payload = $app->request->post();
        $controller = $app->controller;
        $user = $app->user;

        $paymentRequestId = ArrayUtils::get($payload, 'request_id');
        $phoneNumber = ArrayUtils::get($payload, 'phone_number');
        $amount = (float) ArrayUtils::get($payload, 'amount');

        if(!$user->is_admin) {
            return $controller->modal("ERROR", "Error", "Could not approve this request. Try again later");
        }

        $paymentRequest = Db::getInstance()->getRow('SELECT * FROM ' . Db::prefix('payment_request') . ' WHERE payment_request_id = ' . (int) $paymentRequestId);

        if (!$paymentRequest) {
            return $controller->modal("ERROR", "Error", "Could not approve this request. Try again later");
        }

        if(!$phoneNumber) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter the mpesa phone number"
            ]);
        }

        $phoneUtil = PhoneNumberUtil::getInstance();
        try {
            $phoneNumberProto = $phoneUtil->parse($phoneNumber, 'KE' );

            if(!$phoneUtil->isValidNumber($phoneNumberProto)) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Enter a valid phone number"
                ]);
            }

            $phoneNumber = $phoneUtil->format($phoneNumberProto, PhoneNumberFormat::INTERNATIONAL);
        } catch (\libphonenumber\NumberParseException $e) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter a valid phone number"
            ]);
        }

        $phoneNumber = str_replace("+", "", $phoneNumber );
        $phoneNumber = str_replace(" ", "", $phoneNumber );


        if(!$amount || !Validate::isPrice($amount)) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter a valid amount"
            ]);
        }

        $consumerkey = Configuration::get(self::MPESA_CONSUMER_KEY);
        $consumerSecret = Configuration::get(self::MPESA_CONSUMER_SECRET);

        $successCallback  = $app->base_uri . '/mpesa_payout/updateMpesaPayout';
        $timeoutCallback  = $app->base_uri . '/mpesa_payout/updateMpesaPayout';
        $shortCode = Configuration::get(self::MPESA_SHORTCODE);
        $initiator  = Configuration::get(self::MPESA_INITIATOR_NAME);
        $pass = Configuration::get(self::MPESA_PASSWORD);
        $securityCredential  = $this->computeSecurityCredential($pass);
        $commandId  = "BusinessPayment";

        $configParams = [
            'InitiatorName'     => $initiator,
            'SecurityCredential' => $securityCredential,
            'CommandID'         => $commandId,
            'Amount'            => (float) $amount,
            'PartyA'            => $shortCode,
            'PartyB'            => $phoneNumber,
            'Remarks'           => "Payment Request - KES" . $amount,
            'QueueTimeOutURL'   => $timeoutCallback,
            'ResultURL'         => $successCallback,
            "Occassion"         => ""
        ];

        $bearerToken = base64_encode($consumerkey.":".$consumerSecret);

        $ch = curl_init('https://sandbox.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials');
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Basic ' . $bearerToken]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        $response = curl_exec($ch);
        $response = Tools::jsonDecode($response, true);

        if(!ArrayUtils::has($response, 'access_token')) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Something went wrong. Please try again later"
            ]);
        }

        $accessToken = ArrayUtils::get($response, 'access_token');

        $ch = curl_init('https://sandbox.safaricom.co.ke/mpesa/b2c/v1/paymentrequest');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $configParams);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        $response  = curl_exec($ch);
        curl_close($ch);

        if(ArrayUtils::get($response, 'ResponseCode') == "0") {
            $transaction = Db::getInstance()->insert(
                'mpesa_payout',
                [
                    'payment_request_id' => $paymentRequest['payment_request_id'],
                    'amount' => $amount,
                    'phone' => $phoneNumber
                ]
            );

            return $controller->modal("SUCCESS", "Payout Sent", "Your payout request was successfuly sent. The funds may take time to reflect.");
        } else {
            return $app->sendResponse([
                "error" => true,
                "message" => "Something went wrong. Please try again later"
            ]);
        }
    }

    public function updateMpesaPayout() {
        $payload = Tools::getAllValues();

        $Result = ArrayUtils::get($payload, 'Result');

        if(ArrayUtils::has($Result, 'ResultCode') && ArrayUtils::get($Result, 'ResultCode') == "0") {
            $ResultCode = ArrayUtils::get($Result, 'ResultCode');
            $TransactionID = ArrayUtils::get($Result, 'TransactionID');
            $TransactionAmount = ArrayUtils::get($Result, 'TransactionAmount');
            $ReceiverPartyPublicName = ArrayUtils::get($Result, 'ReceiverPartyPublicName');
            $phoneNumber = substr($ReceiverPartyPublicName, 0, 12);
    
            $payoutRequest = Db::getInstance()->getRow('SELECT * FROM ' . Db::prefix('mpesa_payout') . ' WHERE phone = \'' . pSQL($phoneNumber) . '\' AND amount = ' . (float) $TransactionAmount);
    
            if($payoutRequest) {
                Db::getInstance()->update('payment_request', array(
                    'is_paid' => '1',
                    'paid_at' => DateUtils::now(),
                    'transaction_id' => $TransactionID,
                ), 'mpesa_payout_id = ' . (int) $payoutRequest['mpesa_payout_id'], 1, true);
    
                // update payment status
                Db::getInstance()->update('payment_request', array(
                    'status' => '1',
                    'paid_at' => DateUtils::now(),
                ), 'payment_request_id = ' . (int) $payoutRequest['payment_request_id'], 1, true);
            }
        }
    }

    public function updateSettings() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $payload = $app->request->post();

        if(!$user->is_admin) {
            return $controller->modal("ERROR", "Error", "Could not approve this request. Try again later");
        }

        $paybilAccNo = ArrayUtils::get($payload, 'mpesa_paybill_acc');
        $mpesaInitiatorName = ArrayUtils::get($payload, 'mpesa_initiator_name');
        $mpesaPassword = ArrayUtils::get($payload, 'mpesa_password');
        $usdToKes = (float) ArrayUtils::get($payload, 'usd_to_kes_amount');
        $mpesaConsumerKey = ArrayUtils::get($payload, 'mpesa_consumer_key');
        $mpesaConsumerSecret = ArrayUtils::get($payload, 'mpesa_consumer_secret');
        $mpesaShortcode = ArrayUtils::get($payload, 'mpesa_shortcode');

        Configuration::updateValue(self::MPESA_PAYOUT_ENABLED, ArrayUtils::has($payload, 'mpesa_payout_enabled') ? true : false);
        Configuration::updateValue(self::MPESA_CONSUMER_KEY, $mpesaConsumerKey );
        Configuration::updateValue(self::MPESA_CONSUMER_SECRET, $mpesaConsumerSecret );
        Configuration::updateValue(self::MPESA_SHORTCODE, $mpesaShortcode );
        Configuration::updateValue(self::MPESA_PAYBILL_ACC, $paybilAccNo );
        Configuration::updateValue(self::MPESA_INITIATOR_NAME, $mpesaInitiatorName );
        Configuration::updateValue(self::MPESA_PASSWORD, $mpesaPassword );
        Configuration::updateValue(self::MPESA_USD_KES_RATE, $usdToKes );

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}