<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Pager;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Tools;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Util\Formatting;
use Proxim\Validate;

define('CURRENT_NEWS_MODULE_DIR', realpath(dirname(__FILE__)));

require_once(CURRENT_NEWS_MODULE_DIR . '/classes/NewsObject.php');

class News extends Module
{
    public function __construct()
    {
        $this->name = 'news';
        $this->icon = 'fa fa-bullhorn';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'News';
        $this->description = 'Create newsfeeds for your website';

        $this->js_path = $this->_path . 'js/';
        $this->css_path = $this->_path . 'css/';
        $this->module_path = $this->_path;
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->isLogged() == true;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        }

        $this->registerHook([
            'actionDispatcherBeforeRun',
            'displayAdminNavBarBeforeEnd',
            'actionBeforeAdminDelete',
            'actionControllerSetMedia'
        ]);
    }

    public function createTables() {
        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('news') . " (
                `news_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `title` TEXT NOT NULL,
                `content` LONGTEXT DEFAULT NULL,
                `recipient` VARCHAR(50) NOT NULL DEFAULT 'all',
                `views` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`news_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('news_view') . " (
                `news_view_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `news_id` BIGINT(20) UNSIGNED NOT NULL,
                `user_id` BIGINT(20) UNSIGNED NOT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`news_view_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookActionDispatcherBeforeRun() {
        $this->getNotifications();
    }

    public function hookDisplayAdminNavBarBeforeEnd() {
        if($this->checkAccess()) {
            return $this->showTemplate('nav_item');
        }
    }

    public function hookActionControllerSetMedia() {
        $app = Application::getInstance();
        $user = $app->user;

        if($user->isLogged()) {
            $app->controller->addCss($this->css_path . 'news.css');
            $app->controller->addJs($this->js_path . 'news.js');
        }
    }

    public function getNotifications() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        $unread_news_updates = 0;
        
        $sql = new DbQuery();
        $sql->select('COUNT(n.news_id)');
        $sql->from('news', 'n');
        $sql->leftJoin('news_view', 'nv', 'nv.news_id = n.news_id AND nv.user_id = ' . (int) $user->id );
        if($user->is_writer) {
            $sql->where("n.recipient = 'all' OR  n.recipient = 'writers'");
        } elseif ($user->is_editor) {
            $sql->where("n.recipient = 'all' OR  n.recipient = 'editors'");
        } elseif ($user->is_sub_admin) {
            $sql->where("n.recipient = 'all' OR  n.recipient = 'subadmins'");
        }
        $sql->where('nv.news_id IS NULL');
        $unread_news_updates = (int) Db::getInstance(PROX_USE_SQL_SLAVE)->getValue($sql);

        $smarty->assign([
            'unread_news_updates' => $unread_news_updates
        ]);
    }

    public function updates() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $params = $app->request->get();
        $is_search = ArrayUtils::has($params, 'search') ? true : false;
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
        $title = ArrayUtils::has($params, 'title');

        $sql = new DbQuery();
        $sql->select('n.*');
        $sql->from('news', 'n');

        if ($is_search) {
            if ($title) {
                $sql->where('n.title LIKE \'%' . pSQL($title) . '%\'');
            }
        }

        if($user->is_writer) {
            $sql->where("n.recipient = 'all' OR  n.recipient = 'writers'");
        } elseif ($user->is_editor) {
            $sql->where("n.recipient = 'all' OR  n.recipient = 'editors'");
        } elseif ($user->is_sub_admin) {
            $sql->where("n.recipient = 'all' OR  n.recipient = 'subadmins'");
        }

        $sql->orderBy('n.news_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_news = count($result);

        $news_list = array();

        if( $total_news > 0) {
            $params['total_items'] = $total_news;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/news/updates?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            $objectPresenter = new ObjectPresenter();
            foreach( $result as $news ) {
                $news = new NewsObject( (int) $news['news_id'] );
                if( Validate::isLoadedObject($news) ) {

                    $newsPresented = $objectPresenter->present( $news );
                    $content_clean = Tools::getStringClean($news->content);
                    $newsPresented['contentTrunc'] = Tools::truncateString($content_clean, 200);
                    $newsPresented['is_viewed'] = false;

                    $sql = new DbQuery();
                    $sql->select('nv.*');
                    $sql->from('news_view', 'nv');
                    $sql->where('user_id = ' . (int) $user->id);
                    $sql->where('news_id = ' . (int) $news->id);
                    $result = Db::getInstance()->getRow($sql);
                    if( $result ) {
                        $newsPresented['is_viewed'] = true;
                        $newsPresented['read_at'] = DateUtils::date( strtotime($result['date_add']), null, 'jS F \a\t g:ia');
                    }
                    
                    $news_list[] = $newsPresented;
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'news',
            'sub_view' => 'listNews',
            'news' => $news_list,
            'is_search' => $is_search
        ]);

        return $this->showTemplate('news');
    }

    public function publish() {
        $app = $this->application;
        $controller = $app->controller;
        $smarty = $this->smarty;
        $user = $app->user;

        if(!$user->is_admin) {
            $controller->setTemplate('404');
            $controller->display();
        }

        $smarty->assign([
            'view' => 'news',
            'sub_view' => 'addNews'
        ]);

        return $this->showTemplate('news');
    }

    public function viewPost() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->controller;
        $smarty = $this->smarty;
        $params = $app->request->get();

        $newsId = ArrayUtils::get($params, 'newsId');
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
        $active_tab = ArrayUtils::get($params, 'tab', 'info');

        $news = new NewsObject( (int) $newsId );
        if(!Validate::isLoadedObject($news)) {
            $controller->setTemplate('404');
            $controller->display();
        }

        $sql = 'SELECT * FROM ' . Db::prefix('news_view'). ' WHERE news_id = ' . (int) $news->id . ' ORDER BY news_view_id DESC';
        $viewers_result = Db::getInstance()->executeS($sql);
        $total_views = count($viewers_result);
        
        $viewers = $params = array();
        if(count($viewers_result) > 0) {

            $params['total_items'] = $total_views;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/news/viewPost?newsId=".$news->id."&tab=views&page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . ' ' . $limit_query);

            foreach($result as $viewer) {
                $viewer_user = new Employee( (int) $viewer['user_id'] );
                if(Validate::isLoadedObject($viewer_user)) {
                    $viewers[] = array(
                        'id' => $viewer_user->id,
                        'first_name' => $viewer_user->first_name,
                        'last_name' => $viewer_user->last_name,
                        'viewed_at' => $viewer['date_add']
                    );
                }
            }

            $smarty->assign([
                'views_pager' => $pager->getPager()
            ]);
        }

        $objectPresenter = new ObjectPresenter();

        $smarty->assign([
            'view' => 'news',
            'sub_view' => 'viewNews',
            'active_tab' => $active_tab,
            'total_views' => $total_views,
            'viewers' => $viewers,
            'news' => $objectPresenter->present($news)
        ]);

        return $this->showTemplate('news');
    }

    public function apiPublishNews() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        if(!$user->is_admin) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'You don\'t have the rights to publish this'
            ]);
        }

        $update = false;
        if(ArrayUtils::has($payload, 'news_id')) {
            $news_id = ArrayUtils::get($payload, 'news_id');
            $news = new NewsObject( (int) $news_id );
            if(!Validate::isLoadedObject($news)) {
                return $app->sendResponse([
                    'error' => true,
                    'message' => 'This news does not exist'
                ]);
            }

            $update = true;
        } else {
            $news = new NewsObject();
        }

        $title = ArrayUtils::get($payload, 'title');
        $content = ArrayUtils::get($payload, 'content');
        $recipient = ArrayUtils::get($payload, 'recipient');

        if(!$title || !Validate::isCleanHtml($title)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Enter a valid title'
            ]);
        }

        if(!$content) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Enter the news body'
            ]);
        }

        if(!$recipient) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Select a valid recipient'
            ]);
        }

        $news->title = $title;
        $news->content = $content;
        $news->recipient = $recipient;

        if($update) {
            $news->update();

            return $app->sendResponse([
                'success' => true,
                'message' => 'News has successfully been updated'
            ]);
        } else {
            $news->add();

            return $app->sendResponse([
                'callback' => 'window.location.href="/news/viewPost?newsId='.$news->id.'";'
            ]);
        }
    }

    public function markAsRead() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        $news_id = ArrayUtils::get($payload, 'news_id');

        $news = new NewsObject( (int) $news_id );
        if(!Validate::isLoadedObject($news)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'This news does not exist'
            ]);
        }

        $has_viewed = Db::getInstance()->getRow('SELECT news_id FROM ' . Db::prefix('news_view'). ' WHERE news_id = ' . (int) $news->id . ' AND user_id = ' . (int) $user->id);
        if(!$has_viewed) {
            Db::getInstance()->insert(
                'news_view',
                array(
                    'news_id' => (int) $news->id,
                    'user_id' => (int) $user->id,
                    'date_add' => DateUtils::now()
                )
            );

            $news->views = $news->views+1;
            $news->update();
        }

        return $app->sendResponse([
            'success' => true
        ]);
    }
}
