<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class Online_Users extends Module 
{
    /**
     * @var string
     */
    private $js_path;

    /**
     * @var string
     */
    private $css_path;

    /**
     * @var string
     */
    private $module_path;

    public function __construct()
    {
        $this->name = 'online_users';
        $this->icon = 'fas fa-users';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Online Users';
        $this->description = 'This addon enables you to display how many users are online on your admin site.';
        $this->bootstrap = true;

        parent::__construct();

        $this->js_path = $this->_path . 'js/' . PROX_ACTIVE_THEME . '/';
        $this->css_path = $this->_path . 'css/' . PROX_ACTIVE_THEME . '/';
        $this->module_path = $this->_path;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }
        
        $this->registerHook([
            'displayFooterAfter',
            'actionControllerSetMedia',
            'actionBeforeLiveData'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    /**
     * getOnlineEmployees
     * 
     * @return array
     */
    public function getOnlineEmployees() {
        $app = $this->application;
        $activeUser = $app->user;
        
        /* get online users */
        $online_writers = [];

        $sql = new DbQuery();
        $sql->select('e.employee_id');
        $sql->from('employee', 'e');
        $sql->where( 'e.last_activity >= SUBTIME(NOW(), SEC_TO_TIME(60))' );
        $sql->where( 'e.employee_id != ' . (int) $activeUser->id );
        $sql->where( 'e.is_started = 1' );
        $sql->orderBy( 'e.last_activity DESC' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        /* make a list from viewer's users */
        $objectPresenter = new ObjectPresenter();

        foreach($result as $user) {
            $user = new Employee( (int) $user['employee_id'] );
            if(Validate::isLoadedObject($user)) {
                $online_writer = $objectPresenter->present($user);
                $online_writer['is_online'] = '1';
                $online_writer['order_stats'] = $user->order_stats;
                $online_writer['user_picture_default'] = !$user->avatar ? true : false;
                $online_writer['user_picture'] = $user->getPicture($user->avatar, $user->gender);
                $online_writers[] = $online_writer;
            }
        }

        return $online_writers;
    }


    /**
     * getOfflineEmployees
     * 
     * @return array
     */
    public function getOfflineEmployees() {
        $app = $this->application;
        $activeUser = $app->user;

        /* get offline users */
        $offline_writers = [];

        $sql = new DbQuery();
        $sql->select('e.employee_id');
        $sql->from('employee', 'e');
        $sql->where( 'e.last_activity < SUBTIME(NOW(), SEC_TO_TIME(60))' );
        $sql->where( 'e.employee_id != ' . (int) $activeUser->id );
        $sql->where( 'e.is_started = 1' );
        $sql->orderBy( 'e.last_activity DESC' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        /* make a list from viewer's users */
        $objectPresenter = new ObjectPresenter();

        foreach($result as $user) {
            $user = new Employee( (int) $user['employee_id'] );
            if(Validate::isLoadedObject($user)) {
                $offline_writer = $objectPresenter->present($user);
                $offline_writer['is_online'] = '0';
                $offline_writer['order_stats'] = $user->order_stats;
                $offline_writer['user_picture_default'] = !$user->avatar ? true : false;
                $offline_writer['user_picture'] = $user->getPicture($user->avatar, $user->gender);
                $offline_writers[] = $offline_writer;
            }
        }

        return $offline_writers;
    }

    /**
     * getOnlineCustomers
     * 
     * @return array
     */
    public function getOnlineCustomers() {
        $app = $this->application;
        $activeUser = $app->user;
        
        /* get online customers */
        $online_customers = [];

        $sql = new DbQuery();
        $sql->select('c.customer_id');
        $sql->from('customer', 'c');
        $sql->where( 'c.last_activity >= SUBTIME(NOW(), SEC_TO_TIME(60))' );
        $sql->orderBy( 'c.last_activity DESC' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        /* make a list from viewer's users */
        $objectPresenter = new ObjectPresenter();

        foreach($result as $customer) {
            $customer = new Customer( (int) $customer['customer_id'] );
            if(Validate::isLoadedObject($customer)) {
                $online_customer = $objectPresenter->present($customer);
                $online_customer['is_online'] = '1';
                $online_customers[] = $online_customer;
            }
        }

        return $online_customers;
    }


    /**
     * getOfflineCustomers
     * 
     * @return array
     */
    public function getOfflineCustomers() {
        $app = $this->application;

        /* get offline customers */
        $offline_customers = [];

        $sql = new DbQuery();
        $sql->select('c.customer_id');
        $sql->from('customer', 'c');
        $sql->where('c.last_activity < DATE_ADD(NOW(),INTERVAL -1 DAY)');
        $sql->orderBy( 'c.last_activity DESC' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        /* make a list from viewer's customers */
        $objectPresenter = new ObjectPresenter();

        foreach($result as $customer) {
            $customer = new Customer( (int) $customer['customer_id'] );
            if(Validate::isLoadedObject($customer)) {
                $offline_customer = $objectPresenter->present($customer);
                $offline_customer['is_online'] = '0';
                $offline_customers[] = $offline_customer;
            }
        }

        return $offline_customers;
    }

    public function showChatSidebar() {
        $app = $this->application;
        $user = $app->user;

        $routeName = $app->router()->getCurrentRoute()->getName();
        if($routeName == 'tickets') return false;
        
        $detect = new Mobile_Detect();
        return $user->isLogged() && ($user->is_admin || $user->is_sub_admin) && !($detect->isMobile() && !$detect->isTablet());
    }

    public function hookDisplayFooterAfter() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $cookie = $app->cookie;

        if(!isset($cookie->chat_sidebar_closed)) {
            $cookie->chat_sidebar_closed = false;
        } 

        if( $this->showChatSidebar() ) {
            /* get online users */
            $online_writers = $this->getOnlineEmployees();
            /* get offline users */
            $offline_writers = $this->getOfflineEmployees();
            /* get sidebar users */
            $sidebar_writers = array_merge( $online_writers, $offline_writers );

            /* get online customers */
            $online_customers = $this->getOnlineCustomers();
            /* get offline customers */
            $offline_customers = $this->getOfflineCustomers();
            /* get sidebar customers */
            $sidebar_customers = array_merge( $online_customers, $offline_customers );

            /* assign variables */
            $smarty->assign([
                'chat_enabled' => Module::isInstalled('messenger'),
                'chat_sidebar_closed' => (bool) $cookie->chat_sidebar_closed,
                'chat_sidebar_path' => $this->_path . 'views/' . PROX_ACTIVE_THEME . '/ajax.chat.master.sidebar.tpl',
                'sidebar_writers' => $sidebar_writers,
                'online_writers_count' => count($online_writers),
                'sidebar_customers' => $sidebar_customers,
                'online_customers_count' => count($online_customers)
            ]);

            return $this->showTemplate('chat.sidebar');
        } 
    } 

    public function hookActionControllerSetMedia() {
        $app = Application::getInstance();
        $app->controller->addCss($this->css_path . 'chat.css');
        $app->controller->addJs($this->js_path . 'chat.js');
    }

    public function hookActionBeforeLiveData() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        // initialize the return array
        $return = [];

        $smarty->assign([
            'chat_enabled' => Module::isInstalled('messenger')
        ]);

        // [4] [update] master chat sidebar (online & offline users list)
		/* get online users */
        $online_writers = $this->getOnlineEmployees();
        /* get offline users */
        $offline_writers = $this->getOfflineEmployees();
		/* get sidebar users */
		$sidebar_writers = array_merge( $online_writers, $offline_writers );
		// assign variables
		$smarty->assign([
            'sidebar_writers' => $sidebar_writers,
            '_tab' => 'writers'
        ]);
		/* return */
		$return['master']['sidebar_writers'] = $this->getTemplateContent("ajax.chat.master.sidebar");
        $return['master']['online_writers_count'] = count($online_writers);

        /* get online customers */
        $online_customers = $this->getOnlineCustomers();
        /* get offline customers */
        $offline_customers = $this->getOfflineCustomers();
		/* get sidebar customers */
		$sidebar_customers = array_merge( $online_customers, $offline_customers );
		// assign variables
		$smarty->assign([
            'sidebar_customers' => $sidebar_customers,
            '_tab' => 'customers'
        ]);
		/* return */
		$return['master']['sidebar_customers'] = $this->getTemplateContent("ajax.chat.master.sidebar");
        $return['master']['online_customers_count'] = count($online_customers);

        return $return;
    }

    public function toggleChatHeader() {
        $app = $this->application;
        $user = $app->user;
        $cookie = $app->cookie;
        $smarty = $app->smarty;
        $payload = $app->request->post();

        $closed = (bool) ArrayUtils::get($payload, 'closed');

        if(!isset($cookie->chat_sidebar_closed)) {
            $cookie->chat_sidebar_closed = false;
        } 

        $cookie->chat_sidebar_closed = ($closed == true) ? false : true;

        return $app->sendResponse(['success' => true]);
    }
}