<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Preference\Discipline;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;

class Overview extends Module
{
    /**
     * @var string
     */
    private $css_path;

    /**
     * @var string
     */
    private $js_path;

    /**
     * @var string 
     */
    private $module_path;

    public $active_theme = PROX_ACTIVE_THEME;

    public function __construct()
    {
        $this->name = 'overview';
        $this->icon = 'fa fa-tachometer-alt';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Overview';
        $this->description = 'See key statistics for all your orders, customers, sales all with just a glance!';

        $this->css_path = $this->_path . 'css/';
        $this->js_path = $this->_path . 'js/';
        $this->module_path = $this->local_path . 'views';
    }

    public function checkAccess() {
        $user = $this->application->user;
        
        if(!$user->isLogged()) return false;

        return ($user->is_sub_admin && !Configuration::get('ORDER_MANAGER_ENABLED') ) ? false : true;
    }

    public function install()
    {
        return parent::install() && $this->registerHook(['displayAdminNavBarBeforeStart', 'actionControllerSetMedia']);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminNavBarBeforeStart() 
    {
        if($this->checkAccess()) {
            return $this->showTemplate('nav_item');
        }
    }

    public function hookActionControllerSetMedia() {
        $app = Application::getInstance();
        $user = $app->user;

        if($this->checkAccess()) {
            $app->controller->addCss($this->js_path . '/datepicker/daterangepicker.css');
            $app->controller->addJs($this->js_path . '/datepicker/daterangepicker.js');

            if($this->active_theme == 'phantom') {
                $app->controller->addJS('https://code.highcharts.com/highcharts.js', false);
                $app->controller->addJS('https://code.highcharts.com/modules/exporting.js', false);
            } else {
                $app->controller->addJs($this->js_path . $this->active_theme . '/chart.js');
            }
        } 
    }

    public function getContent()
    {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $params = $app->request->get();
        $insights = $chart = array();

        $tipsEnabled = false;

        $this_start = strtotime("1 January ".date('Y')." 12:00am");
        $this_end = strtotime("31 December ".date('Y')." 11:59pm");
        $array = array('01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0);
        $code = 'm';
        $main_range = 'This Year';
        $title = "Yearly";

        $range = ArrayUtils::get($params, 'range', $main_range);
        $hasRange = ArrayUtils::has($params, 'range');

        if(!empty($range) && in_array($range, array('Today', 'Yesterday', 'This Week', 'This Month', 'Last Month'))) {

            switch($range) {
                case 'Today':
                case 'Yesterday':
                    $array = array('00' => 0 ,'01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0 ,'13' => 0 ,'14' => 0 ,'15' => 0 ,'16' => 0 ,'17' => 0 ,'18' => 0 ,'19' => 0 ,'20' => 0 ,'21' => 0 ,'22' => 0 ,'23' => 0);
                    $this_start = strtotime(date('M')." ".date('d').", ".date('Y')." 12:00am");
                    $this_end = strtotime(date('M')." ".date('d').", ".date('Y')." 11:59pm");
                    
                    if ($range == 'Yesterday') {
                        $this_start = strtotime(date('M')." ".date('d',strtotime("-1 days")).", ".date('Y')." 12:00am");
                        $this_end = strtotime(date('M')." ".date('d',strtotime("-1 days")).", ".date('Y')." 11:59pm");
                    }

                    $code = 'H';
                    $main_range = 'Today';
                    $title = "Daily";
                    break;

                case 'This Week':
                    $time = strtotime(date('l').", ".date('M')." ".date('d').", ".date('Y'));

                    if (date('l') == 'Saturday') {
                        $this_start = strtotime(date('M')." ".date('d').", ".date('Y')." 12:00am");
                    }
                    else{
                        $this_start = strtotime('last saturday, 12:00am', $time);
                    }

                    if (date('l') == 'Friday') {
                        $this_end = strtotime(date('M')." ".date('d').", ".date('Y')." 11:59pm");
                    }
                    else{
                        $this_end = strtotime('next Friday, 11:59pm', $time);
                    }
                    
                    $array = array('Saturday' => 0 , 'Sunday' => 0 , 'Monday' => 0 , 'Tuesday' => 0 , 'Wednesday' => 0 , 'Thursday' => 0 , 'Friday' => 0);
                    $main_range = 'This Week';
                    $code = 'l';
                    break;

                case 'This Month':
                case 'Last Month':
                    $this_start = strtotime("1 ".date('M')." ".date('Y')." 12:00am");
                    $this_end = strtotime(cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y'))." ".date('M')." ".date('Y')." 11:59pm");
                    $array = array_fill(1, cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y')),0);
                    $month_days = cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y'));
                    if ($_GET['range'] == 'Last Month') {
                        $this_start = strtotime("1 ".date('M',strtotime("-1 month"))." ".date('Y')." 12:00am");
                        $this_end = strtotime(cal_days_in_month(CAL_GREGORIAN, date('m',strtotime("-1 month")), date('Y'))." ".date('M',strtotime("-1 month"))." ".date('Y')." 11:59pm");
                        $array = array_fill(1, cal_days_in_month(CAL_GREGORIAN, date('m',strtotime("-1 month")), date('Y')),0);
                        $month_days = cal_days_in_month(CAL_GREGORIAN, date('m',strtotime("-1 month")), date('Y'));
                    }
                    if (cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y')) == 31) {
                        $array = array('01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0 ,'13' => 0 ,'14' => 0 ,'15' => 0 ,'16' => 0 ,'17' => 0 ,'18' => 0 ,'19' => 0 ,'20' => 0 ,'21' => 0 ,'22' => 0 ,'23' => 0,'24' => 0 ,'25' => 0 ,'26' => 0 ,'27' => 0 ,'28' => 0 ,'29' => 0 ,'30' => 0 ,'31' => 0);
                    } elseif (cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y')) == 30) {
                        $array = array('01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0 ,'13' => 0 ,'14' => 0 ,'15' => 0 ,'16' => 0 ,'17' => 0 ,'18' => 0 ,'19' => 0 ,'20' => 0 ,'21' => 0 ,'22' => 0 ,'23' => 0,'24' => 0 ,'25' => 0 ,'26' => 0 ,'27' => 0 ,'28' => 0 ,'29' => 0 ,'30' => 0);
                    } elseif (cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y')) == 29) {
                        $array = array('01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0 ,'13' => 0 ,'14' => 0 ,'15' => 0 ,'16' => 0 ,'17' => 0 ,'18' => 0 ,'19' => 0 ,'20' => 0 ,'21' => 0 ,'22' => 0 ,'23' => 0,'24' => 0 ,'25' => 0 ,'26' => 0 ,'27' => 0 ,'28' => 0 ,'29' => 0);
                    } elseif (cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y')) == 28) {
                        $array = array('01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0 ,'13' => 0 ,'14' => 0 ,'15' => 0 ,'16' => 0 ,'17' => 0 ,'18' => 0 ,'19' => 0 ,'20' => 0 ,'21' => 0 ,'22' => 0 ,'23' => 0,'24' => 0 ,'25' => 0 ,'26' => 0 ,'27' => 0 ,'28' => 0);
                    }
                    
                    $code = 'd';
                    $main_range = 'This Month';
                    $title = "Monthly";
                    break;
            }
        }

        $start = '';
        $end = '';
        $first_code = '';
        $second_code = '';

        if(!empty($range) && !in_array($range, array('Today','Yesterday','This Week','This Month','Last Month'))) {
            $arr = explode('-', $range);
            if (preg_match('~(0[1-9]|1[012])[- /.](0[1-9]|[12][0-9]|3[01])[- /.](19|20)\d\d~m', $arr[0]) && preg_match('~(0[1-9]|1[012])[- /.](0[1-9]|[12][0-9]|3[01])[- /.](19|20)\d\d~m', $arr[1])) {
                $start = pSQL($arr[0]);
                $end = pSQL($arr[1]);
                $this_start = strtotime($start);
                $this_end = strtotime($end);
                $month_days = cal_days_in_month(CAL_GREGORIAN, date('m',strtotime($start)), date('Y',strtotime($start)));
                $diff = abs(strtotime($end) - strtotime($start));
                $years = floor($diff / (365*60*60*24));
                $months = floor(($diff - $years * 365*60*60*24) / (30*60*60*24));
                $days = floor(($diff - $years * 365*60*60*24 - $months*30*60*60*24)/ (60*60*24));
                if ($years >= 1) {
                    $array = array();
                    for ($i=date('Y',strtotime($start)); $i <= date('Y',strtotime($end)); $i++) { 
                        $array["'".$i."'"] = 0;
                    }
                    $main_range = 'Custom';
                    $code = 'Y';
                    $title = "Yearly";
                } elseif ($months >= 1) {
                    $array = array('01' => 0 ,'02' => 0 ,'03' => 0 ,'04' => 0 ,'05' => 0 ,'06' => 0 ,'07' => 0 ,'08' => 0 ,'09' => 0 ,'10' => 0 ,'11' => 0 ,'12' => 0);
                    $code = 'm';
                    $main_range = 'This Year';
                    $title = "Monthly";
                    if (date('Y',strtotime($start)) == date('Y',strtotime($end))) {
                        $array = array();
                        for ($i=date('m',strtotime($start)); $i <= date('m',strtotime($end)); $i++) { 
                            $array["'".$i."'"] = 0;
                        }
                        $code = 'm';
                        $main_range = 'Custom';
                    } else {
                        $month = cal_days_in_month(CAL_GREGORIAN, date('m',strtotime($start)), date('Y',strtotime($start)));
                        $array = array();
                        for ($i=(int)date('m',strtotime($start)); $i <= 12; $i++) { 
                            $array["'".(int) $i.'-'.date('Y',strtotime($start))."'"] = 0;
                        }
                        for ($i=1; $i <= (int)date('m',strtotime($end)); $i++) { 
                            $array["'". (int) $i.'-'.date('Y',strtotime($end))."'"] = 0;
                        }
                        $first_code = 'm';
                        $second_code = 'Y';
                        $main_range = 'Custom';
                    }
                } elseif ($days > 7) {
                    if (date('m',strtotime($start)) == date('m',strtotime($end))) {
                        $array = array();
                        for ($i=date('d',strtotime($start)); $i <= date('d',strtotime($end)); $i++) { 
                            $array["'".(int)$i."'"] = 0;
                        }
                        $code = 'd';
                        $main_range = 'Custom';
                    } else {
                        $month = cal_days_in_month(CAL_GREGORIAN, date('m',strtotime($start)), date('Y',strtotime($start)));
                        $array = array();
                        for ($i=(int)date('d',strtotime($start)); $i <= $month; $i++) { 
                            $array["'".(int)$i.'-'.date('m',strtotime($start))."'"] = 0;
                        }
                        for ($i=1; $i <= (int) date('d',strtotime($end)); $i++) { 
                            $array["'".(int)$i.'-'.date('m',strtotime($end))."'"] = 0;
                        }
                        $first_code = 'd';
                        $second_code = 'm';
                        $main_range = 'Custom';
                    }

                    $title = "Daily";
                } elseif ($days >= 1 && $days < 8) {
                    $title = "Daily";
                    $code = 'l';
                    $array = array('Saturday' => 0 , 'Sunday' => 0 , 'Monday' => 0 , 'Tuesday' => 0 , 'Wednesday' => 0 , 'Thursday' => 0 , 'Friday' => 0);
                    if (date('m',strtotime($start)) == date('m',strtotime($end))) {
                        $array = array();
                        for ($i=date('d',strtotime($start)); $i <= date('d',strtotime($end)); $i++) { 
                            $array["'".(int)$i."'"] = 0;
                        }
                        $code = 'd';
                        $main_range = 'Custom';
                    } else {
                        $month = cal_days_in_month(CAL_GREGORIAN, date('m',strtotime($start)), date('Y',strtotime($start)));
                        $array = array();
                        for ($i=(int) date('d',strtotime($start)); $i <= $month; $i++) { 
                            $array["'".(int) $i.'-'.date('m',strtotime($start))."'"] = 0;
                        }
                        for ($i=1; $i <= (int) date('d',strtotime($end)); $i++) { 
                            $array["'".(int) $i.'-'.date('m',strtotime($end))."'"] = 0;
                        }

                        $first_code = 'd';
                        $second_code = 'm';
                        $main_range = 'Custom';
                    }
                }
            }
        }

        $sales_chart = $array;
        $orders_chart = $array;
        $customers_chart = $array;
        $writer_payments_chart = $array;
        $editor_payments_chart = $array;
        $income_chart = $array;
        $tips_revenue_chart = $array;
        $top_customers_chart = $array;
        $top_disciplines_chart = $array;
        $website_stats_chart = $array;

        $chartLabels = array();

        if ($main_range == 'Today' || $main_range == 'Yesterday') {
            $chartLabels = array(
                '00 AM',
                '1 AM',
                '2 AM',
                '3 AM',
                '4 AM',
                '5 AM',
                '6 AM',
                '7 AM',
                '8 AM',
                '9 AM',
                '10 AM',
                '11 AM',
                '12 PM',
                '1 PM',
                '2 PM',
                '3 PM',
                '4 PM',
                '5 PM',
                '6 PM',
                '7 PM',
                '8 PM',
                '9 PM',
                '10 PM',
                '11 PM'
            );
        } elseif ($main_range == 'This Week') {
            $chartLabels = array( 'Saturday', 'Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday');
        } elseif ( $main_range == 'This Month' || $main_range == 'Last Month' ) {
            if ($month_days == 31) {
                $chartLabels = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31];
            } elseif ($month_days == 30) {
                $chartLabels = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30];
            } elseif ($month_days == 29) {
                $chartLabels = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29];
            } elseif ($month_days == 28) {
                $chartLabels = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28];
            }
        } elseif ($main_range == 'This Year') {
            $chartLabels = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct','Nov','Dec'];
        } elseif ($main_range == 'Custom') {
            $customChartLabels = array_keys($array);

            foreach($customChartLabels as $chartLabelKey) {
                $monthNameKey = str_replace("'", '', $chartLabelKey);
                $dateObj = DateTime::createFromFormat('!m', (int) $monthNameKey);
                $monthName = $dateObj->format('M');
                $chartLabels[] = "'" . $monthName . "'";
            }
        }

        $chart['chartLabels'] = $chartLabels;

        if($user->is_admin) {
            $thisStartDate = DateUtils::date($this_start);
            $thisEndDate = DateUtils::date($this_end);

            // get insights
            if(!$hasRange) {
                /** current month customers */
                $insights['current_month_customers'] = Db::getInstance()->getValue("SELECT COUNT(customer_id) FROM " . Db::prefix('customer') . " WHERE reg_date BETWEEN DATE_FORMAT(CURDATE() ,'%Y-%m-01') AND CURDATE()");
                /** last month customers */
                $insights['previous_month_customers'] = Db::getInstance()->getValue('SELECT COUNT(customer_id) FROM '. Db::prefix('customer') . ' WHERE YEAR(reg_date) = YEAR(CURRENT_DATE - INTERVAL 1 MONTH) AND MONTH( reg_date ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');
                /** pct monthly customers */
                $insights['pct_change_monthly_customers'] = pct_change($insights['previous_month_customers'], $insights['current_month_customers']);
                
                /** current month orders */
                $insights['current_month_orders'] = Db::getInstance()->getValue("SELECT COUNT(order_id) FROM " . Db::prefix('order') . " WHERE is_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE()) AND MONTH(date_add) = MONTH(CURRENT_DATE())");
                /** last month orders */
                $insights['previous_month_orders'] = Db::getInstance()->getValue('SELECT COUNT(order_id) FROM '. Db::prefix('order') . ' WHERE is_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE - INTERVAL 1 MONTH) AND MONTH( date_add ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');
                /** pct monthly orders */
                $insights['pct_change_monthly_orders'] = pct_change($insights['previous_month_orders'], $insights['current_month_orders']);

                /* current month sales */
                $insights['current_month_sales'] = Db::getInstance()->getValue("SELECT COALESCE(SUM(total),0) FROM " . Db::prefix('order') . " WHERE is_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE()) AND MONTH(date_add) = MONTH(CURRENT_DATE())");
                /* previous month sales */
                $insights['previous_month_sales'] = Db::getInstance()->getValue('SELECT COALESCE(SUM(total),0) FROM '. Db::prefix('order') . ' WHERE is_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE - INTERVAL 1 MONTH) AND MONTH( date_add ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');
                /** pct monthly sales */
                $insights['pct_change_monthly_sales'] = pct_change($insights['previous_month_sales'], $insights['current_month_sales']);

                /* current writer month payments */
                $insights['current_writer_month_payments'] = Db::getInstance()->getValue("SELECT COALESCE(SUM(writer_pay),0) FROM " . Db::prefix('order') . " WHERE is_writer_paid = 1 AND YEAR(writer_paid_at) = YEAR(CURRENT_DATE()) AND MONTH(writer_paid_at) = MONTH(CURRENT_DATE())");
                /* previous writer month payments */
                $insights['previous_writer_month_payments'] = Db::getInstance()->getValue('SELECT COALESCE(SUM(writer_pay),0) FROM '. Db::prefix('order') . ' WHERE is_writer_paid = 1 AND YEAR(writer_paid_at) = YEAR(CURRENT_DATE - INTERVAL 1 MONTH) AND MONTH( writer_paid_at ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');

                if(Configuration::get('EDITOR_ACCOUNT_ENABLED')) {
                    /* current editor month payments */
                    $insights['current_editor_month_payments'] = Db::getInstance()->getValue("SELECT COALESCE(SUM(editor_pay),0) FROM " . Db::prefix('order') . " WHERE is_editor_paid = 1 AND YEAR(editor_paid_at) = YEAR(CURRENT_DATE()) AND MONTH(editor_paid_at) = MONTH(CURRENT_DATE())");
                    /* previous editor month payments */
                    $insights['previous_editor_month_payments'] = Db::getInstance()->getValue('SELECT COALESCE(SUM(editor_pay),0) FROM '. Db::prefix('order') . ' WHERE is_editor_paid = 1 AND YEAR(editor_paid_at) = YEAR(CURRENT_DATE - INTERVAL 1 MONTH) AND MONTH( editor_paid_at ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');

                    /* current month payments */
                    $insights['current_month_payments'] = $insights['current_writer_month_payments']+$insights['current_editor_month_payments'];
                    /* previous month payments */
                    $insights['previous_month_payments'] = $insights['previous_writer_month_payments']+$insights['previous_editor_month_payments'];
                } else {
                    /* current month payments */
                    $insights['current_month_payments'] = $insights['current_writer_month_payments'];
                    /* previous month payments */
                    $insights['previous_month_payments'] = $insights['previous_writer_month_payments'];
                }

                /** pct monthly payments */
                $insights['pct_change_monthly_payments'] = pct_change($insights['previous_month_payments'], $insights['current_month_payments']);
            }
            
            /* total customers */
            $sql = new DbQuery();
            $sql->select("COUNT(customer_id)")
                ->from("customer", "c");

            if($hasRange) {
                $sql->where('c.`reg_date` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('c.`reg_date` <= \'' . pSQL($thisEndDate) . '\'');
            }
                
            $insights['total_customers'] = Db::getInstance()->getValue($sql);

            /* total employees */
            $sql = new DbQuery();
            $sql->select("COUNT(employee_id)")
                ->from("employee", "e");

            if($hasRange) {
                $sql->where('e.`reg_date` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('e.`reg_date` <= \'' . pSQL($thisEndDate) . '\'');
            }

            $insights['total_employees'] = Db::getInstance()->getValue($sql);

            /* total orders */
            $sql = new DbQuery();
            $sql->select("COUNT(order_id)")
                ->from("order", "o");

            if($hasRange) {
                $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            }

            $insights['total_orders'] = Db::getInstance()->getValue($sql);

            /* total customer payments */
            $sql = new DbQuery();
            $sql->select("COALESCE(SUM(total),0)")
                ->from("order", "o")
                ->where("is_paid = 1");

            if($hasRange) {
                $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            }

            $insights['total_sales'] = Db::getInstance()->getValue($sql);

            /* total tips */
            if(Module::isInstalled('leavetip')) {
                $tipsEnabled = true;
                $sql = new DbQuery();
                $sql->select("COALESCE(SUM(amount),0) AS tips_revenue, COALESCE(SUM(writer_tip_amount+editor_tip_amount),0) AS tips_payout")
                    ->from("order_tip", "ot");
    
                if($hasRange) {
                    $sql->where('ot.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                    $sql->where('ot.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
                }

                $tips_data = Db::getInstance()->getRow($sql);
                    
                $insights['total_tips_revenue'] = $tips_data['tips_revenue'];
                $insights['total_tips_payout'] = $tips_data['tips_payout'];

                /* tips revenue chart */
                $sql = new DbQuery();
                $sql->select("ot.amount, ot.date_add")
                    ->from("order_tip", "ot")
                    ->where('ot.`date_add` >= \'' . pSQL($thisStartDate) . '\'')
                    ->where('ot.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
                $sales = Db::getInstance()->executeS($sql);

                foreach ($sales as $key => $value) {
                    $day = date($code, strtotime($value['date_add']) );
                    if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                        $day = "'". (int) date($first_code, $value['date_add']).'-'.(int)date($second_code, $value['date_add'])."'";
                    } elseif ($main_range == 'Custom') {
                        $day = "'". (int) date($code, strtotime($value['date_add']))."'";
                    }

                    if (in_array($day, array_keys($tips_revenue_chart))) {
                        $tips_revenue_chart[$day] += (float) $value['amount']; 
                    }
                }

                $chart['tips_revenue_chart'] = $tips_revenue_chart;
            }

            /* total employee payments */
            $sql = new DbQuery();
            $sql->select("COALESCE(SUM(writer_pay),0)");
            $sql->from("order", "o");
            $sql->where('o.`is_writer_paid` = 1');
            if($hasRange) {
                $sql->where('o.`writer_paid_at` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`writer_paid_at` <= \'' . pSQL($thisEndDate) . '\'');
            }

            $insights['writer_total_payments'] = Db::getInstance()->getValue($sql);
            
            if(Configuration::get('EDITOR_ACCOUNT_ENABLED')) {
                $sql = new DbQuery();
                $sql->select("COALESCE(SUM(editor_pay),0)");
                $sql->from("order", "o");
                $sql->where('o.`is_editor_paid` = 1');
                if($hasRange) {
                    $sql->where('o.`editor_paid_at` >= \'' . pSQL($thisStartDate) . '\'');
                    $sql->where('o.`editor_paid_at` <= \'' . pSQL($thisEndDate) . '\'');
                }

                $insights['editor_total_payments'] = Db::getInstance()->getValue($sql);

                $insights['total_payments'] = $insights['writer_total_payments']+$insights['editor_total_payments'];
            } else {
                $insights['total_payments'] = $insights['writer_total_payments'];
            }

            /**** CHARTS ***/

            /* get customers */
            $sql = new DbQuery();
            $sql->select("c.reg_date")
                ->from("customer", "c")
                ->where('c.`reg_date` >= \'' . pSQL($thisStartDate) . '\'')
                ->where('c.`reg_date` <= \'' . pSQL($thisEndDate) . '\'');

            $customers = Db::getInstance()->executeS($sql);

            foreach ($customers as $key => $value) {
                $day = date($code, strtotime($value['reg_date']) );
                if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                    $day = "'".(int)date($first_code,$value['date_add']).'-'.(int)date($second_code,$value['date_add'])."'";
                }
                elseif ($main_range == 'Custom') {
                    $day = "'". (int) date($code, strtotime($value['reg_date']))."'";
                }
                if (in_array($day, array_keys($customers_chart))) {
                    $customers_chart[$day] += 1; 
                }
            }

            $chart['customers_chart'] = $customers_chart;

            /* sales chart */
            $sql = new DbQuery();
            $sql->select("o.total, o.date_add")
                ->from("order", "o")
                ->where('o.`is_paid` = 1')
                ->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'')
                ->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            $sales = Db::getInstance()->executeS($sql);

            foreach ($sales as $key => $value) {
                $day = date($code, strtotime($value['date_add']) );
                if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                    $day = "'". (int) date($first_code, $value['date_add']).'-'.(int)date($second_code, $value['date_add'])."'";
                } elseif ($main_range == 'Custom') {
                    $day = "'". (int) date($code, strtotime($value['date_add']))."'";
                }

                if (in_array($day, array_keys($sales_chart))) {
                    $sales_chart[$day] += (float) $value['total']; 
                }
            }

            $chart['sales_chart'] = $sales_chart;

            /* payments chart */
            $sql = new DbQuery();
            $sql->select("o.writer_pay, o.writer_paid_at");
            $sql->from("order", "o");
            $sql->where('o.`is_writer_paid` = 1');
            $sql->where('o.`writer_paid_at` >= \'' . pSQL($thisStartDate) . '\'');
            $sql->where('o.`writer_paid_at` <= \'' . pSQL($thisEndDate) . '\'');
            $writer_payments = Db::getInstance()->executeS($sql);

            foreach ($writer_payments as $key => $value) {
                $day = date($code, strtotime($value['writer_paid_at']) );
                if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                    $day = "'". (int) date($first_code, $value['writer_paid_at']).'-'.(int)date($second_code, $value['writer_paid_at'])."'";
                } elseif ($main_range == 'Custom') {
                    $day = "'". (int) date($code, strtotime($value['writer_paid_at']))."'";
                }

                if (in_array($day, array_keys($writer_payments_chart))) {
                    $writer_payments_chart[$day] += (float) $value['writer_pay']; 
                }
            }
            
            $chart['writer_payments_chart'] = $writer_payments_chart;

            /* editor payments chart */
            if(Configuration::get('EDITOR_ACCOUNT_ENABLED')) {
                $sql = new DbQuery();
                $sql->select("o.editor_pay, o.editor_paid_at");
                $sql->from("order", "o");
                $sql->where('o.`is_editor_paid` = 1');
                $sql->where('o.`editor_paid_at` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`editor_paid_at` <= \'' . pSQL($thisEndDate) . '\'');
                $writer_payments = Db::getInstance()->executeS($sql);

                foreach ($writer_payments as $key => $value) {
                    $day = date($code, strtotime($value['editor_paid_at']) );
                    if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                        $day = "'". (int) date($first_code, $value['editor_paid_at']).'-'.(int)date($second_code, $value['editor_paid_at'])."'";
                    } elseif ($main_range == 'Custom') {
                        $day = "'". (int) date($code, strtotime($value['editor_paid_at']))."'";
                    }

                    if (in_array($day, array_keys($editor_payments_chart))) {
                        $editor_payments_chart[$day] += (float) $value['editor_pay']; 
                    }
                }
                
                $chart['editor_payments_chart'] = $editor_payments_chart;
            } else {
                $chart['editor_payments_chart'] = [];
            }

            /* orders chart */
            $sql = new DbQuery();
            $sql->select("o.date_add")
                ->from("order", "o")
                ->where('o.`is_paid` = 1')
                ->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'')
                ->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            $orders = Db::getInstance()->executeS($sql);

            foreach ($orders as $key => $value) {
                $day = date($code, strtotime($value['date_add']) );
                if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                    $day = "'".(int)date($first_code,$value['date_add']).'-'.(int)date($second_code,$value['date_add'])."'";
                } elseif ($main_range == 'Custom') {
                    $day = "'". (int) date($code, strtotime($value['date_add']))."'";
                }
                if (in_array($day, array_keys($orders_chart))) {
                    $orders_chart[$day] += 1; 
                }
            }

            $chart['orders_chart'] = $orders_chart;

            $sql = new DbQuery();
            $sql->select("c.customer_id, c.name, c.email, COUNT(o.order_id) AS c_total_orders, SUM(o.total) AS c_total_spent");
            $sql->from("customer", "c");
            $sql->innerJoin("order", "o", "c.customer_id = o.customer_id");
            $sql->where("o.is_paid = 1");

            if($hasRange) {
                $sql->where('c.`reg_date` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('c.`reg_date` <= \'' . pSQL($thisEndDate) . '\'');
            }

            $sql->groupBy('c.customer_id');
            $sql->orderBy('c_total_spent DESC');
            $sql->limit(10);
            $insights['top_customers'] = Db::getInstance()->executeS($sql);

            /** Top Disciplines */
            $sql = new DbQuery();
            $sql->select('o.topic_category_id, d.title, COUNT(*) AS discipline_count');
            $sql->from('order', 'o');
            $sql->innerJoin('discipline', 'd', 'd.discipline_id = o.topic_category_id');
            $sql->where('o.topic_category_id != ' . (int) Discipline::DISCIPLINE_OTHER );

            if($hasRange) {
                $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            }
            
            $sql->groupBy('o.topic_category_id');
            $sql->orderBy('discipline_count DESC');
            $sql->limit( 10 );

            $insights['top_disciplines'] = Db::getInstance()->executeS($sql);

            $orderStats = $user->getStats('o.`date_add` >= \'' . pSQL($thisStartDate) . '\' AND o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');

            $insights['statistics'] = array(
                'progress' => $orderStats['progress'],
                'finished' => $orderStats['finished'],
                'unpaid' => $orderStats['unpaid'],
                'revision' => $orderStats['revision'],
                'cancelled' => $orderStats['cancelled']
            );
        } else {
            $thisStartDate = DateUtils::date($this_start);
            $thisEndDate = DateUtils::date($this_end);

            /* get income */
            $sql = new DbQuery();
            $sql->from("order", "o");

            if($user->is_writer) {
                $sql->select("COALESCE(SUM(writer_pay),0) AS pay, o.date_add");
                $sql->where("writer_id = " . (int) $user->id );
                $sql->where("is_writer_paid = 1");
            } elseif ($user->is_editor) {
                $sql->select("COALESCE(SUM(editor_pay),0) AS pay, o.date_add");
                $sql->where("editor_id = " . (int) $user->id );
                $sql->where("is_editor_paid = 1");
            } elseif ($user->is_sub_admin) {
                $sql->select("COALESCE(SUM(ordermanager_pay),0) AS pay, o.date_add");
                $sql->where("ordermanager_id = " . (int) $user->id );
                $sql->where("is_ordermanager_paid = 1");
            }

            $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
            $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');

            $income = Db::getInstance()->executeS($sql);

            foreach ($income as $key => $value) {
                $day = date($code, strtotime($value['date_add']) );
                if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                    $day = "'". (int) date($first_code, $value['date_add']).'-'.(int)date($second_code, $value['date_add'])."'";
                } elseif ($main_range == 'Custom') {
                    $day = "'". (int) date($code, strtotime($value['date_add']))."'";
                }

                if (in_array($day, array_keys($income_chart))) {
                    $income_chart[$day] += (float) $value['pay']; 
                }
            }

            $chart['income_chart'] = $income_chart;

            /* orders chart */
            $sql = new DbQuery();
            $sql->select("o.date_add")
                ->from("order", "o")
                ->where('o.`is_paid` = 1')
                ->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'')
                ->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');

            if($user->is_writer) {
                $sql->where("writer_id = " . (int) $user->id);
            } elseif ($user->is_editor) {
                $sql->where("editor_id = " . (int) $user->id);
            } elseif ($user->is_sub_admin) {
                $sql->where("ordermanager_id = " . (int) $user->id);
            }

            $orders = Db::getInstance()->executeS($sql);

            foreach ($orders as $key => $value) {
                $day = date($code, strtotime($value['date_add']) );
                if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                    $day = "'".(int)date($first_code,$value['date_add']).'-'.(int)date($second_code,$value['date_add'])."'";
                } elseif ($main_range == 'Custom') {
                    $day = "'". (int) date($code, strtotime($value['date_add']))."'";
                }
                if (in_array($day, array_keys($orders_chart))) {
                    $orders_chart[$day] += 1; 
                }
            }

            $chart['orders_chart'] = $orders_chart;

            if(!$hasRange) {
                /* current month payments */
                $insights['current_month_payment'] = Db::getInstance()->getValue("SELECT COALESCE(SUM(writer_pay),0) FROM " . Db::prefix('order') . " WHERE is_writer_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE()) AND MONTH(date_add) = MONTH(CURRENT_DATE())");
                /* previous month payments */
                $insights['previous_month_payment'] = Db::getInstance()->getValue('SELECT COALESCE(SUM(writer_pay),0) FROM '. Db::prefix('order') . ' WHERE is_writer_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE()) AND MONTH( date_add ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');
                /** pct monthly payments */
                $insights['pct_change_monthly_payment'] = pct_change($insights['previous_month_payment'], $insights['current_month_payment']);

                /** current month orders */
                $sql = new DbQuery();
                $sql->select('COUNT(order_id)')->from('order', 'o');
                $sql->where('is_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE()) AND MONTH(date_add) = MONTH(CURRENT_DATE())');

                if($user->is_writer) {
                    $sql->where("writer_id = " . (int) $user->id);
                } elseif ($user->is_editor) {
                    $sql->where("editor_id = " . (int) $user->id);
                } elseif ($user->is_sub_admin) {
                    $sql->where("ordermanager_id = " . (int) $user->id);
                }

                $insights['current_month_orders'] = Db::getInstance()->getValue($sql);

                /** last month orders */
                $sql = new DbQuery();
                $sql->select('COUNT(order_id)')->from('order', 'o');
                $sql->where('is_paid = 1 AND YEAR(date_add) = YEAR(CURRENT_DATE()) AND MONTH( date_add ) = MONTH( DATE_SUB(CURDATE(),INTERVAL 1 MONTH ))');
                if($user->is_writer) {
                    $sql->where("writer_id = " . (int) $user->id);
                } elseif ($user->is_editor) {
                    $sql->where("editor_id = " . (int) $user->id);
                } elseif ($user->is_sub_admin) {
                    $sql->where("ordermanager_id = " . (int) $user->id);
                }

                $insights['previous_month_orders'] = Db::getInstance()->getValue($sql);
                
                /** pct monthly orders */
                $insights['pct_change_monthly_orders'] = pct_change($insights['previous_month_orders'], $insights['current_month_orders']);
            }

            if($hasRange) {
                /* total writer payments */
                $sql = new DbQuery();
                if($user->is_writer) {
                    $sql->select("COALESCE(SUM(writer_pay),0)")->from("order", "o");
                    $sql->where("writer_id = " . (int) $user->id);
                    $sql->where("is_writer_paid = 1");
                } elseif ($user->is_editor) {
                    $sql->select("COALESCE(SUM(editor_pay),0)")->from("order", "o");
                    $sql->where("editor_id = " . (int) $user->id);
                    $sql->where("is_editor_paid = 1");
                } elseif ($user->is_sub_admin) {
                    $sql->select("COALESCE(SUM(ordermanager_pay),0)")->from("order", "o");
                    $sql->where("ordermanager_id = " . (int) $user->id);
                    $sql->where("is_ordermanager_paid = 1");
                }

                $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');

                $insights['total_paid'] = Db::getInstance()->getValue($sql);
            } else {
                $insights['total_paid'] = $user->total_paid; 
            }

            /* total orders */
            $sql = new DbQuery();
            $sql->select("COUNT(order_id)")
                ->from("order", "o");

            if($user->is_writer) {
                $sql->where("writer_id = " . (int) $user->id);
            } elseif ($user->is_editor) {
                $sql->where("editor_id = " . (int) $user->id);
            } elseif ($user->is_sub_admin) {
                $sql->where("ordermanager_id = " . (int) $user->id);
            }

            if($hasRange) {
                $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            }

            $insights['total_orders'] = Db::getInstance()->getValue($sql);


            /** Disciplines */
            $sql = new DbQuery();
            $sql->select('o.topic_category_id, d.title, COUNT(*) AS discipline_count');
            $sql->from('order', 'o');
            $sql->innerJoin('discipline', 'd', 'd.discipline_id = o.topic_category_id');
            $sql->where('o.topic_category_id != ' . (int) Discipline::DISCIPLINE_OTHER );

            if($user->is_writer) {
                $sql->where("o.writer_id = " . (int) $user->id);
            } elseif ($user->is_editor) {
                $sql->where("o.editor_id = " . (int) $user->id);
            } elseif ($user->is_sub_admin) {
                $sql->where("o.ordermanager_id = " . (int) $user->id);
            }

            if($hasRange) {
                $sql->where('o.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                $sql->where('o.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
            }
            
            $sql->groupBy('o.topic_category_id');
            $sql->orderBy('discipline_count DESC');
            $sql->limit( 10 );

            $insights['top_disciplines'] = Db::getInstance()->executeS($sql);

            /* total tips */
            if(($user->is_writer || $user->is_editor) && Module::isInstalled('leavetip')) {
                $tipsEnabled = true;
                $sql = new DbQuery();
                $sql->from("order_tip", "ot");
                $sql->innerJoin('order', 'o', 'o.order_id = ot.order_id');

                if($user->is_writer) {
                    $sql->select("COALESCE(SUM(writer_tip_amount),0) AS tip_amount");
                    $sql->where("o.writer_id = " . (int) $user->id);
                } elseif ($user->is_editor) {
                    $sql->select("COALESCE(SUM(editor_tip_amount),0) AS tip_amount");
                    $sql->where("o.editor_id = " . (int) $user->id);
                } 

                $total_tips_query = clone $sql;
    
                if($hasRange) {
                    $total_tips_query->where('ot.`date_add` >= \'' . pSQL($thisStartDate) . '\'');
                    $total_tips_query->where('ot.`date_add` <= \'' . pSQL($thisEndDate) . '\'');
                }

                $tips_data = Db::getInstance()->getRow($total_tips_query);
                $insights['total_tips_income'] = $tips_data['tip_amount'];
                
                /* tips revenue chart */
                $sql = new DbQuery();
                $sql->select("ot.amount, ot.date_add")
                    ->from("order_tip", "ot")
                    ->innerJoin('order', 'o', 'o.order_id = ot.order_id');

                if($user->is_writer) {
                    $sql->select("ot.writer_tip_amount AS amount, ot.date_add");
                    $sql->where("o.writer_id = " . (int) $user->id);
                } elseif ($user->is_editor) {
                    $sql->select("ot.editor_tip_amount AS amount, ot.date_add");
                    $sql->where("o.editor_id = " . (int) $user->id);
                } 
                
                $sql->where('ot.`date_add` >= \'' . pSQL($thisStartDate) . '\'')
                    ->where('ot.`date_add` <= \'' . pSQL($thisEndDate) . '\'');

                $sales = Db::getInstance()->executeS($sql);

                foreach ($sales as $key => $value) {
                    $day = date($code, strtotime($value['date_add']) );
                    if ($main_range == 'Custom' && !empty($first_code) && !empty($second_code)) {
                        $day = "'". (int) date($first_code, $value['date_add']).'-'.(int)date($second_code, $value['date_add'])."'";
                    } elseif ($main_range == 'Custom') {
                        $day = "'". (int) date($code, strtotime($value['date_add']))."'";
                    }

                    if (in_array($day, array_keys($tips_revenue_chart))) {
                        $tips_revenue_chart[$day] += (float) $value['amount']; 
                    }
                }

                $chart['tips_income_chart'] = $tips_revenue_chart;
            }

            $insights['outstanding_payment'] = $user->getOutstanding(); 
            $insights['last_pay'] = $user->last_pay; 
        }
        
        $smarty->assign([
            'tips_enabled' => $tipsEnabled,
            'insights' => $insights,
            'chart' => $chart,
            'module_no_header' => true,
            'main_range' => $main_range,
            'range' => $range,
            'start' => $start,
            'end' => $end,
            'hasRange' => $hasRange,
            'view' => 'overview'
        ]);

        return $this->getTemplateContent('overview');
    }

}