<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class Paystack extends PaymentModule
{
    /**
    * Paystack configuration
    */
    const PAYSTACK_ENABLED = 'PAYSTACK_ENABLED';
    const PAYSTACK_MODE = 'PAYSTACK_MODE';
    const PAYSTACK_PUBLIC_KEY = 'PAYSTACK_PUBLIC_KEY';
    const PAYSTACK_SECRET_KEY = 'PAYSTACK_SECRET_KEY';

    public function __construct()
    {
        $this->name = 'paystack';
        $this->icon = 'fab fa-paystack';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Paystack';
        $this->description = 'Set up Paystack as a payment method and enable users to pay without leaving your website';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    { 
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $paystackConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::PAYSTACK_ENABLED,
                self::PAYSTACK_MODE,
                self::PAYSTACK_PUBLIC_KEY,
                self::PAYSTACK_SECRET_KEY,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'paystackConfiguration' => $paystackConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updatePaystackSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        Configuration::updateValue('PAYMENT_METHOD_DEFAULT', ArrayUtils::has($payload, 'default_method_enabled') ? $this->name : '' , false, $site_id);
        Configuration::updateValue(self::PAYSTACK_ENABLED, (ArrayUtils::get($payload, 'paystack_enabled') == 'on') ? true : false , false, $site_id);
        Configuration::updateValue(self::PAYSTACK_MODE, ArrayUtils::get($payload, 'paystack_mode'), false, $site_id);
        Configuration::updateValue(self::PAYSTACK_PUBLIC_KEY, ArrayUtils::get($payload, 'paystack_public_key'), false, $site_id);
        Configuration::updateValue(self::PAYSTACK_SECRET_KEY, ArrayUtils::get($payload, 'paystack_secret_key'), false, $site_id);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}