<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2020 Proxim
 * @version    1.0
 * @since      File available since Release 1.0
*/

use Proxim\Presenter\AbstractLazyArray;
use Doctrine\Common\Annotations\AnnotationException;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\Discipline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class PrePaperLazyArray extends AbstractLazyArray {
    /** @var Application */
    private $application;

    /** @var ObjectPresenter */
    private $objectPresenter;

    /** @var PrePaper */
    private $prepaper;

    /**
     * PrePaperLazyArray constructor.
     *
     * @throws AnnotationException
     * @throws ReflectionException
     */
    public function __construct(PrePaper $prepaper)
    {
        $this->application = Application::getInstance();
        $this->objectPresenter = new ObjectPresenter();

        $this->prepaper = $prepaper;

        parent::__construct();
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getId()
    {
        return (int) $this->prepaper->id;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getSlug()
    {
        return $this->prepaper->slug;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->prepaper->title;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getQuestion()
    {
        return $this->prepaper->question;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getAcademicLevel()
    {
        $prepaper = $this->prepaper;
        $academicLevelObj = new AcademicLevel((int) $prepaper->academic_level_id);

        $academicLevel = (array) $academicLevelObj;

        return $academicLevel;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getPaperTypeId()
    {
        return (int) $this->prepaper->paper_type_id;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getPaperType()
    {
        $prepaper = $this->prepaper;

        $paperType = new PaperType( (int) $prepaper->paper_type_id );
        $paperType = $this->objectPresenter->present($paperType);

        return $paperType;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getPaperTypeOption()
    {
        return $this->prepaper->paper_type_option;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getDisciplineId()
    {
        return (int) $this->prepaper->topic_category_id;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getDiscipline()
    {
        $prepaper = $this->prepaper;

        $discipline = new Discipline( (int) $prepaper->topic_category_id );
        $paperDiscipline = $this->objectPresenter->present($discipline);

        return $paperDiscipline;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDisciplineOption()
    {
        return $this->prepaper->topic_category_option;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getPaperFormatId()
    {
        return (int) $this->prepaper->paper_format_id;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getPaperFormat()
    {
        $prepaper = $this->prepaper;

        $paperFormat = new PaperFormat( (int) $prepaper->paper_format_id );
        $paperFormat = $this->objectPresenter->present($paperFormat);

        return $paperFormat;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getPaperFormatOption()
    {
        return $this->prepaper->paper_format_option;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getPrice()
    {
        return (float) $this->prepaper->price;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getPriceFormatted()
    {
        $priceFormatted = formatPrice($this->prepaper->price);
        return $priceFormatted;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getSpacing()
    {
        return $this->prepaper->spacing;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getPages()
    {
        return (int) $this->prepaper->pages;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getWords()
    {
        return (int) $this->prepaper->words;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCharts()
    {
        return (int) $this->prepaper->charts;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSlides()
    {
        return (int) $this->prepaper->slides;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSources()
    {
        return (int) $this->prepaper->sources;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSales()
    {
        return (float) $this->prepaper->sales;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSalesFormatted()
    {
        $salesFormatted = formatPrice($this->prepaper->sales);
        return $salesFormatted;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getViews()
    {
        return (int) $this->prepaper->views;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getDownloads()
    {
        return (int) $this->prepaper->downloads;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getFiles()
    {
        $files = $this->prepaper->getFiles();

        $prepaperFiles = array();

        foreach($files as $prepaperFile) {
            $prepaperFile = new PrePaperFile( (int) $prepaperFile['prepaper_file_id'] );
            if(Validate::isLoadedObject($prepaperFile)) {
                $prepaperFile = $this->objectPresenter->present($prepaperFile);
                $prepaperFile['icon'] = $prepaperFile['extension'];
                $prepaperFile['size_formatted'] = formatBytes($prepaperFile['size']);
                $prepaperFile['download_link'] =  $this->application->base_uri . "/prepapers/downloadPaper?fileId=" . (int) $prepaperFile['id'];
                $prepaperFiles[] = $prepaperFile;
            }
        }

        return $prepaperFiles;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getPreviews()
    {
        $previews = $this->prepaper->getPreviews();

        $prepaperPreviews = array();

        foreach($previews as $prepaperPreview) {
            $prepaperPreview = new PrePaperPreview( (int) $prepaperPreview['prepaper_preview_id'] );
            if(Validate::isLoadedObject($prepaperPreview)) {
                $prepaperPreview = $this->objectPresenter->present($prepaperPreview);
                $prepaperPreview['image_source'] = $this->application->base_uri . '/uploads/' . $prepaperPreview['source'];
                $prepaperPreviews[] = $prepaperPreview;
            }
        }

        return $prepaperPreviews;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getPayments()
    {
        $payments = $this->prepaper->getPayments();

        $prepaperPayments = array();

        foreach($payments as $prepaperPayment) {
            $prepaperPayment = new PrePaperPayment( (int) $prepaperPayment['prepaper_payment_id'] );
            if(Validate::isLoadedObject($prepaperPayment)) {
                $prepaperPayment = $this->objectPresenter->present($prepaperPayment);
                $prepaperPayment['amount_formatted'] = formatPrice($prepaperPayment['amount']);
                $prepaperPayments[] = $prepaperPayment;
            }
        }

        return $prepaperPayments;
    }


    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getCreatedAt()
    {
        return $this->prepaper->date_add;
    }
}