<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class reCAPTCHA extends Module
{
    const reCAPTCHA_ENABLED = 'reCAPTCHA_ENABLED';
    const reCAPTCHA_SITE_KEY = 'reCAPTCHA_SITE_KEY';
    const reCAPTCHA_SECRET_KEY = 'reCAPTCHA_SECRET_KEY';

    public function __construct()
    { 
        $this->name = 'reCAPTCHA';
        $this->icon = 'fas fa-lock';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'reCAPTCHA';
        $this->description = 'reCAPTCHA is an effective security solution that protects your website from spam entries';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user; 
        return $user->is_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        return $this->registerHook([
            'displaySigninForm',
            'displaySignupForm',
            'actionAuthenticationBefore',
            'actionSignupBefore'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    /**
     * return_json
     * 
     * @param array $response
     * @return json
     */
    public function returnJson($response = array()) {
        header('Content-Type: application/json');
        exit(json_encode($response));
    }

    public function getContent()
    {
        $reCAPTCHAConfiguration = Configuration::getMultiple([
            self::reCAPTCHA_ENABLED,
            self::reCAPTCHA_SITE_KEY,
            self::reCAPTCHA_SECRET_KEY
        ]);

        $this->smarty->assign([
            'reCAPTCHAConfiguration' => $reCAPTCHAConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookDisplaySigninForm() {
        if(Configuration::get(self::reCAPTCHA_ENABLED)) {
            $this->smarty->assign([
                'reCAPTCHA_site_key' => Configuration::get(self::reCAPTCHA_SITE_KEY)
            ]);
            return $this->showTemplate('form.reCAPTCHA');
        }
    }

    public function hookDisplaySignupForm() {
        if(Configuration::get(self::reCAPTCHA_ENABLED)) {
            $this->smarty->assign([
                'reCAPTCHA_site_key' => Configuration::get(self::reCAPTCHA_SITE_KEY)
            ]);
            return $this->showTemplate('form.reCAPTCHA');
        }
    }

    public function hookActionAuthenticationBefore() {
        $app = $this->application;
        $payload = $app->request->post();

        if(Configuration::get(self::reCAPTCHA_ENABLED)) {
            $recaptcha_response = ArrayUtils::get($payload, 'g-recaptcha-response');
            try {
                $recaptcha = new \ReCaptcha\ReCaptcha( Configuration::get(self::reCAPTCHA_SECRET_KEY) );
                $resp = $recaptcha->verify($recaptcha_response, Tools::getIp());
                if (!$resp->isSuccess()) {
                    $this->returnJson([
                        "error" => true,
                        "message" => "The security check is incorrect. Please try again"
                    ]);
                }
            } catch (Exception $e) {

            }
        }
    }

    public function hookActionSignupBefore() {
        $app = $this->application;
        $payload = $app->request->post();

        if(Configuration::get(self::reCAPTCHA_ENABLED)) {
            $recaptcha_response = ArrayUtils::get($payload, 'g-recaptcha-response');
            try {
                $recaptcha = new \ReCaptcha\ReCaptcha( Configuration::get(self::reCAPTCHA_SECRET_KEY) );
                $resp = $recaptcha->verify($recaptcha_response, Tools::getIp());
                if (!$resp->isSuccess()) {
                    $this->returnJson([
                        "error" => true,
                        "message" => "The security check is incorrect. Please try again"
                    ]);
                }
            } catch (Exception $e) {

            }
        }
    }

    public function updatereCAPTCHASettings() {
        $app = $this->application;
        $payload = $app->request->post();
        
        Configuration::updateValue(self::reCAPTCHA_ENABLED, ArrayUtils::has($payload, 'reCAPTCHA_enabled') ? true : false);
        Configuration::updateValue(self::reCAPTCHA_SITE_KEY, ArrayUtils::get($payload, 'reCAPTCHA_site_key'));
        Configuration::updateValue(self::reCAPTCHA_SECRET_KEY, ArrayUtils::get($payload, 'reCAPTCHA_secret_key'));

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}