<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Util\ArrayUtils;

class RevisionDialog extends Module
{
    /**
     * @var string
     */
    private $css_path;

    public function __construct()
    {
        $this->name = 'revisiondialog';
        $this->icon = 'fa fa-warning';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Revision Dialog';
        $this->description = 'Allow writers to confirm an order revision before accessing the dashboard';

        $this->css_path = $this->_path . 'css/';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin || $user->is_sub_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        }

        $this->registerHook([
            'displayFooterAfter',
            'actionDispatcherBeforeRun',
            'actionControllerSetMedia'
        ]);
    }

    public function createTables() {
        $result = Db::getInstance()->getRow("SELECT * FROM ". Db::prefix('order'));
        if(!isset($result['writer_revision_confirmed'])) {
            if (!Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix('order') . " ADD (
                    `writer_revision_confirmed` TINYINT(1) NOT NULL DEFAULT '0'
                );
            ")) {
                return false;
            }
        } 

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookActionControllerSetMedia() {
        $app = Application::getInstance();
        $app->controller->addCss($this->css_path . 'revisiondialog.css');
    }

    public function hookActionDispatcherBeforeRun() {
        $app = $this->application;
        $user = $app->user;
        $params = $app->request->get();

        if(ArrayUtils::has($params, 'seen_revison_dialog')) {
            if($user->is_writer) {
                Db::getInstance()->update(
                    'order',
                    ['writer_revision_confirmed' => 1],
                    'status_id = ' . (int) Order::REVISION . ' AND writer_revision_confirmed = 0 AND writer_id = ' . (int) $user->id
                );
            }
        }
    }

    public function hookDisplayFooterAfter() {
        $app = $this->application;
        $smarty = $this->smarty;
        $controller = $app->controller;
        $user = $app->user;
        
        if($user->is_writer) {
            if($user->order_stats['revision'] > 0) {
                $sql = new DbQuery();
                $sql->select('order_id');
                $sql->from('order');
                $sql->where('status_id = ' . (int) Order::REVISION);
                $sql->where('writer_revision_confirmed = 0');
                $sql->where('writer_id = ' . (int) $user->id );

                $result = Db::getInstance()->executeS($sql);
                if(count($result)) {
                    $controller->init();
                    $smarty->assign([
                        'revision_orders' => $user->order_stats['revision'],
                        'accepted_image' => $app->base_uri . '/content/modules/'. $this->name .'/images/accepted.png'
                    ]);
                    $this->showTemplate('dialog');
                }
            }
        }
    }
}
