<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class SecureTrading extends PaymentModule
{
    /**
    * SecureTrading configuration
    */
    const SECURETRADING_ENABLED = 'SECURETRADING_ENABLED';
    const SECURETRADING_REFERENCE = 'SECURETRADING_REFERENCE';

    public function __construct()
    {
        $this->name = 'securetrading';
        $this->icon = 'fa fa-credit-card';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Secure Trading';
        $this->description = 'Set up Secure Trading as a payment method and enable users to pay without leaving your website';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    {
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $secureTradingConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::SECURETRADING_ENABLED,
                self::SECURETRADING_REFERENCE,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'secureTradingConfiguration' => $secureTradingConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateSecureTradingSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        if( ArrayUtils::has($payload, 'default_method_enabled') ) {
            Configuration::updateValue('PAYMENT_METHOD_DEFAULT', $this->name , false, $site_id);
        }

        Configuration::updateValue(self::SECURETRADING_ENABLED, ArrayUtils::has($payload, 'securetrading_enabled') ? true : false , false, $site_id);
        Configuration::updateValue(self::SECURETRADING_REFERENCE, ArrayUtils::get($payload, 'securetrading_reference'), false, $site_id);
        
        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}