<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;
use SendGrid\Mail\Attachment;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class SendGridMailer extends Module
{
    const SENDGRID_ENABLED = 'SENDGRID_ENABLED';
    const SENDGRID_API_KEY = 'SENDGRID_API_KEY';

    public function __construct()
    {
        $this->name = 'sendgridmailer';
        $this->icon = 'fa fa-envelope';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->price = 150;
        $this->displayName = 'SendGrid';
        $this->description = 'Send Email Newsletters, Password Resets, Promotional Emails, Shipping Notifications With Confidence';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin || $user->is_sub_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook([
            'displayAdminEditSiteGeneralTabContent',
            'actionAfterSiteSettingsGeneralUpdate',
            'actionEmailSendBefore',
            'actionMailAlterMessageBeforeSend'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $sendgridConfiguration = Configuration::getMultiple([
                self::SENDGRID_API_KEY,
                self::SENDGRID_ENABLED
            ]
        );

        $this->smarty->assign([
            'sendgridConfiguration' => $sendgridConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookDisplayAdminEditSiteGeneralTabContent() {
        $application = $this->application;
        $smarty = $this->smarty;

        $sendgridConfiguration = Configuration::getMultiple([
                self::SENDGRID_ENABLED
            ],
            $application->site->id
        );
        $smarty->assign('sendgridConfiguration', $sendgridConfiguration);
        
        return $this->showTemplate('tab_content');
    } 

    public function hookActionMailAlterMessageBeforeSend($payload) {
        if(!Configuration::get(self::SENDGRID_ENABLED)) {
            return true;
        }

        $message = ArrayUtils::get($payload, 'message');
        $fileAttachment = ArrayUtils::get($payload, 'fileAttachment');
        $htmlMessage = ArrayUtils::get($payload, 'htmlMessage');

        $email = new \SendGrid\Mail\Mail();
        $from = $message->getFrom();
        $to = $message->getTo();

        if (is_array($from)) {
            foreach ($from as $addr => $addrName) {
                $addr = trim($addr);
                $email->setFrom($addr, $addrName);
            }
        }

        if (is_array($to)) {
            foreach ($to as $addr => $addrName) {
                $addr = trim($addr);
                $email->addTo($addr, $addrName);
            }
        }

        $email->setSubject( $message->getSubject() );
        $email->addContent(
            "text/html", $htmlMessage
        );

        if ($fileAttachment && !empty($fileAttachment)) {
            // Multiple attachments?
            if (!is_array(current($fileAttachment))) {
                $fileAttachment = array($fileAttachment);
            }

            foreach ($fileAttachment as $attachment) {
                if (isset($attachment['content']) && isset($attachment['name']) && isset($attachment['mime'])) {
                    $att = new Attachment();
                    $att->setContent( $attachment['content'] );
                    $att->setType( $attachment['mime'] );
                    $att->setFilename( $attachment['name'] );
                    $att->setDisposition("attachment");
                    $email->addAttachment( $att );
                }
            }
        }

        try {
            $sendgrid = new SendGrid( Configuration::get(self::SENDGRID_API_KEY) );
            $res = $sendgrid->send($email);
        } catch (Exception $e) {
        }

        return false;
    } 

    public function updateSendGridSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        Configuration::updateValue(self::SENDGRID_ENABLED, ArrayUtils::has($payload, 'sendgrid_enabled') ? true : false);
        Configuration::updateValue(self::SENDGRID_API_KEY, ArrayUtils::get($payload, 'sendgrid_api_key'));

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    } 

    public function hookActionAfterSiteSettingsGeneralUpdate( $params ) {
        $site_id = ArrayUtils::get($params, 'site_id');
        $payload = ArrayUtils::get($params, 'payload');

        $site = new Site( (int) $site_id );
        if(Validate::isLoadedObject($site)) {
            Configuration::updateValue(self::SENDGRID_ENABLED, ArrayUtils::has($payload, 'sendgrid_enabled') ? true : false, false, $site->id);
        }
    }
}
