<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Currency;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Order\AdditionalService;
use Proxim\Order\Order;
use Proxim\Pager;
use Proxim\PDF\PDF;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

define('CURRENT_TRX_MODULE_DIR', realpath(dirname(__FILE__)));

require_once(CURRENT_TRX_MODULE_DIR . '/classes/HTMLTemplateTransactionsList.php');

class Transactions extends Module
{
    public function __construct()
    {
        $this->name = 'transactions';
        $this->icon = 'fas fa-receipt';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Transactions';
        $this->description = 'View a list of all payment transactions made by your customers';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user; 
        return $user->is_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        Db::getInstance()->Execute("
            CREATE TABLE IF NOT EXISTS " . Db::prefix("payment") . " ( 
                `payment_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `node_id` BIGINT(20) UNSIGNED NOT NULL,
                `node_type` VARCHAR(20) DEFAULT NULL,
                `currency_id` BIGINT(20) UNSIGNED DEFAULT NULL,
                `conversion_rate` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `payment_method` VARCHAR(255) DEFAULT NULL,
                `transaction_id` VARCHAR(255) DEFAULT NULL,
                `amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`payment_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ");

        $this->installData();
        $this->updatePayments();

        return $this->registerHook([
            'displayAdminNavBarBeforeEnd'
        ]);
    }

    public function installData() {
        $sql = new DbQuery();
        $sql->select('p.*');
        $sql->from('paypal_payment', 'p');
        $payments = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        foreach($payments as $payment) {
            Db::getInstance()->insert(
                'payment',
                array(
                    'node_type' => $payment['node_type'],
                    'node_id' => $payment['node_id'],
                    'amount' => $payment['amount'],
                    'date_add' => $payment['date_add']
                )
            );
        }

        if(Module::isEnabled('wallet')) {
            $sql = new DbQuery();
            $sql->select('*');
            $sql->from('wallet_transaction');
            $wallet_transactions = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
            foreach($wallet_transactions as $wallet_transaction) {
                $exists = Db::getInstance()->getRow("SELECT payment_id FROM " . Db::prefix('payment') . " WHERE node_type = 'wallet' AND transaction_id = '" . pSQL($wallet_transaction['transaction_id']) . "'");
                if(!$exists) {
                    $customer = Db::getInstance()->getRow("SELECT customer_id FROM " . Db::prefix('customer_wallet') . " WHERE wallet_id = " . (int) $wallet_transaction['wallet_id']);

                    Db::getInstance()->insert(
                        'payment',
                        array(
                            'node_type' => $wallet_transaction['source'],
                            'node_id' => $customer['customer_id'],
                            'transaction_id' => $wallet_transaction['transaction_id'],
                            'payment_method' => $wallet_transaction['payment_method'],
                            'currency_id' => $wallet_transaction['currency_id'],
                            'amount' => $wallet_transaction['amount'],
                            'amount' => $wallet_transaction['amount'],
                            'date_add' => $wallet_transaction['date_add']
                        )
                    );
                }
            }
        } 

        if(Module::isEnabled('leavetip')) {
            $sql = new DbQuery();
            $sql->select('*');
            $sql->from('order_tip');
            $order_tips = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
            foreach($order_tips as $order_tip) {
                $exists = Db::getInstance()->getRow("SELECT payment_id FROM " . Db::prefix('payment') . " WHERE node_type = 'tip' AND transaction_id = '" . pSQL($order_tip['transaction_id']) . "'");
                if(!$exists) {
                    Db::getInstance()->insert(
                        'payment',
                        array(
                            'node_type' => 'tip',
                            'transaction_id' => $order_tip['transaction_id'],
                            'payment_method' => $order_tip['payment_method'],
                            'currency_id' => $order_tip['currency_id'],
                            'node_id' => $order_tip['order_id'],
                            'amount' => $order_tip['amount'],
                            'date_add' => $order_tip['date_add']
                        )
                    );
                }
            }
        } 
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminNavBarBeforeEnd() {
        if($this->checkAccess()) {
            return $this->showTemplate('nav_item');
        }
    }

    public function parseTransaction($transaction) {
        $options = array();
        $currency = new Currency( (int) $transaction['currency_id'] );
        if(!Validate::isLoadedObject($currency)) {
            $currency_id = Currency::getIdByIsoCode('USD');
            $currency = new Currency( (int) $currency_id );
        }

        $options['currency'] = array(
            'symbol' => $currency->symbol,
            'rate' => (float) $transaction['conversion_rate'] ? (float) $transaction['conversion_rate'] : (float) $currency->conversion_rate
        );

        $payment_method = $transaction['payment_method'];
        if(!$payment_method) {
            if($transaction['node_type'] == 'order') {
                $order = new Order( (int) $transaction['node_id']);
                if(Validate::isLoadedObject($order)) {
                    $payment_method = $order->payment_method;
                }
            } 
        }
        
        $trx = array(
            'node_type' => $transaction['node_type'],
            'node_id' => $transaction['node_id'],
            'transaction_id' => $transaction['transaction_id'],
            'payment_method' => $payment_method,
            'amount' => $transaction['amount'],
            'amountFormatted' => formatPrice($transaction['amount'], $options),
            'date_add' => DateUtils::convertToISOFormat($transaction['date_add'])
        );

        return $trx;
    }

    public function getContent() {
        $app = $this->application;
        $smarty = $this->smarty;
        $request = $app->request->get();
        $selected_page = (int) ArrayUtils::get($request, 'page');
        ArrayUtils::remove($request, 'page');

        $date_from = ArrayUtils::get($request, 'from');
        $date_to = ArrayUtils::get($request, 'to');
        $payment_method = ArrayUtils::get($request, 'payment_method');
        $export = ArrayUtils::has($request, 'export');

        $payment_methods = PaymentModule::getInstalledPaymentModules();
        if(Module::isEnabled('wallet')) {
            $payment_methods[] = [
                'name' => 'wallet'
            ];
        }

        $amounts = $transactions = $totals = array();

        $sql = new DbQuery();
        $sql->select('p.*');
        $sql->from('payment', 'p');
        $sql->orderBy('p.date_add DESC');

        if($payment_method) {
            $sql->where('p.`payment_method` = "' . pSQL($payment_method) . '"');
        }

        if ($date_from || $date_to) {
            if ( $date_from ) {
                $dateFromSQL = DateUtils::date(strtotime($date_from));
            } 

            if ( $date_to ) {
                $dateToSQL = DateUtils::date(strtotime($date_to));
            } 

            if ( $date_from && $date_to ) {
                $sql->where('p.`date_add` BETWEEN "' . pSQL($dateFromSQL) . '" AND "' . pSQL($dateToSQL) . '"');
            } elseif( $date_from ) {
                $sql->where('p.`date_add` >= "' . pSQL($dateFromSQL) . '"');
            }  elseif( $date_to ) {
                $sql->where('p.`date_add` <= "' . pSQL($dateFromSQL) . '"');
            }
        }

        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        foreach( $result as $transaction ) {
            $options = array();

            $currency = new Currency( (int) $transaction['currency_id'] );
            if(!Validate::isLoadedObject($currency)) {
                $currency_id = Currency::getIdByIsoCode('USD');
                $currency = new Currency( (int) $currency_id );
            }

            $options['currency'] = array(
                'symbol' => $currency->symbol,
                'rate' => (float) $transaction['conversion_rate'] ? (float) $transaction['conversion_rate'] : (float) $currency->conversion_rate
            );

            if(!isset($totals[$currency->iso_code])) {
                $totals[$currency->iso_code] = array(
                    'symbol' => $currency->symbol,
                    'amount' => 0
                );
            }

            $totals[$currency->iso_code]['amount'] = $totals[$currency->iso_code]['amount'] + $transaction['amount']; 
        }

        foreach($totals as $currency_code => $total) {
            $options['currency'] = array(
                'symbol' => $total['symbol']
            );

            $amounts[$currency_code] = array(
                'iso_code' => $currency_code,
                'amount' => $total['amount'],
                'amountFormatted' => formatPrice($total['amount'], $options),
            );
        }

        $total_transactions = count($result);

        if( $total_transactions > 0) {

            if($export) {
                $this->exportTransactions($sql, $amounts);
            }

            $params['total_items'] = $total_transactions;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/transactions?page=%s";

            if( \is_array($request) && !empty($request)) {
                $params['url'] .= "&" . http_build_query($request);
            }

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            foreach( $result as $transaction ) {
                $transactions[] = $this->parseTransaction($transaction);
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'transactions',
            'payment_methods' => $payment_methods,
            'selected_page' => $selected_page,
            'selected_payment_method' => $payment_method,
            'date_from' => $date_from,
            'date_to' => $date_to,
            'amounts' => $amounts,
            'transactions' => $transactions,
            'module_no_header' => true
        ]);

        return $this->getTemplateContent('transactions');
    }

    public function exportTransactions($sql, $amounts) {
        $trx = $transactions = array();
        $trx['amounts'] = $amounts;

        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        foreach( $result as $transaction ) {
            $transactions[] = $this->parseTransaction($transaction);
        }

        $trx['transactions'] = $transactions;

        $pdf = new PDF([$trx], 'TransactionsList', $this->smarty );
        $pdf->render();
    }

    public function updatePayments() {
        $sql = new DbQuery();
        $sql->select('p.*');
        $sql->from('payment', 'p');
        $sql->orderBy('p.date_add DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        foreach( $result as $transaction ) {
            if(!$transaction['payment_method']) {
                switch($transaction['node_type']) {
                    case 'order':
                        $order = new Order( (int) $transaction['node_id']);
                        if(Validate::isLoadedObject($order)) {
                            Db::getInstance()->update(
                                'payment',
                                array(
                                    'payment_method' => $order->payment_method
                                ),
                                'payment_id = ' . (int) $transaction['payment_id']
                            );
                        }

                        break;

                    case 'add-payment':
                        $add_payment = new AdditionalService( (int) $transaction['node_id'] );
                        if(Validate::isLoadedObject($add_payment)) {
                            $order = new Order( (int) $add_payment->order_id );
                            if(Validate::isLoadedObject($order)) {
                                Db::getInstance()->update(
                                    'payment',
                                    array(
                                        'payment_method' => $order->payment_method
                                    ),
                                    'payment_id = ' . (int) $transaction['payment_id']
                                );
                            }
                        }
                        break;
                }
            }
        }
    }
}