<?php
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Pager;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class TrustPilot extends Module
{
    /**
    * TrustPilot configuration
    */
    const TRUSTPILOT_ENABLED = 'TRUSTPILOT_ENABLED';
    const TRUSTPILOT_TITLE = 'TRUSTPILOT_TITLE';
    const TRUSTPILOT_REVIEWS = 'TRUSTPILOT_REVIEWS';
    const TRUSTPILOT_STARS = 'TRUSTPILOT_STARS';
    const TRUSTPILOT_LINK = 'TRUSTPILOT_LINK';
    const TRUSTPILOT_SHOW_REVIEWS = 'TRUSTPILOT_SHOW_REVIEWS';

    public function __construct()
    {
        $this->name = 'trustpilot';
        $this->icon = 'fa fa-rating';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Trustpilot Reviews';
        $this->description = 'Collect and showcase verified reviews that consumers trust on Trustpilot\'s review platform.';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        } 

        Configuration::updateValue(self::TRUSTPILOT_ENABLED, true);
        Configuration::updateValue(self::TRUSTPILOT_TITLE, 'Excellent');
        Configuration::updateValue(self::TRUSTPILOT_STARS, '4.7');
        Configuration::updateValue(self::TRUSTPILOT_REVIEWS, '2,024');

        $this->registerHook([
            'actionBeforeAdminDelete'
        ]);

    }

    public function createTables() {
        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('trustpilot_review') . " (
                `review_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `author_name` VARCHAR(255) NOT NULL,
                `title` TEXT NOT NULL,
                `body` TEXT NOT NULL,  
                `stars` FLOAT(2,1) NOT NULL DEFAULT 1.0,
                `time` DATETIME DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`review_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        return true;
    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }
        
        Configuration::updateValue(self::TRUSTPILOT_ENABLED, false);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $app = $this->application;
        $params = $app->request->get();
        $smarty = $this->smarty;

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $sql = new DbQuery();
        $sql->select('review.*');
        $sql->from('trustpilot_review', 'review');
        $sql->orderBy('review.review_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_reviews = count($result);

        $reviews = array();
        if( $total_reviews > 0) {
            $params['total_items'] = $total_reviews;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/trustpilot/configure?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
            foreach($result as $review) {
                $body_formatted = Tools::nl2br($review['body']);
                $review['body_formatted'] = Tools::truncateString($body_formatted, 150);
                $review['stars_formatted'] = number_format($review['stars'], 1);
                $reviews[] = $review;
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $trustpilotConfiguration = Configuration::getMultiple([
            self::TRUSTPILOT_ENABLED,
            self::TRUSTPILOT_TITLE,
            self::TRUSTPILOT_REVIEWS,
            self::TRUSTPILOT_STARS,
            self::TRUSTPILOT_LINK,
            self::TRUSTPILOT_SHOW_REVIEWS
        ]);

        $smarty->assign([
            'total_reviews' => $total_reviews,
            'reviews' => $reviews,
            'trustpilotConfiguration' => $trustpilotConfiguration,
        ]);
        
        return $this->getTemplateContent('configure');
    }

    public function updateTrustpilotSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        Configuration::updateValue(self::TRUSTPILOT_TITLE, ArrayUtils::get($payload, 'trustpilot_title'), self::TRUSTPILOT_TITLE);
        Configuration::updateValue(self::TRUSTPILOT_REVIEWS, ArrayUtils::get($payload, 'trustpilot_reviews'), self::TRUSTPILOT_REVIEWS);
        Configuration::updateValue(self::TRUSTPILOT_STARS, ArrayUtils::get($payload, 'trustpilot_stars'), self::TRUSTPILOT_STARS);
        Configuration::updateValue(self::TRUSTPILOT_LINK, ArrayUtils::get($payload, 'trustpilot_link'), self::TRUSTPILOT_LINK);
        Configuration::updateValue(self::TRUSTPILOT_SHOW_REVIEWS, ArrayUtils::get($payload, 'trustpilot_show_reviews'), self::TRUSTPILOT_LINK);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }

    public function postReview() {
        $app = $this->application;
        $payload = $app->request->post();

        $author_name = ArrayUtils::get($payload, 'author_name');
        $title = ArrayUtils::get($payload, 'review_title');
        $body = ArrayUtils::get($payload, 'review_body');
        $stars = (float) ArrayUtils::get($payload, 'review_stars');
        $time = ArrayUtils::get($payload, 'date_time');

        $time = DateUtils::date(strtotime($time));

        if(!$title || !Validate::isCleanHtml($title)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Enter a valid title'
            ]);
        }

        if(!$body || !Validate::isCleanHtml($body)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Enter a valid review'
            ]);
        }

        if(!$stars || !Validate::isFloat($stars)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Enter a valid star rating'
            ]);
        }

        if(!$time || !Validate::isDate($time)) {
            return $app->sendResponse([
                'error' => true,
                'message' => 'Enter a valid date'
            ]);
        }

        if(ArrayUtils::has($payload, 'review_id')) {
            $review_id = ArrayUtils::get($payload, 'review_id');

            Db::getInstance()->update(
                'trustpilot_review',
                array(
                    'author_name' => $author_name,
                    'title' => pSQL($title),
                    'body' => pSQL($body),
                    'stars' => (float) $stars,
                    'time' => $time,
                    'date_add' => DateUtils::now()
                ),
                'review_id = ' . (int) $review_id
            );

            return $app->sendResponse([
                'success' => true,
                'message' => 'Review updated successfully'
            ]);
        } else {
            Db::getInstance()->insert(
                'trustpilot_review',
                array(
                    'author_name' => $author_name,
                    'title' => pSQL($title),
                    'body' => pSQL($body),
                    'stars' => (float) $stars,
                    'time' => $time,
                    'date_add' => DateUtils::now()
                )
            );

            return $app->controller->modal("SUCCESS", "Review Published", "Review published successfully");
        }
    }

    public function hookActionBeforeAdminDelete( $params ) {
        $id = ArrayUtils::get($params, 'id');
        $handle = ArrayUtils::get($params, 'handle');

        if($handle == "trustpilot_review") {
            Db::getInstance()->delete(
                'trustpilot_review',
                'review_id = ' . (int) $id
            );
        }
    }
}