<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Proxim\Addon\Theme\ThemeManagerBuilder;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use \Smarty;
use Proxim\User\Employee;
use Proxim\Order\Order;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Slim\BaseRequest;
use Proxim\Slim\BaseResponse;
use Proxim\Slim\Environment;
use Slim\Http\Util;
use Slim\Slim;
use VisualAppeal\AutoUpdate;

/**
 * Application
 */
class Application extends Slim {

    const VERSION = '1.0.2';

    const DEFAULT_CHECK_VERSION_DELAY_HOURS = 24;

    const DEFAULT_VALIDATION_DELAY_HOURS = 48;

    /** @var string */
    public $fingerprint;

    /** @var string */
    public $timezone;

    /** @var string */
    public $base_uri;

    /** @var string */
    public $theme_uri;

    /** @var string */
    public $theme_name;

    /** @var string */
    public $modules_uri;

    /** @var object User */
    public $user;

    /** @var object Site */
    public $site;

    /** @var object Cookie */
    public $cookie;

    /** @var object Smarty */
    public $smarty;

    /** @var object Controller */
    public $controller;

    /** @var object Theme */
    public $theme;

    /** @var array Currency */
    public $writer_currency;

    /**
     * @var Application
     */
    public static $instance = null;

    /**
     * @var bool
     */
    protected $booted = false;

    /**
     * Constructor
     */
    public function __construct( array $appSettings  ) {
        parent::__construct($appSettings);

        $this->container->singleton('environment', function () {
            return Environment::getInstance();
        });

        $this->container->singleton('response', function () {
            return new BaseResponse();
        });

        // Default request
        $this->container->singleton('request', function ($c) {
            return new BaseRequest($c['environment']);
        });

        // initilize theme
        $this->theme_name = Configuration::get('PROX_ACTIVE_THEME');

        if ($this->theme == null) {
            $this->setTheme();
        }

        $request = $this->request;

        $featureList = array(
            'user-agent=' . $request->getUserAgent(),
            'ip-address=' . Tools::getIp()
        );

        $this->fingerprint = sprintf("Proxim (%s)", implode('; ', $featureList));
        $base_uri = $request->getScheme() . '://' . $request->getHost();

        $this->base_uri = $base_uri;
        $activeThemeDir = PROX_DIR_THEMES . $this->theme_name;

        $this->theme_uri = $base_uri . '/content/themes/' . $this->theme_name;
        $this->modules_uri = $base_uri . '/content/modules';

        $smarty = new Smarty();
        $smarty->setCompileDir( PROX_DIR_THEMES . $this->theme_name . '/cache/compile');
        $smarty->setCacheDir( $activeThemeDir . '/cache' );
        $smarty->use_sub_dirs = true;
        $smarty->caching = false;
        $smarty->setTemplateDir( $activeThemeDir . '/templates' ); 
        $smarty->escape_html = true;

        smartyRegisterFunction($smarty, 'modifier', 'classname', 'smartyClassname');
        smartyRegisterFunction($smarty, 'modifier', 'classnames', 'smartyClassnames');
        smartyRegisterFunction($smarty, 'modifier', 'json_encode', array('Proxim\Tools', 'jsonEncode'));
        smartyRegisterFunction($smarty, 'modifier', 'json_decode', array('Proxim\Tools', 'jsonDecode'));
        smartyRegisterFunction($smarty, 'function', 'hook', 'smartyHook');

        $this->smarty = $smarty;

        /* Instantiate cookie */
        $cookie_lifetime = (int) Configuration::get('COOKIE_LIFETIME');
        if ($cookie_lifetime > 0) {
            $cookie_lifetime = time() + (max($cookie_lifetime, 1) * 3600);
        }
        
        $urls = array( $this->base_uri );

        $hook_urls = Hook::exec('actionCookieUrls', array('urls' => &$urls), null, true);

        if(is_array($hook_urls) && !empty($hook_urls)) {
            foreach($hook_urls as $module => $url) {
                $urls = array_merge( $urls, $url );
            }
    
            $urls = array_unique( $urls );
        }

        $this->cookie = new Cookie(
            'Proxim-user', '', $cookie_lifetime, 
            $urls, 
            $request->getScheme() === "https"
        );

        self::$instance = $this;

        $this->booted = true;
    }

    /**
     * Set the theme details from Json data.
     */
    public function setTheme()
    {
        $themeManagerBuilder = new ThemeManagerBuilder(Application::getInstance(), Db::getInstance());
        $themeRepository = $themeManagerBuilder->buildRepository($this instanceof Application ? $this : null);
        if (empty($this->theme_name)) {
            $this->theme_name = 'phantom';
        }

        $this->theme = $themeRepository->getInstanceByName($this->theme_name);
    }

    public function getFingerprint() {
        return $this->fingerprint;
    }

    public function updateUser(Employee $employee)
    {
        $cookie = $this->cookie;

        /* insert user token */
        $session_token = Tools::randomGen(32);
        $user_browser = Tools::getUserBrowser();
        $user_os = Tools::getUserPlatform();

		Db::getInstance()->insert(
			'user_session',
			array(
				'session_token' => $session_token,
				'user_id' => $employee->id,
				'user_browser' => $user_browser,
				'user_os' => $user_os,
				'user_ip' => Tools::getIp(),
				'date_add' => DateUtils::now()
			)
        );
        
        $session_id = Db::getInstance()->Insert_ID();
        
        $cookie->session_id = (int) $session_id;
        $cookie->session_token = $session_token;
        $cookie->session_browser = $user_browser;
        $cookie->session_os = $user_os;

        $cookie->user_id = $employee->id;
        $cookie->email = $employee->email;
        $cookie->password = $employee->password;
        $cookie->logged = 1;

        $cookie->write();
    }

    public function getNotificationEmails() 
    {
        $receivers = array();

        $notification_emails = Configuration::get('NOTIFICATION_EMAILS');
        if(!empty($notification_emails)) {
            $notification_emails = explode(',', $notification_emails);
            foreach($notification_emails as $notification_email) {
                $notification_email = trim($notification_email);
                if(Validate::isEmail($notification_email)) {
                    $receivers[] = $notification_email;
                }
            }
        }

        return $receivers;
    }

    public function runCronjob() {
        Hook::exec('actionBeforeCron');

        $next_cron_date = Configuration::get("NEXT_CRONJOB_RUN");
        if ($next_cron_date && !DateUtils::hasPassed($next_cron_date)) {
            return;
        }

        if(Configuration::get('SHARED_FOLDER_PERMISSIONS')) {
            @chmod(PROX_DIR_ROOT, 0777);
            @chmod(PROX_DIR_UPLOADS, 0777);
        }

        // approve orders after x days
        $sql = new DbQuery();
        $sql->select('o.*');
        $sql->from('order', 'o');
        $sql->where('status_id = ' . Order::DELIVERED);
        $autoApproveTime = (int) Configuration::get('ORDER_AUTOAPPROVE_TIME', null, 168);
        $sql->where( sprintf('delivered_at <= DATE_SUB(NOW(), INTERVAL %s HOUR)', $autoApproveTime));
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        foreach($result as $order) {
            $o = new Order();
            $order = $o->getObject( (array) $order );
            if(Validate::isLoadedObject($order)) {
                $order->status_id = Order::FINISHED;
                $order->approved_at = DateUtils::now();
                $order->update();
            }
        }

        $lastValidationCheck = Configuration::get('LAST_VALIDATION_CHECK');
        if(!$lastValidationCheck) {
            $lastValidationCheck = 0;
        }

        if ($lastValidationCheck < time() - (3600 * self::DEFAULT_VALIDATION_DELAY_HOURS)) {
            $host = Application::getInstance()->request->getHost();
            Tools::file_get_contents( PROX_API_URL .  "/check-activation.php?domain=" . $host . "&node=admin" );
            Configuration::updateValue('LAST_VALIDATION_CHECK', time());
        }

        Hook::exec('actionAfterCron');

        Configuration::updateValue("NEXT_CRONJOB_RUN", DateUtils::date(null, '+1 hour') );
    }

    public function cloudStorageEnabled() {
        $s3_enabled = Configuration::get('S3_ENABLED');
        $digitalocean_enabled = Configuration::get('DIGITALOCEAN_ENABLED');

        if($s3_enabled || $digitalocean_enabled) {
            return true;
        }

        return false;
    }

    /**
     * checkVersion ask to proxim.craftyworks.co if there is a new version. return an array if yes, false otherwise.
     *
     * @return mixed
     */
    public function checkVersion( $force = false )
    {
        $current_version = Configuration::get('PROX_VERSION');
        $lastCheck = Configuration::get('PROX_LAST_VERSION_CHECK');
        if(!$lastCheck) {
            $lastCheck = 0;
        }  

        if ($force || ($lastCheck < time() - (3600 * self::DEFAULT_CHECK_VERSION_DELAY_HOURS))) {
            $update = new AutoUpdate();
            $update->setBranch($this->request->getHost());
            $update->setCurrentVersion( $current_version );
            $update->setUpdateUrl(PROX_UPDATES_URL);

            if ($update->checkUpdate() === false) {
                return false;
            } 

            Configuration::updateValue('PROX_LAST_VERSION_CHECK', time());

            if ($update->newVersionAvailable()) {
                Configuration::updateValue('PROX_NEW_VERSION', true );
                return array('version' => $update->getLatestVersion() );
            } else {
                Configuration::updateValue('PROX_NEW_VERSION', false );
                return false;
            }

            // check theme
            try {
                if(PROX_ACTIVE_THEME != 'phantom') {
                    $licenseLink = Module::$purchase_link . "/verify.php?" . http_build_query([
                        'domain' => $this->request->getHost(),
                        'module' => PROX_ACTIVE_THEME
                    ]);
        
                    $ch = curl_init();
                    curl_setopt ($ch, CURLOPT_URL, $licenseLink);
                    curl_setopt ($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch,  CURLOPT_USERAGENT , "Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1)");
                    curl_setopt ($ch, CURLOPT_CONNECTTIMEOUT, 0);
                    $result = curl_exec($ch);
                    curl_close($ch);
        
                    $result = Tools::jsonDecode($result, true);
        
                    if(isset($result['error'])) {
                        Configuration::updateValue('PROX_ACTIVE_THEME', 'phantom');
                    }
                }
            } catch (\Exception $e) {

            }
        }

        return false;
    }

    /**
     * @inheritdoc
     */
    protected function mapRoute($args)
    {
        $pattern = array_shift($args);
        $callable = $this->resolveCallable(array_pop($args));
        $route = new \Slim\Route($pattern, $callable);
        $this->router->map($route);
        if (count($args) > 0) {
            $route->setMiddleware($args);
        }

        return $route;
    }

    /**
     * Resolve toResolve into a closure that that the router can dispatch.
     *
     * If toResolve is of the format 'class:method', then try to extract 'class'
     * from the container otherwise instantiate it and then dispatch 'method'.
     *
     * @param mixed $toResolve
     *
     * @return callable
     *
     * @throws \RuntimeException if the callable does not exist
     * @throws \RuntimeException if the callable is not resolvable
     */
    public function resolveCallable($toResolve)
    {
        $resolved = $toResolve;

        if (!is_callable($toResolve) && is_string($toResolve)) {
            // check for slim callable as "class:method"
            $callablePattern = '!^([^\:]+)\:([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*)$!';
            if (preg_match($callablePattern, $toResolve, $matches)) {
                $class = $matches[1];
                $method = $matches[2];

                if ($this->container->has($class)) {
                    $resolved = [$this->container->get($class), $method];
                } else {
                    if (!class_exists($class)) {
                        throw new \RuntimeException(sprintf('Callable %s does not exist', $class));
                    }
                    $resolved = [new $class($this), $method];
                }
            } else {
                // check if string is something in the DIC that's callable or is a class name which
                // has an __invoke() method
                $class = $toResolve;
                if ($this->container->has($class)) {
                    $resolved = $this->container->get($class);
                } else {
                    if (!class_exists($class)) {
                        throw new \RuntimeException(sprintf('Callable %s does not exist', $class));
                    }
                    $resolved = new $class($this);
                }
            }
        }

        if (!is_callable($resolved)) {
            throw new \RuntimeException(sprintf(
                '%s is not resolvable',
                is_array($toResolve) || is_object($toResolve) ? json_encode($toResolve) : $toResolve
            ));
        }

        return $resolved;
    }

    public function checkAccess($routeName) {
        $controller = $this->controller;
        $user = $this->user;
        $smarty = $this->smarty;

        /** Enforce required authentication. */
        if ($user->isLogged()) {

            if(!$this->request->isAjax()) {
                if ($user->is_banned) {
                    $controller->setTemplate('dashboard/account-disabled');
                    $controller->display();
                    return $this->stop();
                }

                if($user->is_application_rejected) {
                    $controller->setVars([
                        'page_title' => "Writer Applications"
                    ]); 
                    
                    $controller->setTemplate('dashboard/apply/rejected');
                    $controller->display();
                    return $this->stop();
                }

                if($user->is_applicant && ($routeName != 'started' && $routeName != 'application')) {
                    return $this->redirect('/started');
                }

                if(!$user->is_started && (!$user->is_admin && !$user->is_sub_admin)) {
                    if (!$user->is_profile_completed && ($routeName != 'started' && $routeName != 'application')) {
                        return $this->redirect('/started');
                    } 
    
                    if (!$user->is_application_completed && ($routeName != 'started' && $routeName != 'application')) {
                        return $this->redirect('/application');
                    } 

                    if(!$user->is_started && ($routeName != 'started' && $routeName != 'application')) {
                        return $this->redirect('/started');
                    }
                }
                
            }
        } else {
            if($this->request->isAjax()) {
                $controller->modal('LOGIN');
                return $this->stop();
            } else {
                $smarty->assign([
                    'do' => 'in',
                    'highlight' => 'You must sign in to see this page'
                ]);
    
                $controller->setVars([
                    'page_title' => "Sign in",
                ]);
                $controller->setTemplate('dashboard/sign');
                $controller->display();
                return $this->stop();
            }
        }
    }

    public function sendResponse( $data )
    {
        $response = parent::response();

        if (count($data)) {
            $response->header('Content-Type', 'application/json; charset=utf-8');
            $response->setBody(json_encode($data, JSON_UNESCAPED_UNICODE));
        }

        return $response;
    }

    public function run() {
        // Execute hook dispatcher before
        Hook::exec('actionDispatcherBeforeRun');
        
        parent::run();
    }
}