<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Exception;
use Proxim\Application;
use Proxim\Assets\AbstractAssetManager;
use Proxim\Assets\JavascriptManager;
use Proxim\Assets\StylesheetManager;
use Proxim\Util\ArrayUtils;

class Controller
{    
    /** @var string */
    public $template;

    /** @var array */
    public $templateVars = array(
        'page_title' => '',
        'body_classes' => ['admin_wrap', 'push-body']
    );

    /**
     * @var \Slim\Slim
     */
    protected $app;

    /**
     * List of CSS files.
     *
     * @var array
     */
    public $css_files = array();

    /**
     * List of JavaScript files.
     *
     * @var array
     */
    public $js_files = array();

    /**
     * @var object StylesheetManager
     */
    protected $stylesheetManager;

    /**
     * @var object JavascriptManager
     */
    protected $javascriptManager;

    public function __construct()
    {
        $this->app = Application::getInstance();
        $this->app->controller = $this;

        $this->stylesheetManager = new StylesheetManager(
            array($this->app->base_uri . '/' )
        );

        $this->javascriptManager = new JavascriptManager(
            array($this->app->base_uri . '/' )
        );

        if (
            !headers_sent() &&
            isset($_SERVER['HTTP_USER_AGENT']) &&
            (strpos($_SERVER['HTTP_USER_AGENT'], 'MSIE') !== false ||
            strpos($_SERVER['HTTP_USER_AGENT'], 'Trident') !== false)
        ) {
            header('X-UA-Compatible: IE=edge,chrome=1');
        }
    }

    /**
     * @param string $file
     * @param int $timeout
     *
     * @return bool
     */
    public function isFresh($file, $timeout = 604800)
    {
        if (($time = @filemtime(PROX_DIR_ROOT . $file)) && filesize(PROX_DIR_ROOT . $file) > 0) {
            return (time() - $time) < $timeout;
        }

        return false;
    }

    /** @var bool */
    protected static $is_proxim_up = true;

    /**
     * @param string $file_to_refresh
     * @param string $external_file
     *
     * @return bool
     */
    public function refresh($file_to_refresh, $external_file)
    {
        if (self::$is_proxim_up && $content = Tools::file_get_contents($external_file)) {
            return (bool) file_put_contents(PROX_DIR_ROOT . $file_to_refresh, $content);
        }

        self::$is_proxim_up = false;

        return false;
    }

    public function init()
    {
        $app = $this->app;
        $this->setMedia();
        $this->assignVariables();

        $ajax_page = ($app->request->isAjax() && $app->request->get('ajaxpage')) ? true : false;

        $this->app->smarty->assign(array(
            'ajax_page' => $ajax_page,
            'stylesheets' => $this->getStylesheets(),
            'javascript' => $this->getJavascript(),
            'js_custom_vars' => Media::getJsDef(),
            'css_files' => $app->controller->css_files,
            'js_files' => array_unique($app->controller->js_files)
        ));
    }

    public function response()
    {
        $data = null;
        if (func_num_args() > 0) {
            $data = ArrayUtils::get(func_get_args(), 0);
        }
        
        return $this->app->sendResponse($data);
    }

    public function modal() {
        $args = func_get_args();

        switch ($args[0]) {
            case 'LOGIN':
                $this->response( array("callback" => "modal('#modal-login')") );
                break;
            case 'MESSAGE':
                $this->response( array("callback" => "modal('#modal-message', {title: '".$args[1]."', message: '".addslashes($args[2])."'})") );
                break;
            case 'ERROR':
                $this->response( array("callback" => "modal('#modal-error', {title: '".$args[1]."', message: '".addslashes($args[2])."'})") );
                break;
            case 'SUCCESS':
                if(isset($args[3])) {
                    $this->response( array("callback" => "modal('#modal-success', {title: '".$args[1]."', message: '".addslashes($args[2])."', redirect_link: '".addslashes($args[3])."'})") );
                } else {
                    $this->response( array("callback" => "modal('#modal-success', {title: '".$args[1]."', message: '".addslashes($args[2])."'})") );
                }
                break;
            default:
                if(isset($args[1])) {
                    $this->response( array("callback" => "modal('".$args[0]."', ".$args[1].")") );
                } else {
                    $this->response( array("callback" => "modal('".$args[0]."')") );
                }
                break;
        }
    }

    public function showError() {
        $app = $this->app;
        
        $args = func_get_args();
        if(count($args) > 1) {
            $title = $args[0];
            $message = $args[1];
        } else {
            switch ($args[0]) {
                case 'DB_ERROR':
                    $title = "Database Error";
                    $message = "<div class='text-left'><h1>"."Error establishing a database connection"."</h1>
                                <p>"."This either means that the username and password information in your config.php file is incorrect or we can't contact the database server at localhost. This could mean your host's database server is down."."</p>
                                <ul>
                                    <li>"."Are you sure you have the correct username and password?"."</li>
                                    <li>"."Are you sure that you have typed the correct hostname?"."</li>
                                    <li>"."Are you sure that the database server is running?"."</li>
                                </ul>
                                <p>"."If you're unsure what these terms mean you should probably contact your host. If you still need help you can always email support at"." <a href='mailto:davis@davisonpro.dev'>davis@davisonpro.dev</a></p>
                                </div>";
                    break;

                case 'SQL_ERROR':
                    $title = "Database Error";
                    $message = "An error occurred while writing to database. Please try again later";
                    if(PROX_DEBUG) {
                        $backtrace = debug_backtrace();
                        $line=$backtrace[0]['line'];
                        $file=$backtrace[0]['file'];
                        $message .= "<br><br><small>This error function was called from line $line in file $file</small>";
                    }
                    break;

                case 'SQL_ERROR_THROWEN':
                    $message = "An error occurred while writing to database. Please try again later";
                    if(PROX_DEBUG) {
                        $backtrace = debug_backtrace();
                        $line=$backtrace[0]['line'];
                        $file=$backtrace[0]['file'];
                        $message .= "<br><br><small>This error function was called from line $line in file $file</small>";
                    }
                    throw new Exception($message);
                    break;

                case '404':
                    header('HTTP/1.0 404 Not Found');
                    $title = "404 Not Found";
                    $message = "The requested URL was not found on this server. That's all we know";
                    if(PROX_DEBUG) {
                        $backtrace = debug_backtrace();
                        $line=$backtrace[0]['line'];
                        $file=$backtrace[0]['file'];
                        $message .= "<br><br><small>This error function was called from line $line in file $file</small>";
                    }
                    break;

                case '400':
                    header('HTTP/1.0 400 Bad Request');
                    if(PROX_DEBUG) {
                        $backtrace = debug_backtrace();
                        $line=$backtrace[0]['line'];
                        $file=$backtrace[0]['file'];
                        exit("This error function was called from line $line in file $file");
                    }
                    exit;

                case '403':
                    header('HTTP/1.0 403 Access Denied');
                    if(PROX_DEBUG) {
                        $backtrace = debug_backtrace();
                        $line=$backtrace[0]['line'];
                        $file=$backtrace[0]['file'];
                        exit("This error function was called from line $line in file $file");
                    }
                    exit;
                
                default:
                    $title = 'Error';
                    $message = "There is some thing went wrong";
                    if(PROX_DEBUG) {
                        $backtrace = debug_backtrace();
                        $line=$backtrace[0]['line'];
                        $file=$backtrace[0]['file'];
                        $message .= "<br><br>"."<small>This error function was called from line $line in file $file</small>";
                    }
                    break;
            }
        }

        if($app->request->isAjax()) {
            return $this->response([
                'error' => true,
                'message' => $message
            ]);
        } else {
            $smarty = $app->smarty;
            $smarty->assign([
                'title' => $title,
                'message' => $message
            ]); 
    
            $html = $smarty->fetch('error.tpl');
            echo trim($html);
        }
       
        exit;
    }

    /**
     * Sets template file for page content output.
     *
     * @param string $template
     */
    public function setTemplate($template)
    {
        $this->template = $template;
    }

    /**
     * Sets template variables.
     *
     * @param string $vars
     */
    public function setVars($vars = null)
    {
        $this->templateVars = array_merge($this->templateVars, $vars);
    }

    public function registerStylesheet($id, $relativePath, $params = array())
    {
        if (!is_array($params)) {
            $params = array();
        }

        $default_params = [
            'media' => AbstractAssetManager::DEFAULT_MEDIA,
            'priority' => AbstractAssetManager::DEFAULT_PRIORITY,
            'inline' => false,
            'server' => 'local',
        ];

        $params = array_merge($default_params, $params);

        $this->stylesheetManager->register($id, $relativePath, $params['media'], $params['priority'], $params['inline'], $params['server']);
    }

    public function unregisterStylesheet($id)
    {
        $this->stylesheetManager->unregisterById($id);
    }

    public function registerJavascript($id, $relativePath, $params = array())
    {
        if (!is_array($params)) {
            $params = array();
        }

        $default_params = [
            'position' => AbstractAssetManager::DEFAULT_JS_POSITION,
            'priority' => AbstractAssetManager::DEFAULT_PRIORITY,
            'inline' => false,
            'attributes' => null,
            'server' => 'local',
        ];

        $params = array_merge($default_params, $params);

        $this->javascriptManager->register($id, $relativePath, $params['position'], $params['priority'], $params['inline'], $params['attributes'], $params['server']);
    }

    public function unregisterJavascript($id)
    {
        $this->javascriptManager->unregisterById($id);
    }

    /**
     * Adds a new stylesheet(s) to the page header.
     *
     * @param string|array $css_uri Path to CSS file, or list of css files like this : array(array(uri => media_type), ...)
     * @param string $css_media_type
     * @param int|null $offset
     * @param bool $check_path
     *
     * @return true
     */
    public function addCSS($css_uri, $css_media_type = 'all', $offset = null, $check_path = true)
    {
        if (!is_array($css_uri)) {
            $css_uri = array($css_uri);
        }

        foreach ($css_uri as $css_file => $media) {
            if (is_string($css_file) && strlen($css_file) > 1) {
                if ($check_path) {
                    $css_path = Media::getCSSPath($css_file, $media);
                } else {
                    $css_path = array($css_file => $media);
                }
            } else {
                if ($check_path) {
                    $css_path = Media::getCSSPath($media, $css_media_type);
                } else {
                    $css_path = array($media => $css_media_type);
                }
            }

            $key = is_array($css_path) ? key($css_path) : $css_path;
            if ($css_path && (!isset($this->css_files[$key]) || ($this->css_files[$key] != reset($css_path)))) {
                $size = count($this->css_files);
                if ($offset === null || $offset > $size || $offset < 0 || !is_numeric($offset)) {
                    $offset = $size;
                }

                $this->css_files = array_merge(array_slice($this->css_files, 0, $offset), $css_path, array_slice($this->css_files, $offset));
            }
        }
    }

    /**
     * Removes CSS stylesheet(s) from the queued stylesheet list.
     *
     * @param string|array $css_uri Path to CSS file or an array like: array(array(uri => media_type), ...)
     * @param string $css_media_type
     * @param bool $check_path
     */
    public function removeCSS($css_uri, $css_media_type = 'all', $check_path = true)
    {
        if (!is_array($css_uri)) {
            $css_uri = array($css_uri);
        }

        foreach ($css_uri as $css_file => $media) {
            if (is_string($css_file) && strlen($css_file) > 1) {
                if ($check_path) {
                    $css_path = Media::getCSSPath($css_file, $media);
                } else {
                    $css_path = array($css_file => $media);
                }
            } else {
                if ($check_path) {
                    $css_path = Media::getCSSPath($media, $css_media_type);
                } else {
                    $css_path = array($media => $css_media_type);
                }
            }

            if (
                $css_path
                && isset($this->css_files[key($css_path)])
                && ($this->css_files[key($css_path)] == reset($css_path))
            ) {
                unset($this->css_files[key($css_path)]);
            }
        }
    }

    /**
     * Adds a new JavaScript file(s) to the page header.
     *
     * @param string|array $js_uri Path to JS file or an array like: array(uri, ...)
     * @param bool $check_path
     */
    public function addJS($js_uri, $check_path = true)
    {
        if (!is_array($js_uri)) {
            $js_uri = array($js_uri);
        }

        foreach ($js_uri as $js_file) {
            $js_file = explode('?', $js_file);
            $version = '';
            if (isset($js_file[1]) && $js_file[1]) {
                $version = $js_file[1];
            }
            $js_path = $js_file = $js_file[0];
            if ($check_path) {
                $js_path = Media::getJSPath($js_file);
            }

            if ($js_path && !in_array($js_path, $this->js_files)) {
                $this->js_files[] = $js_path . ($version ? '?' . $version : '');
            }
        }
    }

    /**
     * Removes JS file(s) from the queued JS file list.
     *
     * @param string|array $js_uri Path to JS file or an array like: array(uri, ...)
     * @param bool $check_path
     */
    public function removeJS($js_uri, $check_path = true)
    {
        if (!is_array($js_uri)) {
            $js_uri = array($js_uri);
        }

        foreach ($js_uri as $js_file) {
            if ($check_path) {
                $js_file = Media::getJSPath($js_file);
            }

            if ($js_file && in_array($js_file, $this->js_files)) {
                unset($this->js_files[array_search($js_file, $this->js_files)]);
            }
        }
    }

    /**
     * @return mixed
     */
    public function getJavascript()
    {
        $jsFileList = $this->javascriptManager->getList();

        return $jsFileList;
    }

    /**
     * @return mixed
     */
    public function getStylesheets()
    {
        $cssFileList = $this->stylesheetManager->getList();

        return $cssFileList;
    }

    /**
     * Sets default media list for this controller.
     */
    public function setMedia()
    { 
        $user = $this->app->user;

        $this->registerStylesheet('theme', '/content/themes/' . PROX_ACTIVE_THEME . '/css/theme.css', ['position' => 'head', 'media' => 'all' ]);

        $this->registerJavascript('theme', '/content/themes/' . PROX_ACTIVE_THEME . '/js/theme.js', ['position' => 'footer', 'media' => 'all' ]);
        
        // Execute Hook FrontController SetMedia
        Hook::exec('actionControllerSetMedia');
        return true;
    }

    public function assignVariables() {
        $smarty = $this->app->smarty;
        $user = $this->app->user;
        $cookie = $this->app->cookie;

        if($user->is_writer || $user->is_editor) {
            $user_outstanding_balance = $user->getOutstanding();
            
            $smarty->assign('user_outstanding_balance', formatPrice($user_outstanding_balance));

            $user_average_rating = $user->getWriterAverageRating();
            $smarty->assign('user_average_rating', $user_average_rating);
        }

        $templateVars = $this->templateVars;

        $modulesVariables = Hook::exec('actionControllerSetVariables', [], null, true);

        if (is_array($modulesVariables)) {
            foreach ($modulesVariables as $moduleName => $variables) {
                $templateVars['modules'][$moduleName] = $variables;
            }
        }

        $proxim = array(
            'time' => time(),
            'site_title' => Configuration::get('SITE_NAME'),
            'base_uri' => $this->app->base_uri,
            'api_path' => $this->app->base_uri . '/api',
            'static_path' => $this->app->base_uri . '/static',
            'theme_uri' => $this->app->theme_uri,
            'uploads_path' => Configuration::get('UPLOADS_PATH'),
            'active_theme' => PROX_ACTIVE_THEME,
            'secret' => Tools::getToken(true),
            'notifications_sound' => $user->notification_sound,
            'api' => array()
        );

        $proxim = array_merge($proxim, $templateVars);

        Media::addJsDef(array(
            'proxim' => $proxim,
        ));

        $smarty->assign($proxim);
    }

    /**
     * Compiles and outputs full page content.
     *
     * @return bool
     *
     * @throws Exception
     * @throws SmartyException
     */
    public function display()
    {
        $app = $this->app;
        $smarty = $app->smarty;
        $this->init();

        $html = '';
        $template = $this->template;
        if (is_array($template)) {
            foreach ($template as $tpl) {
                $html .= $smarty->fetch($tpl . '.tpl');
            }
        } else {
            $html = $smarty->fetch($template . '.tpl');
        }

        Hook::exec('actionOutputHTMLBefore', array('html' => &$html));

        echo trim($html);
        exit;
    }
}
