<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\ObjectModel;

/**
 * Class Currency.
 */
class Currency extends ObjectModel
{
    /** @var $id Currency ID */
    public $id;
    
    /** @var string title */
    public $title;
    
    /** @var string iso_code */
    public $iso_code;
    
    /** @var string symbol */
    public $symbol;
    
    /** @var string country_code */
    public $country_code;

    /** @var float conversion_rate */
    public $conversion_rate;

    /** @var bool active */
    public $active = 1;

    /** @var string last_checked */
    public $last_checked;

    /** @var string date_upd */
    public $date_upd;

    /** @var string date_add */
    public $date_add;
    
    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'currency',
        'primary' => 'currency_id',
        'fields' => array(
            'title' => array('type' => self::TYPE_STRING, 'validate' => 'isGenericName', 'required' => true, 'size' => 64),
            'iso_code' => array('type' => self::TYPE_STRING, 'validate' => 'isGenericName', 'size' => 5, 'required' => true),
            'symbol' => array('type' => self::TYPE_STRING, 'validate' => 'isGenericName', 'size' => 5, 'required' => true),
            'country_code' => array('type' => self::TYPE_STRING, 'validate' => 'isGenericName', 'size' => 5),
            'conversion_rate' => array('type' => self::TYPE_FLOAT, 'validate' => 'isUnsignedFloat'),
            'active' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'last_checked' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
        ),
    );

    /** @var array Currency cache */
    protected static $currencies = array();
    protected static $countActiveCurrencies = 0;

    public function __construct($id = null)
    {
        parent::__construct($id);

        if (!$this->conversion_rate) {
            $this->conversion_rate = 1;
        }
    }

    /**
     * Overriding check if currency rate is not empty and if currency with the same iso code already exists.
     * If it's true, currency is not added.
     *
     * @param bool $autoDate Automatically set `date_upd` and `date_add` columns
     * @param bool $nullValues Whether we want to use NULL values instead of empty quotes values
     *
     * @return bool Indicates whether the Currency has been successfully added
     */
    public function add($autoDate = true, $nullValues = false)
    {
        if ((float) $this->conversion_rate <= 0) {
            return false;
        }

        return Currency::exists($this->iso_code) ? false : parent::add($autoDate, $nullValues);
    }

    /**
     * Updates the current object in the database.
     *
     * @param bool $nullValues Whether we want to use NULL values instead of empty quotes values
     *
     * @return bool Indicates whether successfully updated
     *
     * @throws ProximDatabaseException
     * @throws ProximException
     */
    public function update($nullValues = false)
    {
        if ((float) $this->conversion_rate <= 0) {
            return false;
        }

        return parent::update($nullValues);
    }
    
    /**
     * Check if a Currency already exists.
     *
     * @param int|string $isoCode int for iso code number string for iso code
     *
     * @return bool Indicates whether the Currency already exists
     */
    public static function exists($isoCode)
    {
        $idCurrencyExists = Currency::getIdByIsoCode($isoCode);

        if ($idCurrencyExists) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Return available currency.
     *
     * @param bool $active
     *
     * @return array Currencies
     */
    public static function getCurrencies( $active = true )
    {
        $sql = new DbQuery();
        $sql->select('c.*');
        $sql->from('currency', 'c');
        if ($active) {
            $sql->where('c.active = 1');
        }
        
        $result = Db::getInstance()->executeS($sql);

        return $result;
    }

    /**
     * Get Currency.
     *
     * @param int $idCurrency Currency ID
     *
     * @return array|bool|null|object
     */
    public static function getCurrency($idCurrency)
    {
        $sql = new DbQuery();
        $sql->select('c.*');
        $sql->from('currency', 'c');
        $sql->where('`currency_id` = ' . (int) $idCurrency);

        return Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
    }

    /**
     * Get Currency ID by ISO code.
     *
     * @param string $isoCode ISO code
     *
     * @return int Currency ID
     */
    public static function getIdByIsoCode($isoCode)
    {
        $cacheId = 'Currency::getIdByIsoCode_' . pSQL($isoCode);
        if (!Cache::isStored($cacheId)) {
            $query = Currency::getIdByQuery();
            $query->where('iso_code = \'' . pSQL($isoCode) . '\'');

            $result = (int) Db::getInstance(PROX_USE_SQL_SLAVE)->getValue($query->build());
            Cache::store($cacheId, $result);

            return $result;
        }

        return Cache::retrieve($cacheId);
    }

    /**
     * Get Currency ID query.
     *
     * @return DbQuery
     */
    public static function getIdByQuery()
    {
        $query = new DbQuery();
        $query->select('c.currency_id');
        $query->from('currency', 'c');

        return $query;
    }
    
    /**
     * Get default Currency.
     *
     * @return bool|Currency
     */
    public static function getDefaultCurrency()
    {
        $idCurrency = (int) Configuration::get('CURRENCY_DEFAULT');
        if ($idCurrency == 0) {
            return false;
        }

        return new Currency($idCurrency);
    }

    /**
     * Get Currency instance.
     *
     * @param int $id Currency ID
     *
     * @return Currency
     */
    public static function getCurrencyInstance($id)
    {
        if (!isset(self::$currencies[$id])) {
            self::$currencies[(int) ($id)] = new Currency($id);
        }

        return self::$currencies[(int) ($id)];
    }
}