<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;
 
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Tools;
use Proxim\Validate;
use Swift_Attachment;
use Swift_Mailer;
use Swift_Message;
use Swift_SendmailTransport;
use Swift_Signers_DKIMSigner;
use Swift_SmtpTransport;

/**
 * Class Mail
 */
class Mail
{
    /**
     * Mail content type.
     */
    const TYPE_HTML = 1;
    const TYPE_TEXT = 2;
    const TYPE_BOTH = 3;

    /**
     * Send mail under SMTP server.
     */
    const METHOD_SMTP = 2;

    /**
     * Disable mail, will return immediately after calling send method.
     */
    const METHOD_DISABLE = 3;

    /**
     * Send Email.
     *
     * @param string $template Template: the name of template not be a var but a string !
     * @param string $subject Subject of the email
     * @param string $templateVars Template variables for the email
     * @param string $to To email
     * @param string $toName To name
     * @param string $from From email
     * @param string $fromName To email
     * @param array $fileAttachment array with three parameters (content, mime and name).
     *                              You can use an array of array to attach multiple files
     * @param bool $mode_smtp SMTP mode (deprecated)
     * @param string $templatePath Template path
     * @param bool $die Die after error
     * @param string $bcc Bcc recipient address. You can use an array of array to send to multiple recipients
     * @param string $replyTo Reply-To recipient address
     * @param string $replyToName Reply-To recipient name
     *
     * @return bool|int Whether sending was successful. If not at all, false, otherwise amount of recipients succeeded.
     */
    public static function send(
        $template,
        $subject,
        $templateVars,
        $to,
        $toName = null,
        $from = null,
        $fromName = null,
        $fileAttachment = null,
        $mode_smtp = null,
        $templatePath = PROX_DIR_MAIL,
        $die = false,
        $bcc = null,
        $replyTo = null,
        $replyToName = null,
        $siteId = PROX_SITE_ID
    ) {

        $hookBeforeEmailResult = Hook::exec(
            'actionEmailSendBefore',
            [
                'template' => &$template,
                'subject' => &$subject,
                'templateVars' => &$templateVars,
                'to' => &$to,
                'toName' => &$toName,
                'from' => &$from,
                'fromName' => &$fromName,
                'fileAttachment' => &$fileAttachment,
                'mode_smtp' => &$mode_smtp,
                'templatePath' => &$templatePath,
                'die' => &$die,
                'siteId' => &$siteId,
                'bcc' => &$bcc,
                'replyTo' => &$replyTo,
            ],
            null,
            true
        );

        if ($hookBeforeEmailResult === null) {
            $keepGoing = false;
        } else {
            $keepGoing = array_reduce(
                $hookBeforeEmailResult,
                function ($carry, $item) {
                    return ($item === false) ? false : $carry;
                },
                true
            );
        }

        if (!$keepGoing) {
            return true;
        }

        $smarty = Application::getInstance()->smarty;
        
        $configuration = Configuration::getMultiple(
            [
                'SITE_EMAIL',
                'MAIL_METHOD',
                'MAIL_SERVER',
                'MAIL_USER',
                'MAIL_PASSWD',
                'SITE_NAME',
                'MAIL_SMTP_ENCRYPTION',
                'MAIL_SMTP_PORT',
                'MAIL_DKIM_ENABLE',
                'MAIL_DKIM_DOMAIN',
                'MAIL_DKIM_SELECTOR',
                'MAIL_DKIM_KEY'
            ],
            $siteId
        );

        // Hook to alter template vars
        Hook::exec(
            'sendMailAlterTemplateVars',
            [
                'template' => $template,
                'template_vars' => &$templateVars,
            ]
        );

        // Returns immediately if emails are deactivated
        if ($configuration['MAIL_METHOD'] == self::METHOD_DISABLE) {
            return true;
        }

        if (!isset($configuration['MAIL_SMTP_ENCRYPTION']) ||
            Tools::strtolower($configuration['MAIL_SMTP_ENCRYPTION']) === 'off'
        ) {
            $configuration['MAIL_SMTP_ENCRYPTION'] = false;
        }

        if (!isset($configuration['MAIL_SMTP_PORT'])) {
            $configuration['MAIL_SMTP_PORT'] = 'default';
        }

        /*
         * Sending an e-mail can be of vital importance for the merchant, when his password
         * is lost for example, so we must not die but do our best to send the e-mail.
         */
        if (!isset($from) || !Validate::isEmail($from)) {
            $from = $configuration['SITE_EMAIL'];
        }

        if (!Validate::isEmail($from)) {
            $from = null;
        }

        // $from_name is not that important, no need to die if it is not valid
        if (!isset($fromName) || !Validate::isMailName($fromName)) {
            $fromName = $configuration['SITE_NAME'];
        }

        if (!Validate::isMailName($fromName)) {
            $fromName = null;
        }

        /*
         * It would be difficult to send an e-mail if the e-mail is not valid,
         * so this time we can die if there is a problem.
         */
        if (!is_array($to) && !Validate::isEmail($to)) {
            // Tools::dieOrLog('Error: parameter "to" is corrupted');
            return false;
        }

        // if bcc is not null, make sure it's a vaild e-mail
        if (!is_null($bcc) && !is_array($bcc) && !Validate::isEmail($bcc)) {
            // Tools::dieOrLog('Error: parameter "bcc" is corrupted');
            $bcc = null;
        }

        if (!is_array($templateVars)) {
            $templateVars = [];
        }

        // Do not crash for this error, that may be a complicated customer name
        if (is_string($toName) && !empty($toName) && !Validate::isMailName($toName)) {
            $toName = null;
        }

        if (!Validate::isTplName($template)) {
            Tools::dieOrLog('Error: invalid e-mail template');

            return false;
        }

        if (!Validate::isMailSubject($subject)) {
            Tools::dieOrLog('Error: invalid e-mail subject');
            return false;
        }


        $message = new Swift_Message();

        /* Create new message and DKIM sign it, if enabled and all data for signature are provided */
        if ((bool) $configuration['MAIL_DKIM_ENABLE'] === true
            && !empty($configuration['MAIL_DKIM_DOMAIN'])
            && !empty($configuration['MAIL_DKIM_SELECTOR'])
            && !empty($configuration['MAIL_DKIM_KEY'])
        ) {
            $signer = new Swift_Signers_DKIMSigner(
                $configuration['MAIL_DKIM_KEY'],
                $configuration['MAIL_DKIM_DOMAIN'],
                $configuration['MAIL_DKIM_SELECTOR']
            );

            $message->attachSigner($signer);
        }

        /* Construct multiple recipients list if needed */
        if (is_array($to) && isset($to)) {
            foreach ($to as $key => $addr) {
                $addr = trim($addr);
                if (!Validate::isEmail($addr)) {
                    Tools::dieOrLog('Error: invalid e-mail address');
                    return false;
                }

                if (is_array($toName) && isset($toName[$key])) {
                    $addrName = $toName[$key];
                } else {
                    $addrName = $toName;
                }

                $addrName = ($addrName == null || $addrName == $addr || !Validate::isGenericName($addrName)) ?
                          '' :
                          $addrName;
                $message->addTo(self::toPunycode($addr), $addrName);
            }
            $toPlugin = $to[0];
        } else {
            /* Simple recipient, one address */
            $toPlugin = $to;
            $toName = (($toName == null || $toName == $to) ? '' : $toName);
            $message->addTo(self::toPunycode($to), $toName);
        }

        if (isset($bcc) && is_array($bcc)) {
            foreach ($bcc as $addr) {
                $addr = trim($addr);
                if (!Validate::isEmail($addr)) {
                    Tools::dieOrLog('Error: invalid e-mail address');

                    return false;
                }

                $message->addBcc(self::toPunycode($addr));
            }
        } elseif (isset($bcc)) {
            $message->addBcc(self::toPunycode($bcc));
        }

        try {
            /* Connect with the appropriate configuration */
            if ($configuration['MAIL_METHOD'] == self::METHOD_SMTP) {
                if (empty($configuration['MAIL_SERVER']) || empty($configuration['MAIL_SMTP_PORT'])) {
                    Tools::dieOrLog('Error: invalid SMTP server or SMTP port');
                    return false;
                }

                $connection = (new Swift_SmtpTransport(
                    $configuration['MAIL_SERVER'],
                    $configuration['MAIL_SMTP_PORT'],
                    $configuration['MAIL_SMTP_ENCRYPTION']
                ))
                    ->setUsername($configuration['MAIL_USER'])
                    ->setPassword($configuration['MAIL_PASSWD']);
            } else {
                /**
                 * mail() support was removed from SwiftMailer for security reasons
                 * previously => $connection = \Swift_MailTransport::newInstance();
                 * Use Swift_SendmailTransport instead
                 *
                 * @see https://github.com/swiftmailer/swiftmailer/issues/866
                 */

                $connection = new Swift_SendmailTransport();
            }

            if (!$connection) {
                return false;
            }

            $swift = new Swift_Mailer($connection);

            if($templatePath == null) {
                $templatePath = PROX_DIR_MAIL;
            }

            if (!file_exists($templatePath . $template . '.tpl')
            ) {
                Tools::dieOrLog(sprintf('Error - The following e-mail template is missing: %s', $templatePath . $template . '.tpl'));
            } else {
                $templatePathExists = true;
            }

            if (empty($templatePathExists)) {
                Tools::dieOrLog('Error - The following e-mail template is missing: %s', [$template]);
                return false;
            }

            $message->setCharset('utf-8');

            /* Set Message-ID - getmypid() is blocked on some hosting */
            $message->setId(Mail::generateId());

            if (!($replyTo && Validate::isEmail($replyTo))) {
                $replyTo = $from;
            }

            if (isset($replyTo) && $replyTo) {
                $message->setReplyTo($replyTo, ($replyToName !== '' ? $replyToName : null));
            }

            $site_logo = Configuration::get('SITE_EMAIL_LOGO', (int) $siteId);
            $logo = Configuration::get('UPLOADS_PATH') . '/' . $site_logo;
            /* don't attach the logo as */ 
            if (isset($logo)) {
                $templateVars['site_logo'] = $logo;
                // $templateVars['site_logo'] = $message->embed(\Swift_Image::fromPath($logo));
            }
            
            // Get extra template_vars
            $extraTemplateVars = [];

            $templateVars['current_year'] = date('Y');
            $templateVars['subject'] = $subject;
            $templateVars['site_name'] = Tools::safeOutput(Configuration::get('SITE_NAME', (int) $siteId, null));
            $templateVars['site_url'] = Tools::safeOutput(Configuration::get('SITE_DOMAIN', (int) $siteId, null));
            $templateVars['color'] = Tools::safeOutput(Configuration::get('MAIL_COLOR', (int) $siteId, '#3c90df'));

            Hook::exec(
                'actionGetExtraMailTemplateVars',
                [
                    'template' => $template,
                    'template_vars' => $templateVars,
                    'extra_template_vars' => &$extraTemplateVars,
                ],
                null,
                true
            );
           
            $smarty->assign([
                'extra_template_vars' => $extraTemplateVars,
                'templateVars' => $templateVars
            ]);

            $templateVars = array_merge($templateVars, $extraTemplateVars);
            $swiftTemplateVars = array();
            foreach($templateVars as $templateVarKey => $templateVarValue) {
                $swiftTemplateVars['{' . $templateVarKey . '}'] = $templateVarValue;
            }

            $mailTemplateId = MailTemplate::getIdByTemplateName($template);
            $mailTemplate = new MailTemplate( (int) $mailTemplateId );
            if(Validate::isLoadedObject($mailTemplate)) {
                /* Create mail and attach differents parts */
                $message->setSubject( self::messageDecorator($mailTemplate->subject, $swiftTemplateVars) );
                $smarty->assign([
                    'mail_body' => $mailTemplate->message
                ]);
                $templateHtml = $smarty->fetch($templatePath . 'mail_template.tpl');
                $templateHtml = self::messageDecorator($templateHtml, $swiftTemplateVars);
            } else {
                /* Create mail and attach differents parts */
                $message->setSubject($subject);
                $templateHtml = $smarty->fetch($templatePath . $template . '.tpl');
                $templateHtml = self::messageDecorator($templateHtml, $swiftTemplateVars);
            }
        
            $message->addPart($templateHtml, 'text/html', 'utf-8');

            if ($fileAttachment && !empty($fileAttachment)) {
                // Multiple attachments?
                if (!is_array(current($fileAttachment))) {
                    $fileAttachment = array($fileAttachment);
                }
 
                foreach ($fileAttachment as $attachment) {
                    if (isset($attachment['content']) && isset($attachment['name']) && isset($attachment['mime'])) {
                        $message->attach(
                            (new Swift_Attachment())->setFilename(
                                $attachment['name']
                            )->setContentType($attachment['mime'])
                                ->setBody($attachment['content'])
                        );
                    }
                }
            }

            /* Send mail */
            $message->setFrom(array($from => $fromName));

            // Hook to alter Swift Message before sending mail
            $hookAfterEmailResult = Hook::exec(
                'actionMailAlterMessageBeforeSend', 
                [
                    'message' => &$message,
                    'fileAttachment' => &$fileAttachment,
                    'htmlMessage' => &$templateHtml,
                    'siteId' => &$siteId
                ],
                null,
                true
            );

            if ($hookAfterEmailResult === null) {
                $keepGoing = false;
            } else {
                $keepGoing = array_reduce(
                    $hookAfterEmailResult,
                    function ($carry, $item) {
                        return ($item === false) ? false : $carry;
                    },
                    true
                );
            }
    
            if (!$keepGoing) {
                return true;
            }

            $send = $swift->send($message);

            return $send;
        } catch (\Swift_SwiftException $e) {
            // Tools::dieOrLog('Swift Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Send a test email.
     *
     * @param bool $smtpChecked Is SMTP checked?
     * @param string $smtpServer SMTP Server hostname
     * @param string $content Content of the email
     * @param string $subject Subject of the email
     * @param bool $type Deprecated
     * @param string $to To email address
     * @param string $from From email address
     * @param string $smtpLogin SMTP login name
     * @param string $smtpPassword SMTP password
     * @param int $smtpPort SMTP Port
     * @param bool|string $smtpEncryption Encryption type. "off" or false disable encryption.
     *
     * @return bool|string True if succeeded, otherwise the error message
     */
    public static function sendMailTest(
        $smtpChecked,
        $smtpServer,
        $content,
        $subject,
        $type,
        $to,
        $from,
        $smtpLogin,
        $smtpPassword,
        $smtpPort,
        $smtpEncryption,
        bool $dkimEnable = false,
        string $dkimKey = '',
        string $dkimDomain = '',
        string $dkimSelector = ''
    ) {
        $result = false;
        try {
            if ($smtpChecked) {
                if (Tools::strtolower($smtpEncryption) === 'off') {
                    $smtpEncryption = false;
                }
                $connection = (new Swift_SmtpTransport(
                    $smtpServer,
                    $smtpPort,
                    $smtpEncryption
                ))
                    ->setUsername($smtpLogin)
                    ->setPassword($smtpPassword);
            } else {
                /**
                 * mail() support was removed from SwiftMailer for security reasons
                 * previously => $connection = \Swift_MailTransport::newInstance();
                 * Use Swift_SendmailTransport instead
                 *
                 * @see https://github.com/swiftmailer/swiftmailer/issues/866
                 */
                $connection = new Swift_SendmailTransport();
            }

            $swift = new Swift_Mailer($connection);
            $message = new Swift_Message();

            /* Create new message and DKIM sign it, if enabled and all data for signature are provided */
            if ($dkimEnable === true
                && !empty($dkimKey)
                && !empty($dkimDomain)
                && !empty($dkimSelector)
            ) {
                $signer = new Swift_Signers_DKIMSigner(
                    $dkimKey,
                    $dkimDomain,
                    $dkimSelector
                );
                $message->attachSigner($signer);
            }

            $message
                ->setFrom($from)
                ->setTo($to)
                ->setSubject($subject)
                ->setBody($content);

            if ($swift->send($message)) {
                $result = true;
            }
        } catch (\Swift_SwiftException $e) {
            $result = $e->getMessage();
        }

        return $result;
    }
    
    public static function messageDecorator($text, $decorators){
        $text = str_replace(array_keys($decorators), array_values($decorators), $text);
        return $text;
    }
    
    /* Rewrite of Swift_Message::generateId() without getmypid() */
    protected static function generateId($idstring = null)
    {
        $midparams = [
            'utctime' => gmstrftime('%Y%m%d%H%M%S'),
            'randint' => mt_rand(),
            'customstr' => (preg_match('/^(?<!\\.)[a-z0-9\\.]+(?!\\.)$/iD', $idstring) ? $idstring : 'swift'),
            'hostname' => !empty($_SERVER['SERVER_NAME']) ? $_SERVER['SERVER_NAME'] : php_uname('n'),
        ];

        return vsprintf('%s.%d.%s@%s', $midparams);
    }

    /**
     * Check if a multibyte character set is used for the data.
     *
     * @param string $data Data
     *
     * @return bool Whether the string uses a multibyte character set
     */
    public static function isMultibyte($data)
    {
        $length = Tools::strlen($data);
        for ($i = 0; $i < $length; ++$i) {
            if (ord(($data[$i])) > 128) {
                return true;
            }
        }

        return false;
    }

    /**
     * Automatically convert email to Punycode.
     *
     * Try to use INTL_IDNA_VARIANT_UTS46 only if defined, else use INTL_IDNA_VARIANT_2003
     * See https://wiki.php.net/rfc/deprecate-and-remove-intl_idna_variant_2003
     *
     * @param string $to Email address
     *
     * @return string
     */
    public static function toPunycode($to)
    {
        $address = explode('@', $to);
        if (empty($address[0]) || empty($address[1])) {
            return $to;
        }

        if (defined('INTL_IDNA_VARIANT_UTS46')) {
            return $address[0] . '@' . idn_to_ascii($address[1], 0, INTL_IDNA_VARIANT_UTS46);
        }

        /*
         * INTL_IDNA_VARIANT_2003 const will be removed in PHP 8.
         * See https://wiki.php.net/rfc/deprecate-and-remove-intl_idna_variant_2003
         */
        if (defined('INTL_IDNA_VARIANT_2003')) {
            return $address[0] . '@' . idn_to_ascii($address[1], 0, INTL_IDNA_VARIANT_2003);
        }

        return $address[0] . '@' . idn_to_ascii($address[1]);
    }
}