<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Exception;
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\Validate;
use Proxim\Database\EntityInterface;
use Proxim\Adapter\EntityMapper;

/**
 * ObjectModel
 */
class ObjectModel implements EntityInterface {

    /**
     * List of field types.
     */
    const TYPE_INT = 1;
    const TYPE_BOOL = 2;
    const TYPE_STRING = 3;
    const TYPE_FLOAT = 4;
    const TYPE_DATE = 5;
    const TYPE_HTML = 6;
    const TYPE_NOTHING = 7;
    const TYPE_SQL = 8;

    /**
     * List of association types.
     */
    const HAS_ONE = 1;
    const HAS_MANY = 2;

    /** @var int Object ID */
    public $id;

    /** @var array|null Holds required fields for each ObjectModel class */
    protected static $fieldsRequiredDatabase = null;

    /**
     * @deprecated 1.0.0 Define property using $definition['table'] property instead.
     *
     * @var string
     */
    protected $table;

    /**
     * @deprecated 1.0.0 Define property using $definition['table'] property instead.
     *
     * @var string
     */
    protected $identifier;

    /**
     * @deprecated 1.0.0 Define property using $definition['table'] property instead.
     *
     * @var array
     */
    protected $fieldsRequired = array();

    /**
     * @deprecated 1.0.0 Define property using $definition['table'] property instead.
     *
     * @var array
     */
    protected $fieldsSize = array();

    /**
     * @deprecated 1.0.0 Define property using $definition['table'] property instead.
     *
     * @var array
     */
    protected $fieldsValidator = array();

    /**
     * @var array Contains object definition
     *
     * @since 1.0.0
     */
    public static $definition = array();

    /**
     * Holds compiled definitions of each ObjectModel class.
     * Values are assigned during object initialization.
     *
     * @var array
     */
    protected static $loaded_classes = array();

    /** @var array Contains current object definition. */
    protected $def;

    /** @var array|null List of specific fields to update (all fields if null). */
    protected $update_fields = null;

    /** @var Db An instance of the db in order to avoid calling Db::getInstance() thousands of times. */
    protected static $db = false;

    /** @var array|null List of HTML field (based on self::TYPE_HTML) */
    public static $htmlFields = null;

    /** @var bool Enables to define an ID before adding object. */
    public $force_id = false;

    /**
     * @var bool if true, objects are cached in memory
     */
    protected static $cache_objects = true;

    public static function getRepositoryClassName()
    {
        return null;
    }

    /**
     * Returns object validation rules (fields validity).
     *
     * @param string $class Child class name for static use (optional)
     *
     * @return array Validation rules (fields validity)
     */
    public static function getValidationRules($class = __CLASS__)
    {
        $object = new $class();

        return array(
            'required' => $object->fieldsRequired,
            'size' => $object->fieldsSize,
            'validate' => $object->fieldsValidate
        );
    }

    /**
     * Builds the object.
     *
     * @param int|null $id if specified, loads and existing object from DB (optional)
     *
     * @throws Exception
     * @throws Exception
     */
    public function __construct($id = null) {
        $class_name = get_class($this);

        if (!isset(ObjectModel::$loaded_classes[$class_name])) {
            $this->def = ObjectModel::getDefinition($class_name);
            $this->setDefinitionRetrocompatibility();

            if (!Validate::isTableOrIdentifier($this->def['primary']) || !Validate::isTableOrIdentifier($this->def['table'])) {
                throw new Exception('Identifier or table format not valid for class ' . $class_name);
            }

            ObjectModel::$loaded_classes[$class_name] = get_object_vars($this);
        } else {
            foreach (ObjectModel::$loaded_classes[$class_name] as $key => $value) {
                $this->{$key} = $value;
            }
        }

        if ($id) {
            $entity_mapper = new EntityMapper();
            $entity_mapper->load($id, $this, $this->def, self::$cache_objects);
        }

    }

    /**
     * Prepare fields for ObjectModel class (add, update)
     * All fields are verified (pSQL, intval, ...).
     *
     * @return array All object fields
     *
     * @throws Exception
     */
    public function getFields()
    {
        $this->validateFields();
        $fields = $this->formatFields();

        // Ensure that we get something to insert
        if (!$fields && isset($this->id)) {
            $fields[$this->def['primary']] = $this->id;
        }

        return $fields;
    }

    /**
     * Formats values of each fields.
     *
     * @since 1.0.0
     *
     * @return array
     */
    protected function formatFields()
    {
        $fields = array();

        // Set primary key in fields
        if (isset($this->id)) {
            $fields[$this->def['primary']] = $this->id;
        }

        foreach ($this->def['fields'] as $field => $data) {
            // Only get fields we need for the type
            $value = $this->$field;

            $purify = (isset($data['validate']) && Tools::strtolower($data['validate']) == 'iscleanhtml') ? true : false;
            // Format field value
            $fields[$field] = ObjectModel::formatValue($value, $data['type'], false, $purify, !empty($data['allow_null']));
        }

        return $fields;
    }

    /**
     * Formats a value.
     *
     * @param mixed $value
     * @param int $type
     * @param bool $with_quotes
     * @param bool $purify
     * @param bool $allow_null
     *
     * @return mixed
     */
    public static function formatValue($value, $type, $with_quotes = false, $purify = true, $allow_null = false)
    {
        if ($allow_null && $value === null) {
            return array('type' => 'sql', 'value' => 'NULL');
        }

        switch ($type) {
            case self::TYPE_INT:
                return (int) $value;

            case self::TYPE_BOOL:
                return (int) $value;

            case self::TYPE_FLOAT:
                return (float) str_replace(',', '.', $value);

            case self::TYPE_DATE:
                if (!$value) {
                    $value = '0000-00-00';
                }

                if ($with_quotes) {
                    return '\'' . pSQL($value) . '\'';
                }

                return pSQL($value);

            case self::TYPE_HTML:
                if ($purify) {
                    $value = Tools::purifyHTML($value);
                }
                if ($with_quotes) {
                    return '\'' . pSQL($value, true) . '\'';
                }

                return pSQL($value, true);

            case self::TYPE_SQL:
                if ($with_quotes) {
                    return '\'' . pSQL($value, true) . '\'';
                }

                return pSQL($value, true);

            case self::TYPE_NOTHING:
                return $value;

            case self::TYPE_STRING:
            default:
                if ($with_quotes) {
                    return '\'' . pSQL($value) . '\'';
                }

                return pSQL($value);
        }
    }

    private function getFullyQualifiedName()
    {
        return str_replace('\\', '', get_class($this));
    }

    /**
     * Get object name
     * Used for read/write in required fields table.
     *
     * @return string
     */
    public function getObjectName()
    {
        return get_class($this);
    }

    /**
     * Saves current object to database (add or update).
     *
     * @param bool $null_values
     * @param bool $auto_date
     *
     * @return bool Insertion result
     *
     * @throws Exception
     */
    public function save($null_values = false, $auto_date = true)
    {
        return $this->id ? $this->update($null_values) : $this->add($auto_date, $null_values);
    }

    /**
     * Adds current object to the database.
     *
     * @param bool $auto_date
     * @param bool $null_values
     *
     * @return bool Insertion result
     *
     * @throws Exception
     * @throws Exception
     */
    public function add($auto_date = true, $null_values = false)
    {
        if (isset($this->id) && !$this->force_id) {
            unset($this->id);
        }

        // @hook actionObject*AddBefore
        Hook::exec('actionObjectAddBefore', ['object' => $this]);
        Hook::exec('actionObject' . $this->getFullyQualifiedName() . 'AddBefore', ['object' => $this]);

        // Automatically fill dates
        if ($auto_date && property_exists($this, 'date_add')) {
            $this->date_add = date('Y-m-d H:i:s');
        }
        if ($auto_date && property_exists($this, 'date_upd')) {
            $this->date_upd = date('Y-m-d H:i:s');
        }

        if (!$result = Db::getInstance()->insert($this->def['table'], $this->getFields(), $null_values)) {
            return false;
        }

        // Get object id in database
        if ( !$this->force_id) {
            $this->id = Db::getInstance()->Insert_ID();
        }
        
        if (!$result) {
            return false;
        }

        // @hook actionObject*AddAfter
        Hook::exec('actionObjectAddAfter', ['object' => $this]);
        Hook::exec('actionObject' . $this->getFullyQualifiedName() . 'AddAfter', ['object' => $this]);

        return $result;
    }

    /**
     * Takes current object ID, gets its values from database,
     * saves them in a new row and loads newly saved values as a new object.
     *
     * @return ObjectModel|false
     *
     * @throws Exception
     */
    public function duplicateObject()
    {
        $definition = ObjectModel::getDefinition($this);

        $res = Db::getInstance()->getRow('
					SELECT *
					FROM `' . DB_PREFIX . bqSQL($definition['table']) . '`
					WHERE `' . bqSQL($definition['primary']) . '` = ' . $this->id
                );
        if (!$res) {
            return false;
        }

        unset($res[$definition['primary']]);
        foreach ($res as $field => &$value) {
            if (isset($definition['fields'][$field])) {
                $value = ObjectModel::formatValue($value, $definition['fields'][$field]['type'], false, true,
                                                  !empty($definition['fields'][$field]['allow_null']));
            }
        }

        if (!Db::getInstance()->insert($definition['table'], $res)) {
            return false;
        }

        $object_id = Db::getInstance()->Insert_ID();

        /** @var ObjectModel $object_duplicated */
        $object_duplicated = new $definition['classname']($object_id);

        return $object_duplicated;
    }

    /**
     * Updates the current object in the database.
     *
     * @param bool $null_values
     *
     * @return bool
     *
     * @throws Exception
     * @throws Exception
     */
    public function update($null_values = false)
    {
        // @hook actionObject*UpdateBefore
        Hook::exec('actionObjectUpdateBefore', array('object' => $this));
        Hook::exec('actionObject' . $this->getFullyQualifiedName() . 'UpdateBefore', array('object' => $this));

        $this->clearCache();

        // Automatically fill dates
        if (property_exists($this, 'date_upd')) {
            $this->date_upd = date('Y-m-d H:i:s');
            if (isset($this->update_fields) && is_array($this->update_fields) && count($this->update_fields)) {
                $this->update_fields['date_upd'] = true;
            }
        }

        // Automatically fill dates
        if (property_exists($this, 'date_add') && $this->date_add == null) {
            $this->date_add = date('Y-m-d H:i:s');
            if (isset($this->update_fields) && is_array($this->update_fields) && count($this->update_fields)) {
                $this->update_fields['date_add'] = true;
            }
        }

        // Database update
        if (!$result = Db::getInstance()->update($this->def['table'], $this->getFields(), '`' . pSQL($this->def['primary']) . '` = "' . pSQL($this->id) . '"', 0, $null_values)) {
            return false;
        }

        // @hook actionObject*UpdateAfter
        Hook::exec('actionObjectUpdateAfter', array('object' => $this));
        Hook::exec('actionObject' . $this->getFullyQualifiedName() . 'UpdateAfter', array('object' => $this));

        return $result;
    }

    /**
     * Deletes current object from database.
     *
     * @return bool True if delete was successful
     *
     * @throws Exception
     */
    public function delete()
    {
        // @hook actionObject*DeleteBefore
        Hook::exec('actionObjectDeleteBefore', ['object' => $this]);
        Hook::exec('actionObject' . $this->getFullyQualifiedName() . 'DeleteBefore', ['object' => $this]);

        $this->clearCache();
        
        $result = Db::getInstance()->delete($this->def['table'], '`' . bqSQL($this->def['primary']) . '` = "' . $this->id . '"');

        if (!$result) {
            return false;
        }

        // @hook actionObject*DeleteAfter
        Hook::exec('actionObjectDeleteAfter', ['object' => $this]);
        Hook::exec('actionObject' . $this->getFullyQualifiedName() . 'DeleteAfter', ['object' => $this]);

        return $result;
    }

    /**
     * Deletes multiple objects from the database at once.
     *
     * @param array $ids array of objects IDs
     *
     * @return bool
     */
    public function deleteSelection($ids)
    {
        $result = true;
        foreach ($ids as $id) {
            $this->id = $id;
            $result = $result && $this->delete();
        }

        return $result;
    }

    /**
     * Toggles object status in database.
     *
     * @return bool Update result
     *
     * @throws Exception
     */
    public function toggleStatus()
    {
        // Object must have a variable called 'active'
        if (!property_exists($this, 'active')) {
            throw new Exception('property "active" is missing in object ' . get_class($this));
        }

        // Update only active field
        $this->setFieldsToUpdate(array('active' => true));

        // Update active status on object
        $this->active = !(int) $this->active;

        // Change status to active/inactive
        return $this->update(false);
    }

    /**
     * Validate a single field.
     *
     * @since 1.0.0
     *
     * @param string $field Field name
     * @param mixed $value Field value
     * @param array $skip array of fields to skip
     * @param bool $human_errors if true, uses more descriptive error strings
     *
     * @return true|string true or error message string
     *
     * @throws Exception
     */
    public function validateField($field, $value, $skip = array(), $human_errors = false)
    {
        static $prox_allow_html_iframe = null;

        $data = $this->def['fields'][$field];

        // Default value
        if (!$value && !empty($data['default'])) {
            $value = $data['default'];
            $this->$field = $value;
        }

        // Check field values
        if (!in_array('values', $skip) && !empty($data['values']) && is_array($data['values']) && !in_array($value, $data['values'])) {
            return sprintf('Property %1$s has a bad value (allowed values are: %2$s).', array(get_class($this) . '->' . $field, implode(', ', $data['values'])), 'Admin.Notifications.Error');
        }

        // Check field size
        if (!in_array('size', $skip) && !empty($data['size'])) {
            $size = $data['size'];
            if (!is_array($data['size'])) {
                $size = array('min' => 0, 'max' => $data['size']);
            }

            $length = Tools::strlen($value);
            if ($length < $size['min'] || $length > $size['max']) {
                if ($human_errors) {
                    return sprintf('The %1$s field is too long (%2$d chars max).', $this->displayFieldName($field, get_class($this), $size['max']));
                } else {
                    return sprintf('The length of property %1$s is currently %2$d chars. It must be between %3$d and %4$d chars.', 
                        get_class($this) . '->' . $field,
                        $length,
                        $size['min'],
                        $size['max']
                    );
                }
            }
        }

        // Check field validator
        if (!in_array('validate', $skip) && !empty($data['validate'])) {
            if (!method_exists('Proxim\Validate', $data['validate'])) {
                throw new Exception(
                    sprintf('Validation function not found: %s.', $data['validate'])
                );
            }

            if (!empty($value)) {
                $res = true;
                if (Tools::strtolower($data['validate']) == 'iscleanhtml') {
                    if (!call_user_func(array('Proxim\Validate', $data['validate']), $value, $prox_allow_html_iframe)) {
                        $res = false;
                    }
                } else {
                    if (!call_user_func(array('Proxim\Validate', $data['validate']), $value)) {
                        $res = false;
                    }
                }
                if (!$res) {
                    if ($human_errors) {
                        return sprintf('The %s field is invalid.', $this->displayFieldName($field, get_class($this)));
                    } else {
                        return sprintf('Property %s is not valid', get_class($this) . '->' . $field);
                    }
                }
            }
        }

        return true;
    }

    /**
     * Returns field name translation.
     *
     * @param string $field Field name
     * @param string $class ObjectModel class name
     * @param bool $htmlentities If true, applies htmlentities() to result string
     *
     * @return string
     */
    public static function displayFieldName($field, $class = __CLASS__, $htmlentities = true)
    {
        global $_FIELDS;

        $key = $class . '_' . md5($field);

        return (is_array($_FIELDS) && isset( $_FIELDS[$key])) ? ($htmlentities ? htmlentities($_FIELDS[$key], ENT_QUOTES, 'utf-8') : $_FIELDS[$key]) : $field;
    }

    /**
     * Checks if object field values are valid before database interaction.
     *
     * @param bool $die
     * @param bool $error_return
     *
     * @return bool|string true, false or error message
     *
     * @throws Exception
     */
    public function validateFields($die = true, $error_return = false)
    {   
        foreach ($this->def['fields'] as $field => $data) {
            if ( is_array($this->update_fields) && empty($this->update_fields[$field]) && isset($this->def['fields'][$field]) && $this->def['fields'][$field]) {
                continue;
            }

            $message = $this->validateField($field, $this->$field);
            if ($message !== true) {
                if ($die) {
                    throw new Exception($message);
                }

                return $error_return ? $message : false;
            }
        }

        return true;
    }

    /**
     * Validate required fields.
     *
     * @param bool $htmlentities
     *
     * @return array
     *
     * @throws Exception
     */
    public function validateFieldsRequiredDatabase($htmlentities = true)
    {
        $this->cacheFieldsRequiredDatabase();
        $errors = array();
        $required_fields = $this->getCachedFieldsRequiredDatabase();

        foreach ($this->def['fields'] as $field => $data) {
            if (!in_array($field, $required_fields)) {
                continue;
            }

            if (!method_exists('Proxim\Validate', $data['validate'])) {
                throw new Exception('Validation function not found. ' . $data['validate']);
            }

            $value = Tools::getValue($field);

            if (empty($value)) {
                $errors[$field] = sprintf('The field %s is required.', array(self::displayFieldName($field, get_class($this), $htmlentities)), 'Admin.Notifications.Error');
            }
        }

        return $errors;
    }

    /**
     * Returns an array of required fields.
     *
     * @param bool $all if true, returns required fields of all object classes
     *
     * @return array|null
     *
     * @throws Exception
     */
    public function getFieldsRequiredDatabase($all = false)
    {
        return Db::getInstance()->executeS('
		SELECT id_required_field, object_name, field_name
		FROM ' . DB_PREFIX . 'required_field
		' . (!$all ? 'WHERE object_name = \'' . pSQL($this->getObjectName()) . '\'' : ''));
    }

    /**
     * Returns true if required field exists.
     *
     * @param string $field_name to search
     * @param bool $all if true, returns required fields of all object classes
     *
     * @return bool
     */
    public function isFieldRequired($field_name, $all = false)
    {
        if (empty($field_name)) {
            return false;
        } else {
            return (bool) Db::getInstance()->getValue('
            SELECT id_required_field
            FROM ' . DB_PREFIX . 'required_field
            WHERE field_name = "' . Db::getInstance()->escape($field_name) . '"
            ' . (!$all ? ' AND object_name = \'' . pSQL($this->getObjectName()) . '\'' : ''));
        }
    }

    /**
     * Caches data about required objects fields in memory.
     *
     * @param bool $all if true, caches required fields of all object classes
     */
    public function cacheFieldsRequiredDatabase($all = true)
    {
        if (!is_array(self::$fieldsRequiredDatabase)) {
            $fields = $this->getFieldsRequiredDatabase((bool) $all);
            if ($fields) {
                foreach ($fields as $row) {
                    self::$fieldsRequiredDatabase[$row['object_name']][(int) $row['id_required_field']] = pSQL($row['field_name']);
                }
            } else {
                self::$fieldsRequiredDatabase = array();
            }
        }
    }

    /**
     * Get required fields list for this model or for all the models.
     *
     * @param bool $all : whether it should return required fields for this model or all the models
     *
     * @return array
     */
    public function getCachedFieldsRequiredDatabase($all = false)
    {
        $this->cacheFieldsRequiredDatabase($all);

        if ($all) {
            return self::$fieldsRequiredDatabase;
        }

        $objectName = $this->getObjectName();

        return !empty(self::$fieldsRequiredDatabase[$objectName])
            ? self::$fieldsRequiredDatabase[$objectName]
            : array();
    }

    /**
     * Sets required field for this class in the database.
     *
     * @param array $fields
     *
     * @return bool
     *
     * @throws Exception
     */
    public function addFieldsRequiredDatabase($fields)
    {
        if (!is_array($fields)) {
            return false;
        }

        $objectName = $this->getObjectName();
        if (!Db::getInstance()->execute(
            'DELETE FROM ' . DB_PREFIX . 'required_field'
            . " WHERE object_name = '" . Db::getInstance()->escape($objectName) . "'")
        ) {
            return false;
        }

        foreach ($fields as $field) {
            if (!Db::getInstance()->insert(
                'required_field',
                array('object_name' => $objectName, 'field_name' => pSQL($field))
            )) {
                return false;
            }
        }

        return true;
    }

    /**
     * Clears cache entries that have this object's ID.
     *
     * @param bool $all If true, clears cache for all objects
     */
    public function clearCache($all = false)
    {
        if ($all) {
            Cache::clean('objectmodel_' . $this->def['classname'] . '_*');
        } elseif ($this->id) {
            Cache::clean('objectmodel_' . $this->def['classname'] . '_' . $this->id . '_*');
        }
    }

    /**
     * Checks if an object exists in database.
     *
     * @param int $id_entity
     * @param string $table
     *
     * @return bool
     */
    public static function existsInDatabase($id_entity, $table)
    {
        $row = Db::getInstance()->getRow('
			SELECT `id_' . bqSQL($table) . '` as id
			FROM `' . DB_PREFIX . bqSQL($table) . '` e
			WHERE e.`id_' . bqSQL($table) . '` = ' . (int) $id_entity, false
        );

        return isset($row['id']);
    }

    /**
     * Checks if an object type exists in the database.
     *
     * @since 1.0.0
     *
     * @param string|null $table Name of table linked to entity
     * @param bool $has_active_column True if the table has an active column
     *
     * @return bool
     */
    public static function isCurrentlyUsed($table = null, $has_active_column = false)
    {
        if ($table === null) {
            $table = self::$definition['table'];
        }

        $query = new DbQuery();
        $query->select('`id_' . bqSQL($table) . '`');
        $query->from($table);
        if ($has_active_column) {
            $query->where('`active` = 1');
        }

        return (bool) Db::getInstance()->getValue($query);
    }

    /**
     * Fill an object with given data. Data must be an array with this syntax:
     * array(objProperty => value, objProperty2 => value, etc.).
     *
     * @since 1.0.0
     *
     * @param array $data
     */
    public function hydrate(array $data)
    {
        if (isset($data[$this->def['primary']])) {
            $this->id = $data[$this->def['primary']];
        }

        foreach ($data as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Fill (hydrate) a list of objects in order to get a collection of these objects.
     *
     * @since 1.0.0
     *
     * @param string $class Class of objects to hydrate
     * @param array $datas List of data (multi-dimensional array)
     *
     * @return array
     *
     * @throws Exception
     */
    public static function hydrateCollection( $class, array $datas )
    {
        if (!class_exists($class)) {
            throw new Exception("Class '$class' not found");
        }

        $collection = array();
        $rows = array();
        if ($datas) {
            $definition = ObjectModel::getDefinition($class);
            if (!isset( $datas[0][$definition['primary']])) {
                throw new Exception("Identifier '{$definition['primary']}' not found for class '$class'");
            }

            foreach ($datas as $row) {
                // Get object common properties
                $id = $row[$definition['primary']];
                if (!isset($rows[$id])) {
                    $rows[$id] = $row;
                }
            }
        }

        // Hydrate objects
        foreach ($rows as $row) {
            /** @var ObjectModel $obj */
            $obj = new $class();
            $obj->hydrate($row);
            $collection[] = $obj;
        }

        return $collection;
    }
    
    /**
     * Returns object definition.
     *
     * @param string $class Name of object
     * @param string|null $field Name of field if we want the definition of one field only
     *
     * @return array
     */
    public static function getDefinition($class, $field = null)
    {
        if (is_object($class)) {
            $class = get_class($class);
        }

        if ($field === null) {
            $cache_id = 'objectmodel_def_' . $class;
        }

        if ($field !== null || !Cache::isStored($cache_id)) {
            $definition = $class::$definition;

            $definition['classname'] = $class;

            if ($field) {
                return isset($definition['fields'][$field]) ? $definition['fields'][$field] : null;
            }

            Cache::store($cache_id, $definition);

            return $definition;
        }

        return Cache::retrieve($cache_id);
    }

    /**
     * Retrocompatibility for classes without $definition static.
     *
     * @TODO Remove this in 1.6 !
     *
     * @since 1.0.0
     */
    protected function setDefinitionRetrocompatibility()
    {
        // Retrocompatibility with $table property ($definition['table'])
        if (isset($this->def['table'])) {
            $this->table = $this->def['table'];
        } else {
            $this->def['table'] = $this->table;
        }

        // Retrocompatibility with $identifier property ($definition['primary'])
        if (isset($this->def['primary'])) {
            $this->identifier = $this->def['primary'];
        } else {
            $this->def['primary'] = $this->identifier;
        }

        // Retrocompatibility with $identifier property ($definition['primary'])
        if (isset($this->def['primary'])) {
            $this->identifier = $this->def['primary'];
        } else {
            $this->def['primary'] = $this->identifier;
        }

        // Retrocompatibility with $fieldsValidator, $fieldsRequired and $fieldsSize properties ($definition['fields'])
        if (isset($this->def['fields'])) {
            foreach ($this->def['fields'] as $field => $data) {
                if (isset($data['validate'])) {
                    $this->{'fieldsValidator'}[$field] = $data['validate'];
                }
                if (isset($data['required']) && $data['required']) {
                    $this->{'fieldsRequired'}[] = $field;
                }
                if (isset($data['size'])) {
                    $this->{'fieldsSize'}[$field] = $data['size'];
                }
            }
        } else {
            $this->def['fields'] = array();
     
            foreach ($this->{'fieldsValidator'} as $field => $validate) {
                $this->def['fields'][$field]['validate'] = $validate;
            }
            foreach ($this->{'fieldsRequired'} as $field) {
                $this->def['fields'][$field]['required'] = true;
            }
            foreach ($this->{'fieldsSize'} as $field => $size) {
                $this->def['fields'][$field]['size'] = $size;
            }
        }
    }

    /**
     * Set a list of specific fields to update
     * array(field1 => true, field2 => false,
     * langfield1 => array(1 => true, 2 => false)).
     *
     * @since 1.0.0
     *
     * @param array $fields
     */
    public function setFieldsToUpdate(array $fields)
    {
        $this->update_fields = $fields;
    }

    /**
     * Enables object caching.
     */
    public static function enableCache()
    {
        ObjectModel::$cache_objects = true;
    }

    /**
     * Disables object caching.
     */
    public static function disableCache()
    {
        ObjectModel::$cache_objects = false;
    }

    /**
     * Return HtmlFields for object.
     */
    public function getHtmlFields()
    {
        $isDefinitionValid = !empty($this->def) && is_array($this->def) && isset($this->def['table']);
        if (!$isDefinitionValid) {
            return false;
        }

        if (isset(self::$htmlFields[$this->def['table']])) {
            return self::$htmlFields[$this->def['table']];
        }

        self::$htmlFields[$this->def['table']] = array();

        if (isset($this->def['fields'])) {
            foreach ($this->def['fields'] as $name => $field) {
                if (is_array($field) && isset($field['type']) && self::TYPE_HTML === $field['type']) {
                    self::$htmlFields[$this->def['table']][] = $name;
                }
            }
        }

        return self::$htmlFields[$this->def['table']];
    }
}