<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Order;

use Db;
use Proxim\Configuration;
use Proxim\ObjectModel;
use Proxim\Util\StringUtils;
use Aws\S3\Exception\S3Exception;
use Aws\S3\S3Client;

/**
 * File
 */
class File extends ObjectModel
{
	/** The default db first file id */
	const START_FILE_ID = 23455;

	/** File types */
	const LECTURE_NOTES = 59;

	const GRADIC_RUBIC = 57;

	const SOURCES = 55;

	const CHAPTERS = 53;

	const QUESTIONS = 51;

	const OTHER = 39;

	const PROPOSAL = 31;

	const EBOOK = 27;

	const ARTICLE_TO_USE = 25;

	const SAMPLE = 23;

	const DRAFT = 21;

	const OUTLINE = 19;

	const WRITING_GUILDLINES = 17;

	const ORDER_INSTRUCTIONS = 15;

	const SELECTED_FILE_FOR_PAID_REVISION = 37;

	const UNSELECTED_FILE_FOR_PAID_REVISION = 43;

	const BASE_ORDER_FILE_FOR_PAID_REVISION = 61;

	const COMPLETE_PAPER = 63;

	const COMPLETED_SMART_PAPER_FILE = 65;
	
	const CONVERTED_SMART_PAPER_FILE = 67;

	const SCREENSHOT = 68;

	const FINAL_PAPER = 69;

	/** @var $id File ID */
	public $id;

	/** @var int order_id */
	public $order_id;

	/** @var int site_id */
	public $site_id;

	/** @var int uploader_id */
	public $uploader_id;

	/** @var int file_type_id */
	public $file_type_id;
	
	/** @var string file_description */
	public $file_description;
	
	/** @var string name */
	public $name;
	
	/** @var int size */
	public $size = 0;
	
	/** @var string source */
	public $source;
	
	/** @var string extension */
	public $extension;
	
	/** @var bool is_part */
	public $is_part = 0;

	/** @var bool downloaded */
	public $approved = 0;

	/** @var string approved_at */
	public $approved_at;
	
	/** @var bool is_final_file */
	public $is_final_file = 0;
	
	/** @var bool downloaded */
	public $downloaded = 0;
	
	/** @var string downloaded_at */
	public $downloaded_at;
	
	/** @var string date_upd */
	public $date_upd;
	
	/** @var string date_add */
	public $date_add;

	/**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'order_file',
        'primary' => 'file_id',
        'fields' => array(
			'order_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt', 'required' => true),
			'site_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
			'uploader_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt', 'required' => true),
			'file_type_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt', 'required' => true),
			'file_description' => array('type' => self::TYPE_STRING, 'size' => 255),
			'name' => array('type' => self::TYPE_STRING),
			'size' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
			'source' => array('type' => self::TYPE_STRING),
			'extension' => array('type' => self::TYPE_STRING),
			'is_part' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
			'is_final_file' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
			'approved' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
			'approved_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
			'downloaded' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
			'downloaded_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
			'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
			'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
		)
	);

	/**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null)
    {
		parent::__construct($id);
		
		if($this->id) {
			if(!$this->site_id) {
				$this->site_id = Configuration::get('SITE_ID');
				$this->update();
			}
		}
	}

	public function add($autodate = true, $null_values = true) {
		$newFileId = Db::getInstance()->getValue('SELECT MAX(`file_id`)+2 FROM `' . DB_PREFIX . 'order_file`');
		if ($newFileId) {
			$this->id = $newFileId;
		} else {
			$this->id = self::START_FILE_ID;
		}

		$this->site_id = (int) Configuration::get('SITE_ID');
		$this->force_id = true;

		return parent::add($autodate, $null_values);
	}

	public function attach() {
		$file_path = PROX_DIR_UPLOADS . $this->source;
        
		$mime_types = array(
			"htm" => "text/html",
			"exe" => "application/octet-stream",
			"zip" => "application/zip",
			"doc" => "application/msword",
			"docx" => "application/msword",
			"jpg" => "image/jpg",
			"php" => "text/plain",
			"xls" => "application/vnd.ms-excel",
			"ppt" => "application/vnd.ms-powerpoint",
			"gif" => "image/gif",
			"pdf" => "application/pdf",
			"txt" => "text/plain",
			"html"=> "text/html",
			"png" => "image/png",
			"jpeg"=> "image/jpg",
			"js" => "application/x-javascript"
		);
		
		if( array_key_exists($this->extension, $mime_types) ) {
			$mime_type = $mime_types[$this->extension];
		} else {
			$mime_type = "application/msword";
		}

		$file_name = $this->name;
		if($this->extension && !StringUtils::endsWith($file_name, $this->extension)) {
			$file_name .= "." . $this->extension;
		}

		if(Configuration::get('S3_ENABLED')) {
			$s3_region = Configuration::get('S3_REGION');
			$s3_key = Configuration::get('S3_KEY');
			$s3_secret = Configuration::get('S3_SECRET');
			$s3_bucket = Configuration::get('S3_BUCKET');

			$s3Client = S3Client::factory(array(
				'version'    => 'latest',
				'region'      => $s3_region,
				'credentials' => array(
					'key'    => $s3_key,
					'secret' => $s3_secret,
				)
			));

			$s3Client->registerStreamWrapper();
			
            $uploadsPath = "s3://{$s3_bucket}/uploads/{$this->source}";

			$file = array(
				"name" => $file_name,
				"mime" => $mime_type,
				"file_path" => $uploadsPath,
				"content" => file_get_contents($uploadsPath)
			); 
		} else {
			$file = array(
				"name" => $file_name,
				"mime" => $mime_type,
				"file_path" => $file_path,
				"content" => file_get_contents( $file_path ),
			);
		}
		
		return $file;
	}
}
