<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\PDF;

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Site\Site;
use Proxim\Validate;

/**
 * @since 1.5
 */
abstract class HTMLTemplate
{
    public $title;

    public $date;

    /** @var Smarty */
    public $smarty;

    /** @var Site */
    public $site;

    /**
     * Returns the template's HTML header.
     *
     * @return string HTML header
     */
    public function getHeader()
    {
        $this->assignCommonHeaderData();

        return $this->smarty->fetch($this->getTemplate('header'));
    }

    /**
     * Returns the template's HTML footer.
     *
     * @return string HTML footer
     */
    public function getFooter()
    {
        $site_id = (int) $this->site->id;

        $this->smarty->assign([
            'site_email' => Configuration::get('SITE_EMAIL', $site_id, null)
        ]);

        return $this->smarty->fetch($this->getTemplate('footer'));
    }

    /**
     * Returns the invoice logo.
     */
    protected function getLogo()
    {
        $email_logo = Configuration::get('SITE_EMAIL_LOGO');
        if ($email_logo && file_exists(PROX_DIR_UPLOADS . '/' . $email_logo)) {
            return $email_logo;
        }

        $logo = Configuration::get('SITE_LOGO');
        if ($logo && file_exists(PROX_DIR_UPLOADS . '/' . $logo)) {
            return $logo;
        }

        return null;
    }

    /**
     * Assign common header data to smarty variables.
     */
    public function assignCommonHeaderData()
    {
        $this->setSiteId();
        $site_id = (int) $this->site->id;
        $site_name = Configuration::get('SITE_NAME', $site_id, null);
        $site_email = Configuration::get('SITE_EMAIL', $site_id, null);

        $logo = $this->getLogo();

        $width = 0;
        $height = 0;
        if (!empty($logo)) {
            list($width, $height) = getimagesize(PROX_DIR_UPLOADS . '/' . $logo);
        }

        // Limit the height of the logo for the PDF render
        $maximum_height = 100;
        if ($height > $maximum_height) {
            $ratio = $maximum_height / $height;
            $height *= $ratio;
            $width *= $ratio;
        }

        $this->smarty->assign([
            'logo_path' => Configuration::get('UPLOADS_PATH') . '/' . $logo,
            'date' => $this->date,
            'title' => $this->title,
            'site_name' => $site_name,
            'site_email' => $site_email,
            'width_logo' => $width,
            'height_logo' => $height,
        ]);
    }

    /**
     * Assign hook data.
     *
     * @param ObjectModel $object generally the object used in the constructor
     */
    public function assignHookData($object)
    {
        $classname = explode('\\', get_class($this));
        $template = ucfirst(str_replace('HTMLTemplate', '',  end($classname) ));
        $hook_name = 'displayPDF' . $template;

        $this->smarty->assign([
            'HOOK_DISPLAY_PDF' => Hook::exec($hook_name, ['object' => $object]),
        ]);
    }

    /**
     * Returns the template's HTML content.
     *
     * @return string HTML content
     */
    abstract public function getContent();

    /**
     * Returns the template filename.
     *
     * @return string filename
     */
    abstract public function getFilename();

    /**
     * Returns the template filename when using bulk rendering.
     *
     * @return string filename
     */
    abstract public function getBulkFilename();

    /**
     * If the template is not present in the theme directory, it will return the default template
     * in PROX_DIR_PDF directory.
     *
     * @param $template_name
     *
     * @return string
     */
    protected function getTemplate($template_name)
    {
        $template = false;
        $default_template = rtrim(PROX_DIR_PDF, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $template_name . '.tpl';
        $overridden_template = PROX_DIR_THEMES . DIRECTORY_SEPARATOR . 'pdf' . DIRECTORY_SEPARATOR . $template_name . '.tpl';
        if (file_exists($overridden_template)) {
            $template = $overridden_template;
        } elseif (file_exists($default_template)) {
            $template = $default_template;
        }

        return $template;
    }

    protected function setSiteId()
    {
        if (isset($this->order) && Validate::isLoadedObject($this->order)) {
            $site_id = (int) $this->order->site_id;
        } else {
            $site_id = (int) Application::getInstance()->site->id;
        }

        $this->site = new Site($site_id);
        if (Validate::isLoadedObject($this->site)) {
            Application::getInstance()->site = $this->site;
        }
    }

    /**
     * Returns the template's HTML pagination block.
     *
     * @return string HTML pagination block
     */
    public function getPagination()
    {
        return $this->smarty->fetch($this->getTemplate('pagination'));
    }
}
