<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Preference;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\Configuration;
use Proxim\ObjectModel;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;
use Proxim\Tools;

/**
 * Deadline
 */
class Deadline extends ObjectModel
{
	/** @var $id Deadline ID */
    public $id;

	/** @var $id Site ID */
    public $site_id;

    /** @var int academic_level_id */
    public $academic_level_id;

    /** @var int career_level_id */
    public $career_level_id;

    /** @var string name */
    public $name;

    /** @var float price_per_page */
    public $price_per_page = 0;

    /** @var float price_modifier */
    public $price_modifier = 0;

    /** @var int hrs_customer */
    public $hrs_customer;
    
    /** @var int hrs_writer */
    public $hrs_writer;
    
    /** @var int min_pages */
	public $min_pages = 1;

    /** @var int is_academic_writing */
    public $is_academic_writing = 0;

    /** @var int is_calculations */
    public $is_calculations = 0;

    /** @var int is_programming */
    public $is_programming = 0;

    /** @var int is_article_writing */
    public $is_article_writing = 0;

    /** @var int is_resume */
    public $is_resume = 0;
    
    /** @var string date_upd */
	public $date_upd;
	
	/** @var string date_add */
	public $date_add;
	
	/**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'deadline',
        'primary' => 'deadline_id',
        'fields' => array( 
            'site_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt', 'required'=> true), 
            'academic_level_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'career_level_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'name' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 255),
            'price_per_page' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'price_modifier' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'hrs_customer' => array('type' => self::TYPE_INT, 'validate' => 'isInt', 'required'=> true),
            'hrs_writer' => array('type' => self::TYPE_INT, 'validate' => 'isInt', 'required'=> true),
            'min_pages' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'is_academic_writing' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_calculations' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_programming' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_article_writing' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_resume' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
			'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
        ),
    );

    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null) {
		parent::__construct($id);

        if($this->id) {
            $percentage = (int) Configuration::get('CPP_PERCENTAGE', $this->site_id, 100);
            $this->price_per_page = ($percentage*$this->price_per_page)/100;
        }
    }
    
    /**
     * Return deadlines by site ID.
     *
     * @return array Deadlines
     */
    public static function getDeadlines( $site_id = null )
    {
        $sql = self::getDeadlinesQuery( $site_id );
        $sql->orderBy('d.hrs_customer ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return deadlines by site ID.
     *
     * @return array Deadlines
     */
    public static function getAcademicWritingDeadlines( $site_id = null )
    {
        $sql = self::getDeadlinesQuery( $site_id );
        $sql->where('d.is_academic_writing = 1');
        $sql->orderBy('d.hrs_customer ASC');
        $result = Db::getInstance()->executeS($sql);

        $results = Db::getInstance()->executeS($sql);
        
        $deadlines = array();
        foreach($results as $deadline) {
            $percentage = (float) Configuration::get('CPP_PERCENTAGE', $deadline['site_id'], 100);
            $deadline['price_per_page'] = ($percentage*$deadline['price_per_page'])/100;
            $deadlines[] = $deadline;
        }
        
        return $deadlines;
    }

    /**
     * Return deadlines by site ID.
     *
     * @return array Deadlines
     */
    public static function getArticleWritingDeadlines( $site_id = null )
    {
        $sql = self::getDeadlinesQuery( $site_id );
        $sql->where('d.is_article_writing = 1');
        $sql->orderBy('d.hrs_customer ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return deadlines by site ID.
     *
     * @return array Deadlines
     */
    public static function getProgrammingDeadlines( $site_id = null )
    {
        $sql = self::getDeadlinesQuery( $site_id );
        $sql->where('d.is_programming = 1');
        $sql->orderBy('d.hrs_customer ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return deadlines by site ID.
     *
     * @return array Deadlines
     */
    public static function getCalculationsDeadlines( $site_id = null )
    {
        $sql = self::getDeadlinesQuery( $site_id );
        $sql->where('d.is_calculations = 1');
        $sql->orderBy('d.hrs_customer ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Get Disciplines query.
     *
     * @return DbQuery
     */
    public static function getDeadlinesQuery( $site_id )
    {
        $sql = new DbQuery();
        $sql->select('d.*');
        $sql->from('deadline', 'd');
        if($site_id) {
            $sql->where('d.site_id = ' . (int) $site_id );
        }

        return $sql;
    }
}