<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Preference;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\ObjectModel;
use Proxim\Order\Order;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;
use Proxim\Tools;

/**
 * Discipline
 */
class Discipline extends ObjectModel
{
    const DISCIPLINE_OTHER = 52;

	/** @var $id Discipline ID */
    public $id;

    /** @var int group_id */
    public $group_id;

    /** @var string title */
    public $title;

    /** @var int is_complex_assignment */
    public $is_complex_assignment = 0;

    /** @var int is_academic_writing */
    public $is_academic_writing = 0;

    /** @var int is_calculations */
    public $is_calculations = 0;

    /** @var int is_programming */
    public $is_programming = 0;

    /** @var int is_article_writing */
    public $is_article_writing = 0;
    
    /** @var string date_upd */
	public $date_upd;
	
	/** @var string date_add */
	public $date_add;
	
	/**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'discipline',
        'primary' => 'discipline_id',
        'fields' => array( 
            'group_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt', 'required'=> true),
            'title' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 255),
            'is_complex_assignment' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_academic_writing' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_calculations' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_programming' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_article_writing' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
			'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
        ),
    );

    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null) {
		parent::__construct($id);
    }
    
    /**
     * Return disciplines 
     *
     * @return array Disciplines
     */
    public static function getDisciplines( $orderServiceType = null )
    {
        switch($orderServiceType) {
            case Order::ARTICLE_WRITING:
                $result = self::getArticleWritingsDisciplines();
                break;

            case Order::PROGRAMMING:
                $result = self::getProgrammingDisciplines();
                break;

            case Order::CALCULATIONS:
                $result = self::getCalculationsDisciplines();
                break;

            case Order::COMPLEX_ASSIGNMENT:
                $result = self::getAcademicWritingDisciplines();
                break;

            default:
                $sql = self::getDisciplinesQuery();
                $sql->orderBy('d.discipline_id ASC');
                $result = Db::getInstance()->executeS($sql);
                break;
        }

        return $result;
    }

    /**
     * Return Programming disciplines 
     *
     * @return array Disciplines
     */
    public static function getAcademicWritingDisciplines()
    {
        $sql = self::getDisciplinesQuery();
        $sql->where('d.is_academic_writing = 1');
        $sql->orderBy('d.discipline_id ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return Programming disciplines 
     *
     * @return array Disciplines
     */
    public static function getProgrammingDisciplines()
    {
        $sql = self::getDisciplinesQuery();
        $sql->where('d.is_programming = 1');
        $sql->orderBy('d.discipline_id ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return Calculations disciplines 
     *
     * @return array Disciplines
     */
    public static function getCalculationsDisciplines()
    {
        $sql = self::getDisciplinesQuery();
        $sql->where('d.is_calculations = 1');
        $sql->orderBy('d.discipline_id ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return Calculations disciplines 
     *
     * @return array Disciplines
     */
    public static function getArticleWritingsDisciplines()
    {
        $sql = self::getDisciplinesQuery();
        $sql->where('d.is_article_writing = 1');
        $sql->orderBy('d.discipline_id ASC');
        $result = Db::getInstance()->executeS($sql);
        return $result;
    }

    /**
     * Return disciplines by Group ID.
     *
     * @return array Disciplines
     */
    public static function getDisciplinesByGroup( $orderServiceType = null )
    {
        $disciplines = Discipline::getDisciplines( $orderServiceType );
        $groupedDisciplineGroups = ArrayUtils::groupByKey($disciplines, 'group_id');

        $disciplineGroups = array();

        // discipline groups
        if(count($groupedDisciplineGroups) > 0) {
            foreach($groupedDisciplineGroups as $disciplineGroupId => $disciplineGroupItems) {
                $discipline = new Discipline( (int) $disciplineGroupId );
                if(Validate::isLoadedObject($discipline)) {
                    $disciplineGroups[] = array(
                        'id' => $discipline->id,
                        'title' => $discipline->title,
                        'disciplines' => $disciplineGroupItems
                    );
                } else {
                    $disciplineGroups[] = array(
                        'id' => self::DISCIPLINE_OTHER,
                        'title' => 'Other',
                        'disciplines' => $disciplineGroupItems
                    );
                }
            }
        } else {
            $disciplineGroups[] = array(
                'id' => self::DISCIPLINE_OTHER,
                'title' => 'Other',
                'disciplines' => $disciplines
            );
        }
        

        return $disciplineGroups;
    }

    public static function getTopDisciplines( $limit = 5 ) {
        $sql = new DbQuery();
        $sql->select('o.topic_category_id, d.title, COUNT(*) AS discipline_count');
        $sql->from('order', 'o');
        $sql->innerJoin('discipline', 'd', 'd.discipline_id = o.topic_category_id');
        $sql->where('o.topic_category_id != ' . (int) self::DISCIPLINE_OTHER );
        $sql->groupBy('o.topic_category_id');
        $sql->orderBy('discipline_count DESC');
        $sql->limit( $limit );

        $result = Db::getInstance()->executeS($sql);

        return $result;
    }

    /**
     * Get Disciplines query.
     *
     * @return DbQuery
     */
    public static function getDisciplinesQuery()
    {
        $query = new DbQuery();
        $query->select('d.*');
        $query->from('discipline', 'd');

        return $query;
    }
}