<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2020 Proxim
 * @version    1.0
 * @since      File available since Release 1.0
*/

namespace Proxim\Presenter\Deadline;

use Proxim\Presenter\AbstractLazyArray;
use Doctrine\Common\Annotations\AnnotationException;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Currency;
use Proxim\Preference\Deadline;
use Proxim\Util\ArrayUtils;

class DeadlineLazyArray extends AbstractLazyArray {
    /** @var Application */
    private $application;

    /** @var Deadline */
    private $deadline;

    /**
     * DeadlineArray constructor.
     *
     * @throws AnnotationException
     * @throws ReflectionException
     */
    public function __construct(Deadline $deadline)
    {
        $this->application = Application::getInstance();
        $this->deadline = $deadline;

        parent::__construct();
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getId()
    {
        return (int) $this->deadline->id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getAcademicLevel()
    {
        return (int) $this->deadline->academic_level_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getAcademicLevelTitle()
    {
        global $globals;
        if($this->deadline->academic_level_id) {
            $academicLevels = ArrayUtils::get($globals, 'academicLevels');
            $academicLevel = ArrayUtils::get($academicLevels, $this->deadline->academic_level_id);
    
            return (string) ArrayUtils::get($academicLevel, 'title');
        }

        return "";
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCareerLevel()
    {
        return (int) $this->deadline->career_level_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCareerLevelTitle()
    {
        
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getName()
    {
        return $this->deadline->name;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getPricePerPage()
    {
        $deadline = $this->deadline;
        $currency_id = Currency::getIdByIsoCode('USD');
        $currency = Currency::getCurrencyInstance( $currency_id );

        return array(
            'currency' => $currency->iso_code,
            'value' => toFixed($this->deadline->price_per_page),
            'amountFormatted' => formatPrice($this->deadline->price_per_page),
        );
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getPricePerWord()
    {
        $deadline = $this->deadline;
        $currency_id = Currency::getIdByIsoCode('USD');
        $currency = Currency::getCurrencyInstance( $currency_id );

        return array(
            'currency' => $currency->iso_code,
            'value' => toFixed($deadline->price_per_page, 3),
            'amountFormatted' => formatPrice($deadline->price_per_page, [
                'fixed' => 3
            ]),
        );
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getPriceModifier()
    {
        $deadline = $this->deadline;
        $currency_id = Currency::getIdByIsoCode('USD');
        $currency = Currency::getCurrencyInstance( $currency_id );

        return array(
            'currency' => $currency->iso_code,
            'value' => toFixed($deadline->price_modifier),
            'amountFormatted' => formatPrice($deadline->price_modifier),
        );
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getHrs()
    {
        return (int) $this->deadline->hrs_customer;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getHrsWriter()
    {
        return (int) $this->deadline->hrs_writer;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getMinPages()
    {
        return (int) $this->deadline->min_pages;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getIsAcademicWriting()
    {
        return (bool) $this->deadline->is_academic_writing;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getIsCalculations()
    {
        return (bool) $this->deadline->is_calculations;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getIsProgramming()
    {
        return (bool) $this->deadline->is_programming;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getIsArticleWriting()
    {
        return (bool) $this->deadline->is_article_writing;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getIsResume()
    {
        return (bool) $this->deadline->is_resume;
    }
}