<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2020 Proxim
 * @version    1.0
 * @since      File available since Release 1.0
*/

namespace Proxim\Presenter\Order;

use Proxim\Presenter\AbstractLazyArray;
use Doctrine\Common\Annotations\AnnotationException;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Order\File as OrderFile;
use Proxim\Order\Order;
use Proxim\Preference\UploadType;
use Proxim\Tools;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class OrderFileLazyArray extends AbstractLazyArray {
    /** @var Application */
    private $application;

    /** @var User */
    public $user;

    /** @var Order */
    private $order;

    /** @var OrderFile */
    private $orderFile;

    /**
     * OrderFileLazyArray constructor.
     *
     * @throws AnnotationException
     * @throws ReflectionException
     */
    public function __construct(OrderFile $orderFile)
    {
        $this->application = Application::getInstance();
        $this->user = $this->application->user;
        $this->orderFile = $orderFile;
        $this->order = new Order( (int) $orderFile->order_id);

        parent::__construct();
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getId()
    {
        return (int) $this->orderFile->id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getDownloadId()
    {
        return (int) $this->orderFile->id;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getName()
    {
        return $this->orderFile->name;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getFileTypeId()
    {
        return (int) $this->orderFile->file_type_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getFileDescription()
    {
        if($this->orderFile->file_type_id == OrderFile::OTHER) {
            return $this->orderFile->file_description;
        }

        $uploadType = new UploadType( $this->orderFile->file_type_id );
        if(Validate::isLoadedObject($uploadType)) {
            return $uploadType->title;
        }
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getExtension()
    {
        return $this->orderFile->extension;
    }
    
    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSize()
    {
        return (int) $this->orderFile->size;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSizeFormatted()
    {
        return formatBytes($this->orderFile->size);
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDownloadLink()
    {
        if(Configuration::get('S3_ENABLED') || Configuration::get('DIGITALOCEAN_ENABLED')) {
            $site_url = Configuration::get('SITE_DOMAIN');
            return $site_url . '/api/files/' . $this->orderFile->id . '/download';
        } elseif( Configuration::get('INTERNAL_FILE_STORAGE') ) {
            $site_url = Configuration::get('SITE_DOMAIN', $this->orderFile->site_id);
            return $site_url . '/api/ifs/download/' . $this->orderFile->id;
        } else { 
            $site_url = Configuration::get('SITE_DOMAIN');
            return $site_url . '/api/files/' . $this->orderFile->id . '/download';
        }
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsUploadedByCustomer() {
        return $this->orderFile->uploader_id == $this->order->customer_id ? true : false;
    }
    
    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsApprovedByCustomer()
    {
        return (bool) $this->order->status_id == Order::FINISHED;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsDownloaded()
    {
        return $this->orderFile->downloaded;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDownloadedAt()
    {
        return DateUtils::convertUtcDateTimeToTimeZone($this->orderFile->downloaded_at, $this->user->timezone);
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsNew()
    {
        return (bool) $this->orderFile->downloaded === false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getActionLabel()
    {
        if ( $this->order->status_id == Order::FINISHED ) {
			return "Final version available for download. If you have any issues with the latest version of your paper, be sure to contact our support team.";
        } else {
            return "Preview version awaiting to be checked before approval. MS Word version of the paper will be available after you accept it.";
        }
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getDownloadButtonVisible()
    {
        return (bool) $this->order->status_id === Order::DELIVERED && $this->order->status_id === Order::FINISHED;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getApproveButtonVisible()
    {
        return (bool) $this->order->status_id === Order::DELIVERED;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getHoursAgoUploaded()
    {
        return DateUtils::since($this->orderFile->date_add, 2);
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getCreatedAt()
    {
        $user = $this->user;
        return DateUtils::convertUtcDateTimeToTimeZone($this->orderFile->date_add, $user->timezone);
    }
}