<?php 
/**
 * The base configuration for Proxim
 * @package    Proxim
 * @author     Davison Pro <davisonpro.coder@gmail.com | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.0.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Aws\S3\Exception\S3Exception;
use Aws\S3\S3Client;
use Exception;
use Proxim\Util\ArrayUtils;
use Proxim\Util\Formatting;
use Proxim\Util\StringUtils;
use SpacesConnect;

/**
 * Class Uploader.
 */
class Uploader
{
    const DEFAULT_MAX_SIZE = 100485760;

    private $_check_file_size;
    private $_accept_types;
    private $_files;
    private $_max_size;
    private $_name;
    private $_prefix;
    private $_save_path;
    private $_cloud_storage = true;

    /**
     * Uploader constructor.
     *
     * @param null $name
     */
    public function __construct($name = null)
    {
        $app = Application::getInstance();
        $this->setName($name);
        $this->setCheckFileSize(true);
        $this->files = array();
    }

    public function setCloudStorage( $cloudStorage = true ) {
        $this->_cloud_storage = $cloudStorage;
        return $this;
    }

    /**
     * @param $value
     *
     * @return $this
     */
    public function setAcceptTypes($value)
    {
        $this->_accept_types = $value;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getAcceptTypes()
    {
        return $this->_accept_types;
    }

    /**
     * @param $value
     *
     * @return $this
     */
    public function setCheckFileSize($value)
    {
        $this->_check_file_size = $value;

        return $this;
    }

    /**
     * @param string|null $fileName
     *
     * @return string
     */
    public function getFilePath($fileName = null)
    {
        if (!isset($fileName)) {
            return tempnam($this->getSavePath(), $this->getUniqueFileName());
        }

        $fileName = str_replace('#', '', $fileName);

        $pathInfo = pathinfo($fileName);
        $rawFileName = ArrayUtils::get($pathInfo, 'filename');
        $extension = Tools::strtolower(ArrayUtils::get($pathInfo, 'extension'));

        $prefix = (string) $this->getPrefix();
        if($prefix && !StringUtils::startsWith($rawFileName, $prefix)) {
			$rawFileName = $prefix . '-'. $rawFileName;
        }

        if(Configuration::get('S3_ENABLED') && $this->_cloud_storage) {
            $s3_region = Configuration::get('S3_REGION');
            $s3_key = Configuration::get('S3_KEY');
            $s3_secret = Configuration::get('S3_SECRET');
            $s3_bucket = Configuration::get('S3_BUCKET');

            $s3Client = S3Client::factory(array(
                'version'    => 'latest',
                'region'      => $s3_region,
                'credentials' => array(
                    'key'    => $s3_key,
                    'secret' => $s3_secret,
                )
            ));

            $Key = 'uploads/' . $fileName;

            // generate a new name if it exists in droplet
            try {
                $result = $s3Client->headObject([
                    'Bucket' => $s3_bucket,
                    'Key' => $Key,
                ]);

                $fileName = $rawFileName . '_' . time() .'.'. $extension;

            } catch (S3Exception $e) {
                // do nothing
            }
        } else {
            if (file_exists($this->getSavePath() . $fileName)) {
                $i = 1;
                while (file_exists( $this->getSavePath() . $rawFileName . '_' . $i .'.'. $extension)) $i++;
                $fileName = $rawFileName . '_' . $i .'.'. $extension;
            } 
        }

        return $fileName;
    }

    /**
     * @return array
     */
    public function getFiles()
    {
        if (!isset($this->_files)) {
            $this->_files = array();
        }

        return $this->_files;
    }

    /**
     * @param $value
     *
     * @return $this
     */
    public function setMaxSize($value)
    {
        $this->_max_size = (int) $value;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getMaxSize()
    {
        if (!isset($this->_max_size) || empty($this->_max_size)) {
            $this->setMaxSize(self::DEFAULT_MAX_SIZE);
        }

        return $this->_max_size;
    }

    /**
     * @return mixed
     */
    public function getName()
    {
        return $this->_name;
    }

    /**
     * @param $value
     *
     * @return $this
     */
    public function setName($value)
    {
        $this->_name = $value;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getPrefix()
    {
        return $this->_prefix;
    }

    /**
     * @param $value
     *
     * @return $this
     */
    public function setPrefix($value)
    {
        $this->_prefix = $value;

        return $this;
    }

    /**
     * @param $value
     *
     * @return $this
     */
    public function setSavePath($value)
    {
        $this->_save_path = $value;

        return $this;
    }

    /**
     * @return int|null
     */
    public function getPostMaxSizeBytes()
    {
        $postMaxSize = ini_get('post_max_size');
        $bytes = (int) trim($postMaxSize);
        $last = strtolower($postMaxSize[strlen($postMaxSize) - 1]);

        switch ($last) {
            case 'g':
                $bytes *= 1024;
                // no break
            case 'm':
                $bytes *= 1024;
                // no break
            case 'k':
                $bytes *= 1024;
        }

        if ($bytes == '') {
            $bytes = null;
        }

        return $bytes;
    }

    /**
     * @return string
     */
    public function getSavePath()
    {
        if (!isset($this->_save_path)) {
            $this->setSavePath(PROX_DIR_UPLOADS);
        }

        return $this->_normalizeDirectory($this->_save_path);
    }

    /**
     * @param string $prefix
     *
     * @return string
     */
    public function getUniqueFileName($prefix = 'PROX')
    {
        return uniqid($prefix, true);
    }

    /**
     * @return bool
     */
    public function checkFileSize()
    {
        return isset($this->_check_file_size) && $this->_check_file_size;
    }

    /**
     * @param null $dest
     *
     * @return array
     */
    public function process($dest = null)
    {
        $upload = isset($_FILES[$this->getName()]) ? $_FILES[$this->getName()] : null;

        if ($upload && is_array($upload['tmp_name'])) {
            $tmp = array();
            foreach ($upload['tmp_name'] as $index => $value) {
                $tmp[$index] = array(
                    'tmp_name' => $upload['tmp_name'][$index],
                    'name' => $upload['name'][$index],
                    'size' => $upload['size'][$index],
                    'type' => $upload['type'][$index],
                    'error' => $upload['error'][$index],
                );

                $this->files[] = $this->upload($tmp[$index], $dest);
            }
        } elseif ($upload) {
            $this->files = $this->upload($upload, $dest);
        }

        return $this->files;
    }

    /**
     * @param $filename
     *
     * @return mixed
     */
    public function getMemeType($filename)
    {
        $mimeType = '';
       
        // Try with FileInfo
        if (!$mimeType && function_exists('finfo_open')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $filename);
            finfo_close($finfo);
        }

        // Try with Mime
        if (!$mimeType && function_exists('mime_content_type')) {
            $mimeType = mime_content_type($filename);
        }

        // Try with exec command and file binary
        if (!$mimeType && function_exists('exec')) {
            $mimeType = trim(exec('file -b --mime-type ' . escapeshellarg($filename)));
            if (!$mimeType) {
                $mimeType = trim(exec('file --mime ' . escapeshellarg($filename)));
            }
            if (!$mimeType) {
                $mimeType = trim(exec('file -bi ' . escapeshellarg($filename)));
            }
        }

        return $mimeType;
    }

    /**
     * @param $file
     * @param null $dest
     *
     * @return mixed
     */
    public function upload($file)
    {
        if ($this->validate($file)) {
            $info = pathinfo($file['name']);
            $file['extension'] = ArrayUtils::get($info, 'extension');

            $sourceFileName = $this->getFilePath(isset($dest) ? $dest : $file['name']);
            $file['source_name'] = $sourceFileName;
            $savePath = $this->getSavePath();

            if(!file_exists($savePath)) {
                @mkdir($savePath, 0777, true);
            }

            $filePath = $savePath . $sourceFileName;

            if ($file['tmp_name'] && is_uploaded_file($file['tmp_name'])) {
                move_uploaded_file($file['tmp_name'], $filePath);
            } else {
                // Non-multipart uploads (PUT method support)
                file_put_contents($filePath, fopen('php://input', 'rb'));
            }

            $fileSize = $this->_getFileSize($filePath, true);
    
            if ($fileSize === $file['size']) {
                $file['save_path'] = $filePath;
            } else {
                $file['size'] = $fileSize;
                unlink($filePath);
                $file['error'] = 'Server file size is different from local file size';
            }

            // upload to
            if(Configuration::get('S3_ENABLED') && $this->_cloud_storage) {
                $contentType = $this->getMemeType($filePath);

                /* Amazon S3 */
                $this->awsS3Upload($filePath, $sourceFileName, $contentType);
            } elseif( Configuration::get('DIGITALOCEAN_ENABLED') && $this->_cloud_storage ) {
                $contentType = $this->getMemeType($filePath);
                
                /* DigitalOcean */
                $this->digitaloceanSpaceUpload($filePath, $sourceFileName, $contentType);
            }
        }

        $prefix = (string) $this->getPrefix();
        if($prefix && !StringUtils::startsWith($file['name'], $prefix)) {
			$file['name'] = $prefix . '-'. $file['name'];
        }

        return $file;
    }

    public function cloudUpload($filePath, $sourceFileName, $contentType = '') {
        // upload to
        if(Configuration::get('S3_ENABLED')) {
            /* Amazon S3 */
            $this->awsS3Upload($filePath, $sourceFileName, $contentType);
        } elseif( Configuration::get('DIGITALOCEAN_ENABLED')) {
            /* DigitalOcean */
            $this->digitaloceanSpaceUpload($filePath, $sourceFileName, $contentType);
        }
    }

    /**
     * @param $error_code
     *
     * @return array|int|mixed|string
     */
    protected function checkUploadError($error_code)
    {
        $error = 0;
        switch ($error_code) {
            case 1:
                $error = sprintf('The uploaded file exceeds %s', ini_get('upload_max_filesize'));

                break;
            case 2:
                $error = sprintf('The uploaded file exceeds %s', ini_get('post_max_size'));

                break;
            case 3:
                $error = 'The uploaded file was only partially uploaded';

                break;
            case 4:
                $error = 'No file was uploaded';

                break;
            case 6:
                $error = 'Missing temporary folder';

                break;
            case 7:
                $error = 'Failed to write file to disk';

                break;
            case 8:
                $error = 'A PHP extension stopped the file upload';

                break;
            default:
                break;
        }

        return $error;
    }

    /**
     * @param $file
     *
     * @return bool
     */
    protected function validate(&$file)
    {
        $file['error'] = $this->checkUploadError($file['error']);

        $postMaxSize = $this->getPostMaxSizeBytes();

        if ($postMaxSize && ($this->_getServerVars('CONTENT_LENGTH') > $postMaxSize)) {
            $file['error'] = 'The uploaded file exceeds the post_max_size directive in php.ini';

            return false;
        }

        if (preg_match('/\%00/', $file['name'])) {
            $file['error'] = 'Invalid file name';

            return false;
        }

        $types = $this->getAcceptTypes();

        //TODO check mime type.
        if (isset($types) && !in_array(Tools::strtolower(pathinfo($file['name'], PATHINFO_EXTENSION)), $types)) {
            $file['error'] = 'Filetype not allowed';
            return false;
        }

        if ($this->checkFileSize() && $file['size'] > $this->getMaxSize()) {
            $file['error'] = sprintf('File is too big. Current size is %1s, maximum size is %2s.', $file['size'], $this->getMaxSize());

            return false;
        }

        return true;
    }

    /**
     * @param string $filePath
     * @param bool $clearStatCache
     *
     * @return int
     *
     * @deprecated 1.7.0
     */
    protected function _getFileSize($filePath, $clearStatCache = false)
    {
        return $this->getFileSize($filePath, $clearStatCache);
    }

    /**
     * @param string $filePath
     * @param bool $clearStatCache
     *
     * @return int
     *
     * @since 1.7.0
     */
    protected function getFileSize($filePath, $clearStatCache = false)
    {
        if ($clearStatCache) {
            clearstatcache(true, $filePath);
        }

        return filesize($filePath);
    }

    /**
     * @param $var
     *
     * @return string
     *
     * @deprecated 1.7.0
     */
    protected function _getServerVars($var)
    {
        return $this->getServerVars($var);
    }

    /**
     * @param $var
     *
     * @return string
     *
     * @since 1.7.0
     */
    protected function getServerVars($var)
    {
        return isset($_SERVER[$var]) ? $_SERVER[$var] : '';
    }

    /**
     * @param $directory
     *
     * @return string
     *
     * @deprecated 1.7.0
     */
    protected function _normalizeDirectory($directory)
    {
        return $this->normalizeDirectory($directory);
    }

    /**
     * @param $directory
     *
     * @return string
     *
     * @since 1.7.0
     */
    protected function normalizeDirectory($directory)
    {
        $last = $directory[strlen($directory) - 1];

        if (in_array($last, array('/', '\\'))) {
            $directory[strlen($directory) - 1] = DIRECTORY_SEPARATOR;

            return $directory;
        }

        $directory .= DIRECTORY_SEPARATOR;

        return $directory;
    }

    public function digitaloceanSpaceTest() {
        try {
            $digitalocean_key = Configuration::get('DIGITALOCEAN_KEY');
            $digitalocean_secret = Configuration::get('DIGITALOCEAN_SECRET');
            $digitalocean_space_name = Configuration::get('DIGITALOCEAN_SPACE_NAME');
            $digitalocean_space_region = Configuration::get('DIGITALOCEAN_SPACE_REGION');

            $space = new SpacesConnect(
                $digitalocean_key, 
                $digitalocean_secret, 
                $digitalocean_space_name, 
                $digitalocean_space_region
            );

            $buckets = $space->ListSpaces();
            if(empty($buckets)) {
                throw new Exception("There is no spaces in your account");
            }
            if(!$space->GetSpaceName()) {
                throw new Exception("There is no space with this name in your account");
            }
            $space->PutCORS(array(
                'AllowedHeaders' => array(
                    'Authorization'
                ),
                'AllowedMethods' => array(
                    'POST',
                    'GET',
                    'PUT'
                ), // REQUIRED
                'AllowedOrigins' => array(
                    '*'
                ), // REQUIRED
                'ExposeHeaders' => array(),
                'MaxAgeSeconds' => 3000
            ));
        } catch (Exception $e) {
            throw new Exception("Connection Failed, Please check your settings");
        }
    }

    /**
     * digitaloceanSpaceUpload
     *
     * @param string $file_source
     * @param string $file_name
     * @return void
     */
    public function digitaloceanSpaceUpload($file_source, $file_name, $content_type = '') {
        $digitalocean_key = Configuration::get('DIGITALOCEAN_KEY');
        $digitalocean_secret = Configuration::get('DIGITALOCEAN_SECRET');
        $digitalocean_space_name = Configuration::get('DIGITALOCEAN_SPACE_NAME');
        $digitalocean_space_region = Configuration::get('DIGITALOCEAN_SPACE_REGION');

        $space = new SpacesConnect(
            $digitalocean_key, 
            $digitalocean_secret, 
            $digitalocean_space_name, 
            $digitalocean_space_region
        );
        
        $Key = 'uploads/' . $file_name;
        $space->UploadFile($file_source, "public", $Key, "");
        /* remove local file */
        if($space->DoesObjectExist($Key)) {
            unlink($file_source);
        }
    }

    /* ------------------------------- */
    /* Cloud Storage */
    /* ------------------------------- */
    public function awsS3Test() {
        try {
            $s3_region = Configuration::get('S3_REGION');
            $s3_key = Configuration::get('S3_KEY');
            $s3_secret = Configuration::get('S3_SECRET');
            $s3_bucket = Configuration::get('S3_BUCKET');

            $s3Client = S3Client::factory(array(
                'version'    => 'latest',
                'region'      => $s3_region,
                'credentials' => array(
                    'key'    => $s3_key,
                    'secret' => $s3_secret,
                )
            ));
            $buckets = $s3Client->listBuckets();
            if(empty($buckets)) {
                throw new Exception("There is no buckets in your account");
            }
            if(!$s3Client->doesBucketExist($s3_bucket)) {
                throw new Exception("There is no bucket with this name in your account");
            }
        } catch (Exception $e) {
            throw new Exception("Connection Failed, Please check your settings");
        }
    }

    /**
     * awsS3Upload
     *
     * @param string $file_source
     * @param string $file_name
     * @return void
     */
    public function awsS3Upload($file_source, $file_name, $content_type = '') {
        $s3_region = Configuration::get('S3_REGION');
        $s3_key = Configuration::get('S3_KEY');
        $s3_secret = Configuration::get('S3_SECRET');
        $s3_bucket = Configuration::get('S3_BUCKET');

        $s3Client = S3Client::factory(array(
            'version'    => 'latest',
            'region'      => $s3_region,
            'credentials' => array(
                'key'    => $s3_key,
                'secret' => $s3_secret,
            )
        ));

        $Key = 'uploads/' . $file_name;

        // $exists = $s3Client->getObjectInfo($s3_bucket, $Key);

        $s3Client->putObject([
            'Bucket' => $s3_bucket,
            'Key'    => $Key,
            'Body'   => fopen($file_source, 'r+'),
            'ACL'    => 'public-read',
            'ContentDisposition' => 'inline',
            'ContentType' => $content_type,
        ]);

        /* remove local file */
        gc_collect_cycles();

        if($s3Client->doesObjectExist($s3_bucket, $Key)) {
            unlink($file_source);
        }
    }
}
