<?php 
/**
 * @package    Proxim 
 * @author     Davison Pro <davis@davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\User;

use Db;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use Proxim\Application;
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\ObjectModel;
use Proxim\Site\Site;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Util\ArrayUtils;

/**
 * Customer
 */
class Customer extends ObjectModel {
    /** @var $id Customer ID */
    public $id;

    /** @var $id Employee Pool ID */
    public $employee_pool_id;

    /** @var string $google_analytics_id */
    public $google_analytics_id;

    /** @var string $email */
    public $email;

    /** @var string $name */
    public $name;

    /** @var string $phone */
    public $phone;

    /** @var int $country_id */
    public $country_id;

    /** @var int $site_id */
    public $site_id = PROX_SITE_ID;

    /** @var string $password */
    public $password;

    /** @var string $new_password */
    public $new_password;

    /** @var string $reset_password_token */
    public $reset_password_token;

    /** @var string $reset_password_validity */
    public $reset_password_validity;

    /** @var string $last_passwd_gen */
    public $last_passwd_gen;

    /** @var string $secure_key */
    public $secure_key;

    /** @var bool $policy_accepted */
    public $policy_accepted = 0;

    /** @var string $policy_accepted_at */
    public $policy_accepted_at;

    /** @var float $total_spent */
    public $total_spent = 0;

    /** @var int $referrer_id */
    public $referrer_id;

    /** @var bool $affiliate_used */
    public $affiliate_used = 0;

    /** @var float $affiliate_balance */
    public $affiliate_balance = 0;

    /** @var string $last_academic_level */
    public $last_academic_level;

    /** @var string $last_payment_method */
    public $last_payment_method;

    /** @var bool $is_subscribed */
    public $is_subscribed = 0;

    /** @var bool $is_test */
    public $is_test = 0;

    /** @var bool is_anonymous */
    public $is_anonymous = 0;

    /** @var bool anonymous_id_saved */
    public $anonymous_id_saved = 0;

    /** @var bool $is_banned */
    public $is_banned = 0;

    /** @var bool $push_notifications */
    public $push_notifications = 0;

    /** @var bool $social_connected */
    public $social_connected = 0;

    /** @var string $social_id */
    public $social_id;

    /** @var string $last_login */
    public $last_login;

    /** @var string $last_activity */
    public $last_activity;

    /** @var string $reg_date */
    public $reg_date;

    /** @var bool is the user logged in */
    public $logged = false;

    /** @var int $friends_invited */
    public $friends_invited = 0;

    /** @var int $total_orders */
    public $total_orders = 0;

    /** @var string $site_name */
    public $site_name;

    /** @var string $site_url */
    public $site_url;

    /** @var array $country */
    public $country = array();
    
    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'customer',
        'primary' => 'customer_id',
        'fields' => array(
            'employee_pool_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'google_analytics_id' => array('type' => self::TYPE_STRING, 'size' => 255),
            'email' => array('type' => self::TYPE_STRING, 'validate' => 'isEmail', 'required' => true, 'size' => 255),
            'name' => array('type' => self::TYPE_STRING, 'validate' => 'isName', 'required' => false, 'size' => 255),
            'phone' => array('type' => self::TYPE_STRING, 'validate' => 'isPhoneNumber', 'required' => false, 'size' => 50),
            'country_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'site_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'password' => array('type' => self::TYPE_STRING, 'validate' => 'isPasswd', 'required' => true, 'size' => 255),
            'new_password' => array('type' => self::TYPE_STRING, 'validate' => 'isPasswd', 'required' => true, 'size' => 255),
            'reset_password_token' => array('type' => self::TYPE_STRING, 'validate' => 'isSha1', 'size' => 40),
            'reset_password_validity' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'last_passwd_gen' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'secure_key' => array('type' => self::TYPE_STRING, 'validate' => 'isMd5'),
            'policy_accepted' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'policy_accepted_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'total_spent' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total_orders' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'referrer_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'affiliate_used' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'affiliate_balance' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'last_academic_level' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'last_payment_method' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'is_subscribed' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_test' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_anonymous' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'anonymous_id_saved' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'is_banned' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'push_notifications' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'social_connected' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'social_id' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'last_login' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'last_activity' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'reg_date' => array('type' => self::TYPE_DATE, 'validate' => 'isDate')
        )
    );

    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null)
    {
        parent::__construct($id);

        if($this->id) {
            $this->getPhoneNumber();

            if($this->site_id) {
                $site = new Site( (int) $this->site_id );
                if(Validate::isLoadedObject($site)) {
                    $this->site_name = $site->name;
                    $this->site_url = $site->domain;
                }
            }
        }
	}

    public function getPhoneNumber() {
        global $globals;

        // phone number parsing
        $phoneUtil = PhoneNumberUtil::getInstance();
        $countriesByName = $globals['countries'];
        $countriesByCode = array();
        foreach($countriesByName as $countryCode => $country) {
            $country['countryCode'] = $countryCode;
            $countriesByCode[$country['code']] = $country;
        }

        if(!$this->country_id) {
            $this->country_id = '840';
        }

        $country = ArrayUtils::get($countriesByCode, $this->country_id, array());
        $this->country = $country;
        if( is_array($this->country) && ArrayUtils::has($this->country, 'countryCode')) {
            $phoneCode = ArrayUtils::get($this->country, 'countryCode');
            try {
                $phoneNumberProto = $phoneUtil->parse($this->phone, strtoupper($phoneCode) );
                $this->phone = $phoneUtil->format($phoneNumberProto, PhoneNumberFormat::INTERNATIONAL);
            } catch (\libphonenumber\NumberParseException $e) {
            }
        }
    }

    public function add($autoDate = true, $nullValues = true)
    {
        $this->secure_key = md5(uniqid(rand(), true));
        $this->last_passwd_gen = DateUtils::now();

        if ( !$this->getNextCustomerId() ) {
            $this->id = 452441;
        } else {
            $next_customer_id = Db::getInstance()->getValue('SELECT MAX(`customer_id`)+2 FROM `' . DB_PREFIX . 'customer`');
            $this->id = $next_customer_id;
        }

        $this->force_id = true;

        $success = parent::add($autoDate, $nullValues);

        return $success;
    }

    /**
     * Return user instance from its e-mail (optionally check password).
     *
     * @param string $email e-mail
     * @param string $plaintextPassword Password is also checked if specified
     *
     * @return bool|User User instance
     */
    public function getByEmail($email, $plaintextPassword = null, $siteId = null )
    {
        if (!Validate::isEmail($email) || ($plaintextPassword && !Validate::isPasswd($plaintextPassword))) {
            return false;
        }

        if(\is_null($siteId)) {
            $siteId = PROX_SITE_ID;
        }
        
        $sql = new DbQuery();
        $sql->select('c.*');
        $sql->from('customer', 'c');
        $sql->where('c.`email` = \'' . pSQL($email) . '\'');

        if ($siteId) {
            $sql->where('c.`site_id` = \'' . (int) $siteId . '\'');
        }

        $result = Db::getInstance()->getRow($sql);
        if (!$result) {
            return false;
        }

        $crypto = new Hashing();

        $passwordHash = $result['password'];
        $shouldCheckPassword = !is_null($plaintextPassword);
        if ($shouldCheckPassword && !$crypto->checkHash($plaintextPassword, $passwordHash)) {
            return false;
        }
        
        $this->id = $result['customer_id'];
        foreach ($result as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }
        
        if ($shouldCheckPassword && !$crypto->isFirstHash($plaintextPassword, $passwordHash)) {
            $this->password = $crypto->hash($plaintextPassword);
            $this->update();
        }

        $this->getPhoneNumber();

        return $this;

    }

    /**
     * Get current writer stats
     *
     * @return array Order employee details
     */
    public function getEmployeePoolFull()
    {
        return Db::getInstance()->getRow('SELECT ep.* FROM ' . Db::prefix('employee_pool') . ' ep WHERE ep.`employee_pool_id` = ' . (int) $this->employee_pool_id );
    }
    
    /**
     * Check user information and return user validity.
     *
     * @return bool user validity
     */
    public function isLogged()
    {
        /* Customer is valid only if it can be load and if session is active */
        return $this->logged == 1 && (int) $this->id && Customer::checkPassword($this->id, $this->password);
    }

    /**
     * Soft logout, delete everything that links to the user
     *
     * @since 1.0.0
     */
    public function logout() {
        // remove auth_token
        if (isset(Application::getInstance()->cookie)) {
            Application::getInstance()->cookie->userLogout();
        }

        $this->logged = false;
    }

    /**
     * Check if customer password is the right one.
     *
     * @param int $customer_id Customer ID
     * @param string $passwordHash Hashed password
     *
     * @return bool result
     */
    public static function checkPassword($customer_id, $passwordHash)
    {
        if (!Validate::isUnsignedId($customer_id)) {
            die(Tools::displayError());
        }

        $cacheId = 'Customer::checkPassword' . (int) $customer_id . '-' . $passwordHash;
        if (!Cache::isStored($cacheId)) {
            $sql = new DbQuery();
            $sql->select('c.`customer_id`');
            $sql->from('customer', 'c');
            $sql->where('c.`customer_id` = ' . (int) $customer_id);
            $sql->where('c.`password` = \'' . pSQL($passwordHash) . '\'');

            $result = (bool) Db::getInstance(PROX_USE_SQL_SLAVE)->getValue($sql);

            Cache::store($cacheId, $result);

            return $result;
        }

        return Cache::retrieve($cacheId);
    }

    /**
     * used to cache customer's website.
     */
    protected $cacheCustomerSite = null;

    /**
     * Get customer's website.
     *
     * @return Site $cacheCustomerSite
     */
    public function getCustomerSite()
    {
        if (is_null($this->cacheCustomerSite)) {
            $this->cacheCustomerSite = new Site((int) $this->site_id);
        }

        return $this->cacheCustomerSite;
	}

    /**
     * Fill Reset password unique token with random sha1 and its validity date. For forgot password feature.
     */
    public function stampResetPasswordToken()
    {
        $salt = $this->id . '-' . $this->secure_key;
        $this->reset_password_token = sha1(time() . $salt);
        $this->reset_password_validity = date('Y-m-d H:i:s', strtotime('+' . 1440 . ' minutes'));
    }

    /**
     * Test if a reset password token is present and is recent enough to avoid creating a new one (in case of customer triggering the forgot password link too often).
     */
    public function hasRecentResetPasswordToken()
    {
        if (!$this->reset_password_token || $this->reset_password_token == '') {
            return false;
        }

        // TODO maybe use another 'recent' value for this test. For instance, equals password validity value.
        if (!$this->reset_password_validity || strtotime($this->reset_password_validity) < time()) {
            return false;
        }

        return true;
    }
    
    /**
     * Returns the valid reset password token if it validity date is > now().
     */
    public function getValidResetPasswordToken()
    {
        if (!$this->reset_password_token || $this->reset_password_token == '') {
            return false;
        }

        if (!$this->reset_password_validity || strtotime($this->reset_password_validity) < time()) {
            return false;
        }

        return $this->reset_password_token;
    }

    /**
     * Delete reset password token data.
     */
    public function removeResetPasswordToken()
    {
        $this->reset_password_token = null;
        $this->reset_password_validity = null;
    }

    /**
     * This method return the ID of the next customer.
     *
     * @since 1.0.0
     *
     * @return int
     */
    public function getNextCustomerId()
    {
        return Db::getInstance()->getValue('
            SELECT customer_id
            FROM ' . DB_PREFIX . 'customer
            WHERE customer_id > ' . (int) $this->id . ' 
            ORDER BY customer_id ASC');
    }
}