<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Exception;
use Proxim\Controller;
use Proxim\Coupon;
use Proxim\Validate;
use Proxim\Util\ArrayUtils;
use Proxim\Hook;
use Proxim\Order\Order;
use Proxim\Site\Site;
use Proxim\User\Customer;

class Coupons extends Controller {
    public function coupon() {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $do = ArrayUtils::get($params, 'do');

        switch( $do ) {
            case 'add':

                $site_id = ArrayUtils::get($payload, 'site_id');
                if(!$site_id) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select the Site ID"
                    ]);
                }
        
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid Site ID"
                    ]);
                }
        
                $coupon_code = ArrayUtils::get($payload, 'coupon_code');
                if (!$coupon_code || !Validate::isDiscountName($coupon_code)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid Coupon code"
                    ]);
                }

                if( Coupon::exists( $coupon_code, $site->id ) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "The Coupon code you entered already exists"
                    ]);
                }

                $coupon_type = (int) ArrayUtils::get($payload, 'coupon_type');

                $coupon_expiry = ArrayUtils::get($payload, 'coupon_expiry');
                $coupon_expiry = date('Y-m-d H:i:s', strtotime($coupon_expiry));
                if(!Validate::isDate($coupon_expiry)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid expiry date"
                    ]);
                }

                $coupon = new Coupon();
                $coupon->site_id = $site->id;
                $coupon->coupon_code = $coupon_code;
                $coupon->coupon_type = $coupon_type;
                $coupon->expiry = $coupon_expiry;
                $coupon->active = 1;

                switch( $coupon_type ) {
                    case Coupon::PERCENT:
                        $coupon_percent = (int) ArrayUtils::get($payload, 'coupon_percent');
                        if(!$coupon_percent) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the coupon percent"
                            ]);
                        }
        
                        $coupon->coupon_value = $coupon_percent;
                        break;
                    
                    case Coupon::AMOUNT:
                        $coupon_amount = (float) ArrayUtils::get($payload, 'coupon_amount');
                        if(!$coupon_amount) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the coupon amount"
                            ]);
                        }
        
                        $coupon->coupon_value = $coupon_amount;
                        break;

                    case Coupon::PRICE_PER_PAGE:
                        $coupon_cpp_amount = (float) ArrayUtils::get($payload, 'coupon_cpp_amount');
                        if(!$coupon_cpp_amount) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the coupon amount"
                            ]);
                        }
        
                        $coupon->coupon_value = $coupon_cpp_amount;
                        break;
        
                    case Coupon::FREE_UNIT:  
                        $service_type_id = (int) ArrayUtils::get($payload, 'service_type_id');
                        $coupon->service_type_id = $service_type_id;
                        break;
        
                    case Coupon::WRITER_CATEGORY:
                        $writer_category_id = (int) ArrayUtils::get($payload, 'writer_category_id');
                        $coupon->writer_category_id = $writer_category_id;
                        break;
                }

                $success = $coupon->add();

                Hook::exec('activitylog', [
                    'object' => 'coupon',
                    'object_id' => $coupon->id,
                    'event' => 'add_coupon'
                ]);
        
                if(!$success) {
                    return $this->response([
                        "error" => true
                    ]);
                }
                
                return $this->response([
                    'callback' => 'window.location.href="/coupons/edit/'.$coupon->id.'";'
                ]);
                break;
            
            case 'edit':
                $coupon_id = ArrayUtils::get($payload, 'coupon_id');

                $coupon = new Coupon( (int) $coupon_id );
                if(!Validate::isLoadedObject($coupon)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This coupon does not exist"
                    ]);
                }

                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $coupon_code = ArrayUtils::get($payload, 'coupon_code');
                $coupon_type = (int) ArrayUtils::get($payload, 'coupon_type');
                $user_id = (int) ArrayUtils::get($payload, 'user_id');
                $order_id = (int) ArrayUtils::get($payload, 'order_id');
                $coupon_expiry = ArrayUtils::get($payload, 'coupon_expiry');

                if(!$site_id) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select the Site ID"
                    ]);
                }

                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid Site ID"
                    ]);
                }

                $coupon_code = ArrayUtils::get($payload, 'coupon_code');
                if (!$coupon_code || !Validate::isDiscountName($coupon_code)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid Coupon code"
                    ]);
                }

                if( $coupon_code != $coupon->coupon_code && Coupon::exists( $coupon_code, $site->id ) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "The Coupon code you entered already exists"
                    ]);
                }

                $coupon->site_id = $site->id;
                $coupon->coupon_code = $coupon_code;
                $coupon->coupon_type = $coupon_type;
                $coupon->one_time = ArrayUtils::has($payload, 'one_time') ? true : false;
                $coupon->active = ArrayUtils::has($payload, 'active') ? true : false;

                if( $user_id ) {
                    $customer = new Customer( (int) $user_id );
                    if(!Validate::isLoadedObject($customer)) {
                        return $this->response([
                            "error" => true,
                            "message" => "The customer #$user_id does not exist"
                        ]);
                    }

                    $coupon->user_id = $customer->id;
                } else {
                    $coupon->user_id = null;
                }

                if($order_id) {
                    $order = new Order( (int) $order_id );
                    if(!Validate::isLoadedObject($order)) {
                        return $this->response([
                            "error" => true,
                            "message" => "The order $order_id does not exist"
                        ]);
                    }

                    $coupon->order_id = $order->id;
                } else {
                    $coupon->order_id = null;
                }

                $coupon_expiry = date('Y-m-d H:i:s', strtotime($coupon_expiry));
                if(!Validate::isDate($coupon_expiry)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid expiry date"
                    ]);
                }

                $coupon->expiry = $coupon_expiry;

                switch( $coupon_type ) {
                    case Coupon::PERCENT:
                        $coupon_percent = (int) ArrayUtils::get($payload, 'coupon_percent');
                        if(!$coupon_percent) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the coupon percent"
                            ]);
                        }

                        $coupon->coupon_value = $coupon_percent;
                        break;
                    
                    case Coupon::AMOUNT:
                        $coupon_amount = ArrayUtils::get($payload, 'coupon_amount');
                        if(!$coupon_amount) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the coupon amount"
                            ]);
                        }

                        $coupon->coupon_value = $coupon_amount;
                        break;

                    case Coupon::PRICE_PER_PAGE:
                        $coupon_cpp_amount = (float) ArrayUtils::get($payload, 'coupon_cpp_amount');
                        if(!$coupon_cpp_amount) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the coupon amount"
                            ]);
                        }
        
                        $coupon->coupon_value = $coupon_cpp_amount;
                        break;

                    case Coupon::FREE_UNIT:  
                        $service_type_id = (int) ArrayUtils::get($payload, 'service_type_id');
                        $coupon->service_type_id = $service_type_id;
                        break;

                    case Coupon::WRITER_CATEGORY:
                        $writer_category_id = (int) ArrayUtils::get($payload, 'writer_category_id');
                        $coupon->writer_category_id = $writer_category_id;
                        break;
                }

                $coupon->update();

                Hook::exec('activitylog', [
                    'object' => 'coupon',
                    'object_id' => $coupon->id,
                    'event' => 'edit_coupon'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "System settings have been updated"
                ]);
                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }
}