<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Proxim\Configuration;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Preference\Deadline;
use Proxim\Preference\Discipline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Preference\UploadType;
use Proxim\Validate;
use Proxim\Util\ArrayUtils;

class Customize extends Controller {
    public function index() {
        $app = $this->app;
        $user = $app->container->user;
        $smarty = $app->container->smarty;
        $params = $app->request->get();
        $payload = $app->request->post();

        $do = ArrayUtils::get($params, 'do');

        switch($do) {
            case 'addDeadline':
            case 'editDeadline':
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $deadline_id = (int) ArrayUtils::get($payload, 'deadline_id');
                $academic_level_id = (int) ArrayUtils::get($payload, 'academic_level_id');
                $name = ArrayUtils::get($payload, 'name');
                $price_per_page = (float) ArrayUtils::get($payload, 'price_per_page');
                $hrs_customer = (int) ArrayUtils::get($payload, 'hrs_customer');
                $hrs_writer = (int) ArrayUtils::get($payload, 'hrs_writer');
                $min_pages = (int) ArrayUtils::get($payload, 'min_pages');

                if(!Validate::isGenericName($name)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the name'
                    ]);	
                }

                if(!Validate::isPrice($price_per_page)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the price per page'
                    ]);	
                }

                if(!Validate::isInt($hrs_customer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Customer'
                    ]);	
                }

                if(!Validate::isInt($hrs_writer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Writer'
                    ]);	
                }

                $update = false;
                if($deadline_id) {
                    $deadline = new Deadline( $deadline_id );
                    if(Validate::isLoadedObject($deadline)) {
                        $update = true;
                    }
                }

                if(!$update) {
                    $deadline = new Deadline();
                    $deadline->site_id = $site_id;
                    $deadline->academic_level_id = $academic_level_id;
                }

                $deadline->name = $name;
                $deadline->price_per_page = $price_per_page;
                $deadline->hrs_customer = $hrs_customer;
                $deadline->hrs_writer = $hrs_writer;
                $deadline->min_pages = $min_pages;

                if($update) {
                    $deadline->update();

                    return $this->response([
                        'success' => true,
                        'message' => 'Deadline updated successfully'
                    ]);	
                } else {
                    $deadline->add();

		            return $this->modal('SUCCESS', "Added successfully", "Paper type added successfully");	
                }

                break;

            case 'addPaperType':
            case 'editPaperType':
                $paper_type_id = (int) ArrayUtils::get($payload, 'paper_type_id');
                $title = ArrayUtils::get($payload, 'title');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if($paper_type_id) {
                    $paper_type = new PaperType( $paper_type_id );
                    if(Validate::isLoadedObject($paper_type)) {
                        $update = true;
                    }
                }

                if(!$update) {
                    $paper_type = new PaperType();
                }

                $paper_type->title = $title;

                if($update) {
                    $paper_type->update();

                    return $this->response([
                        'success' => true,
                        'message' => 'Paper type updated successfully'
                    ]);	
                } else {
                    $paper_type->add();

		            return $this->modal('SUCCESS', "Added successfully", "Paper type added successfully");	
                }

                break;

            case 'addPaperFormat':
            case 'editPaperFormat':
                $paper_format_id = (int) ArrayUtils::get($payload, 'paper_format_id');
                $title = ArrayUtils::get($payload, 'title');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }
                
                $update = false;
                if($paper_format_id) {
                    $paper_format = new PaperFormat( $paper_format_id );
                    if(Validate::isLoadedObject($paper_format)) {
                        $update = true;
                    }
                }

                if(!$update) {
                    $paper_format = new PaperFormat();
                }

                $paper_format->title = $title;

                if($update) {
                    $paper_format->update();
                    return $this->response([
                        'success' => true,
                        'message' => 'Paper format updated successfully'
                    ]);	
                } else {
                    $paper_format->add();
                    return $this->modal('SUCCESS', "Added successfully", "Paper format added successfully");	
                }

                break;

            case 'addUploadType':
            case 'editUploadType':
                $upload_type_id = (int) ArrayUtils::get($payload, 'upload_type_id');
                $title = ArrayUtils::get($payload, 'title');
                $customer = ArrayUtils::has($payload, 'customer');
                $writer = ArrayUtils::has($payload, 'writer');
                $editor = ArrayUtils::has($payload, 'editor');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                if(!$customer && !$writer && !$editor) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select customer, writer or editor'
                    ]);	
                }

                $update = false;
                if($upload_type_id) {
                    $upload_type = new UploadType( $upload_type_id );
                    if(Validate::isLoadedObject($upload_type)) {
                        $update = true;
                    }
                }

                if(!$update) {
                    $upload_type = new UploadType();
                }

                $upload_type->title = $title;
                $upload_type->customer = $customer;
                $upload_type->writer = $writer;
                $upload_type->editor = $editor;

                if($update) {
                    $upload_type->update();

                    return $this->response([
                        'success' => true,
                        'message' => 'Upload type updated successfully'
                    ]);	
                } else {
                    $upload_type->add();

		            return $this->modal('SUCCESS', "Added successfully", "Upload type added successfully");	
                }

                break;

            case 'addDiscipline':
            case 'editDiscipline':
                $discipline_id = (int) ArrayUtils::get($payload, 'discipline_id');
                $discipline_group = (int) ArrayUtils::get($payload, 'discipline_group');
                $title = ArrayUtils::get($payload, 'title');
                $is_complex_assignment = (bool) ArrayUtils::has($payload, 'is_complex_assignment');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if($discipline_id) {
                    $discipline = new Discipline( $discipline_id );
                    if(Validate::isLoadedObject($discipline)) {
                        $update = true;
                    }
                }

                if(!$update) {
                    $discipline = new Discipline();
                }

                if($discipline_group) {
                    $discipline_group = new Discipline( $discipline_group );
                    if(Validate::isLoadedObject($discipline_group)) {
                        $discipline->group_id = $discipline_group->id;
                    }
                }

                $discipline->title = $title;
                $discipline->is_complex_assignment = $is_complex_assignment;

                if($update) {
                    $discipline->update();

                    return $this->response([
                        'success' => true,
                        'message' => 'Discipline updated successfully'
                    ]);	
                } else {
                    $discipline->add();

                    return $this->modal('SUCCESS', "Added successfully", "Discipline added successfully");	
                }

                break;
        }
    }

    public function survey() {
		$app = $this->app;
		$payload = $app->request->post();
		$do = ArrayUtils::get($payload, 'do', 'info');

		switch($do) {
			case 'info':
				$sql = new DbQuery();
				$sql->select('s.*');
				$sql->from('site', 's');
				$result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

				$sites = array();

				$sites[] = array(
					'site_id' => PROX_SITE_ID,
					'node' => 'admin',
					'name' => Configuration::get('SITE_NAME', PROX_SITE_ID),
					'email' => Configuration::get('SITE_EMAIL', PROX_SITE_ID),
					'domain' => Configuration::get('SITE_DOMAIN', PROX_SITE_ID),
				);

				foreach($result as $site) {
					$sites[] = array(
						'site_id' => $site['site_id'],
						'node' => 'client',
						'name' => $site['name'],
						'email' => $site['email'],
						'domain' => $site['domain'],
						'customers' => $site['customers'],
						'orders' => $site['orders'],
						'payments' => $site['payments']
					);
				}

				return $this->response([
					"success" => true,
					"sites" => $sites
				]);
				break;

			case 'zigby':
				Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("customer") . ' DROP `total_spent`');
				Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("customer") . ' DROP `last_activity`');
				Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("order") . ' DROP `parent_id`');
        		Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("order") . ' DROP `currency_to_usd_rate`');
				break;
		}
	}
}