<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Aws\S3\Exception\S3Exception;
use Aws\S3\S3Client;
use Db;
use ZipArchive;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Mail;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\Controller;
use Proxim\Order\Order;
use Proxim\Order\File;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\EmployeeApplication\EmployeeApplication;
use Proxim\Hook;
use Proxim\Uploader;
use Proxim\User\EmployeeFile;
use Proxim\Util\StringUtils;

class Files extends Controller {

    public function uploadFiles() {
        $app = $this->app;
        $payload = $app->request->post();
		$user = $app->user;

		$handle = ArrayUtils::get($payload, 'handle', 'order');
		$id = ArrayUtils::get($payload, 'id');
		$instanceId = ArrayUtils::get($payload, 'instanceId');

		switch($handle) {
			case 'order':
				$order = new Order( (int) $id );
				if(!Validate::isLoadedObject($order)) {
					return $this->response([
						"error" => true,
						"message" => "Could not upload file"
					]);	
				} 

				global $globals;
                $uploader = new Uploader();
				$uploader->setAcceptTypes($globals['extensions']);
				$uploader->setName('files');
				$uploader->setPrefix($order->id);
				$uploadedFile = $uploader->process()[0];

				if(isset($uploadedFile['error']) && !empty($uploadedFile['error'])) {
					return $this->response([
						"error" => true,
						"message" => $uploadedFile['error']
					]);	
				}

				try {
					$file = new File();
					$file->uploader_id = $user->id;
					$file->file_type_id = File::COMPLETE_PAPER;
					$file->name = $uploadedFile['name'];
					$file->source = $uploadedFile['source_name'];
					$file->size = $uploadedFile['size'];
					$file->extension = $uploadedFile['extension'];
					$success = $file->add(true);

					Hook::exec('activitylog', [
                        'object' => 'order_file',
                        'object_id' => $file->id,
                        'event' => 'upload_order_file'
                    ]);
		
					if (!$success) {
						return $this->response([
							"error" => true,
							"message" => "Could not upload file"
						]);	
					}
		
					$file = array(
						"id" => $file->id, 
						"name" => $file->source, 
						"newName" => $file->name, 
						"size" =>  $file->size, 
						"uploadTypeId" => null,
					);
					
					return $this->response([
						"success" => true,
						"file" => $file
					]);
		
				} catch (\Exception $e) {
					return $this->response([
						"error" => true,
						"message" => "Could not upload file"
					]);	
				}

				break;

			case 'plugin':
				$uploader = new Uploader();
				$uploader->setCloudStorage(false);
				$uploader->setAcceptTypes(['zip']);
				$uploader->setName('files');
				$uploadedFile = $uploader->process()[0];

				if(isset($uploadedFile['error']) && !empty($uploadedFile['error'])) {
					return $this->response([
						"error" => true,
						"message" => $uploadedFile['error']
					]);	
				}

				$file = array(
					"id" => $instanceId, 
					"name" => $uploadedFile['name'], 
					"newName" => $uploadedFile['source_name'], 
					"size" =>  $uploadedFile['size']
				);

				Hook::exec('activitylog', [
					'object' => 'plugin',
					'object_id' => $instanceId,
					'event' => 'upload_plugin'
				]);
				
				return $this->response([
					"success" => true,
					"file" => $file
				]);

				break;

			case 'application-essay':
			case 'application-sample':
			case 'application-degree':
			case 'application-cv':
				global $globals;
                $uploader = new Uploader();
				$uploader->setAcceptTypes($globals['extensions']);
				$uploader->setName('files');
				$uploadedFile = $uploader->process()[0];

				if(isset($uploadedFile['error']) && !empty($uploadedFile['error'])) {
					return $this->response([
						"error" => true,
						"message" => $uploadedFile['error']
					]);	
				} 

				try {
					$file = new EmployeeFile();
					$file->uploader_id = $user->id;
					$file->handle = $handle;
					$file->name = $uploadedFile['name'];
					$file->source = $uploadedFile['source_name'];
					$file->size = $uploadedFile['size'];
					$file->extension = $uploadedFile['extension'];
					$success = $file->add(true);

					Hook::exec('activitylog', [
                        'object' => 'application_file',
                        'object_id' => $file->id,
                        'event' => 'upload_application_file'
                    ]);
		
					if (!$success) {
						return $this->response([
							"error" => true,
							"message" => "Could not upload file"
						]);	
					}
		
					$file = array(
						"id" => $file->id, 
						"name" => $file->source, 
						"newName" => $file->name, 
						"size" =>  $file->size
					);
					
					return $this->response([
						"success" => true,
						"file" => $file
					]);
		
				} catch (\Exception $e) {
					return $this->response([
						"error" => true,
						"message" => "Could not upload file"
					]);	
				}
				break;

			default:
				$id = ArrayUtils::get($payload, 'id', $instanceId);
				global $globals;
                $uploader = new Uploader();
				$uploader->setAcceptTypes($globals['extensions']);
				$uploader->setName('files');
				$uploadedFile = $uploader->process()[0];

				if(isset($uploadedFile['error']) && !empty($uploadedFile['error'])) {
					return $this->response([
						"error" => true,
						"message" => $uploadedFile['error']
					]);	
				}

				$file = array(
					"id" => $instanceId, 
					"name" => $uploadedFile['name'], 
					"source" => $uploadedFile['source_name'], 
					"extension" => $uploadedFile['extension'],
					"size" =>  $uploadedFile['size']
				);

				Hook::exec('activitylog', [
					'object' => 'plugin',
					'object_id' => $instanceId,
					'event' => 'upload_plugin'
				]);

				Hook::exec('actionAfterFileUpload', [
					'handle' => $handle,
					'id' => $id,
					'file' => $file
				]);
				
				return $this->response([
					"success" => true,
					"file" => $file
				]);
				break;
		}
	}
	
	public function dataUpload() {
		$app = $this->app;
		$payload = $app->request->post();
		$user = $app->user;

		$type = ArrayUtils::get($payload, 'type');
		$multiple = ArrayUtils::has($payload, 'multiple', false);
		$localstorage = ArrayUtils::has($payload, 'localstorage', false);
		$handle = ArrayUtils::get($payload, 'handle');
		$id = ArrayUtils::get($payload, 'id');

		if ($multiple) {
			global $globals;
			$uploader = new Uploader();
			if($localstorage) {
				$uploader->setCloudStorage(false);
			}
			$uploader->setAcceptTypes($globals['extensions']);
			$uploader->setName('file');
			$uploadedFiles = $uploader->process();

			$return_files = array();

			foreach ($uploadedFiles as $uploadedFile) {
				$return_files[] = array(
					"id" => Tools::randomGen(10, 'NUMERIC'),
					"name" => $uploadedFile['name'], 
					"source" => $uploadedFile['source_name'], 
					"extension" => $uploadedFile['extension'],
					"size" =>  $uploadedFile['size']
				);

				Hook::exec('actionAfterFileUpload', [
					'handle' => $handle,
					'id' => $id,
					'file' => $uploadedFile
				]);
			}

			return $this->response([
				"files" => $return_files
			]);
		} else {
			global $globals;
			$uploader = new Uploader();
			if($localstorage) {
				$uploader->setCloudStorage(false);
			}
			$uploader->setAcceptTypes($globals['extensions']);
			$uploader->setName('file');
			$uploadedFile = $uploader->process();

			switch( $type ) {
				case 'photos':
					switch($handle) {
						case 'picture-user':
							$user->avatar = $uploadedFile['source_name'];
							$user->update();
							break;
					}

					return $this->response(
						array("file" => $uploadedFile['source_name'])
					);
					break;

				default:
					return $this->response(
						array("file" => $uploadedFile['source_name'])
					);
					break;
			}
		}
	}

    public function downloadFile( $fileId ) {
		$app = $this->app;
		$user = $app->user;

		if( !Configuration::get('INTERNAL_FILE_STORAGE') && !$user->isLogged()) {
			$this->setTemplate('404');
        	$this->display();
		}

		$file = new File( (int) $fileId ); 
		if (!Validate::isLoadedObject($file)) {
			$this->setTemplate('404');
        	$this->display();
		}

		$order = new Order( (int) $file->order_id );
		if (!Validate::isLoadedObject($order)) {
			$this->setTemplate('404');
        	$this->display();
        }

		Hook::exec('activitylog', [
			'object' => 'order_file',
			'object_id' => $file->id,
			'event' => 'file_download'
		]);
        
		$this->fileDownload( $file );
    }

	public function downloadEmployeeFile( $fileId ) {
		$app = $this->app;
		$user = $app->user;

		if(!$user->isLogged()) {
			$this->setTemplate('404');
        	$this->display();
		}

		$file = new EmployeeFile( (int) $fileId ); 
		if (!Validate::isLoadedObject($file)) {
			$this->setTemplate('404');
        	$this->display();
		}
		
		print_r($file);

		Hook::exec('activitylog', [
			'object' => 'employee_file',
			'object_id' => $file->id,
			'event' => 'file_download'
		]);
        
		$this->fileDownload( $file );
    }

    public function downloadAllAttachments( $orderId ) {
        $app = $this->app;
        $user = $app->user;

        $order = new Order( (int) $orderId );
        if(!Validate::isLoadedObject($order)) {
            $this->setTemplate('404');
        	$this->display();
        }

		Hook::exec('activitylog', [
			'object' => 'order_files',
			'object_id' => $order->id,
			'event' => 'order_files_download'
		]);

        $order_files = $order->getFiles();
        
        if( !extension_loaded('zip') ) {
            $this->setTemplate('404');
        	$this->display();
        }

        $zip = new ZipArchive(); // Load zip library 
        $zip_name = 'fid_' . $order->id . '_files.zip';
        if( $zip->open($zip_name, ZIPARCHIVE::CREATE)!==TRUE ) { 
            $this->setTemplate('404');
            $this->display();
        }

        foreach( $order_files as $file ) {
			$file = new File( (int) $file['file_id'] );
			if(Validate::isLoadedObject($file)) {
				$file_path = PROX_DIR_UPLOADS . $file->source;

				$fileName = $file->name;
				if($file->extension && !StringUtils::endsWith($fileName, $file->extension)) {
					$fileName .= "." . $file->extension;
				}
				
            	$zip->addFile( $file_path, $fileName );
			}
        }

        $zip->close();

        if(file_exists($zip_name)) {
            header('Content-type: application/zip');
            header('Content-Disposition: attachment; filename="'.$zip_name.'"');
            readfile($zip_name);
            unlink($zip_name);
        }
    }

    public function fileDownload( $file ) {
		$app = $this->app;
        $user = $app->user;

		$downloadFileName = $file->name;
		if($file->extension && !StringUtils::endsWith($downloadFileName, $file->extension)) {
			$downloadFileName .= "." . $file->extension;
		}

		$mime_types = array(
			"htm" => "text/html",
			"exe" => "application/octet-stream",
			"zip" => "application/zip",
			"doc" => "application/msword",
			"docx" => "application/msword",
			"jpg" => "image/jpg",
			"php" => "text/plain",
			"xls" => "application/vnd.ms-excel",
			"ppt" => "application/vnd.ms-powerpoint",
			"gif" => "image/gif",
			"pdf" => "application/pdf",
			"txt" => "text/plain",
			"html"=> "text/html",
			"png" => "image/png",
			"jpeg"=> "image/jpg",
			"js" => "application/x-javascript"
		);

		if( array_key_exists($file->extension, $mime_types) ) {
			$mime_type = $mime_types[$file->extension];
		} else {
			$mime_type = "application/force-download";
		}

		/** Amazon S3 File Download */
		if(Configuration::get('S3_ENABLED')) {
			$s3_region = Configuration::get('S3_REGION');
			$s3_key = Configuration::get('S3_KEY');
			$s3_secret = Configuration::get('S3_SECRET');
			$s3_bucket = Configuration::get('S3_BUCKET');
			
			try {
				$s3Client = S3Client::factory(array(
					'version'    => 'latest',
					'region'      => $s3_region,
					'credentials' => array(
						'key'    => $s3_key,
						'secret' => $s3_secret,
					)
				));

				// Generate a signed URL that expires in a specific time (e.g., 5 minutes)
				$signedUrl = $s3Client->getObjectUrl($s3_bucket, 'uploads/' . $file->source, '+5 minutes');

				// Set the appropriate headers to force the download
				header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
				header('Pragma: no-cache');
				header('Expires: 0');
				header('Content-Disposition: attachment; filename="'.$downloadFileName.'"');
				header('Content-Type: application/octet-stream');

				// Output the file content directly to the browser
				readfile($signedUrl);
				exit;

				// Get the object.
				$result = $s3Client->getObject([
					'Bucket' => $s3_bucket,
					'Key'    => 'uploads/' . $file->source
				]);

				header('Pragma: public');
				header("Expires: 0");
				header('Connection: Keep-Alive');
				header("Cache-Control: public, must-revalidate, post-check=0, pre-check=0");
				header('Content-Disposition: attachment; filename="'.$downloadFileName.'"');
				header("Content-Transfer-Encoding: binary");
				header('Accept-Ranges: bytes');
				header("Content-Type: {$result['ContentType']}");
				echo $result['Body'];
			} catch (S3Exception $e) {
				$this->setTemplate('404');
				$this->display();
			}
		} else {
			$file_path = PROX_DIR_UPLOADS . $file->source;

			$downloadFile = @fopen($file_path,"rb");
			if (!$downloadFile) {
				$this->setTemplate('404');
				$this->display();
			}

			Hook::exec('activitylog', [
				'object' => 'order_file',
				'object_id' => $file->id,
				'event' => 'file_download'
			]);

			$downloadFileName = $file->name;
			if($file->extension && !StringUtils::endsWith($downloadFileName, $file->extension)) {
				$downloadFileName .= "." . $file->extension;
			}

			$downloadFileSize = filesize($file_path);

			@ob_end_clean();
			if( ini_get('zlib.output_compression') ) {
				ini_set('zlib.output_compression', 'Off');
			}

			header('Pragma: public');
			header("Expires: 0");
			header('Connection: Keep-Alive');
			header("Cache-Control: public, must-revalidate, post-check=0, pre-check=0");
			header('Content-Type: ' . $mime_type);
			header('Content-Disposition: attachment; filename="'.$downloadFileName.'"');
			header("Content-Transfer-Encoding: binary");
			header('Accept-Ranges: bytes');

			if( isset($_SERVER['HTTP_RANGE']) ) {
				list($a, $range) = explode("=",$_SERVER['HTTP_RANGE'],2);
				list($range) = explode(",",$range,2);
				list($range, $range_end) = explode("-", $range);
				$range=intval($range);
				if(!$range_end) {
					$range_end=$downloadFileSize-1;
				} else {
					$range_end=intval($range_end);
				}

				$new_length = $range_end-$range+1;
				header("HTTP/1.1 206 Partial Content");
				header("Content-Length: $new_length");
				header("Content-Range: bytes $range-$range_end/$downloadFileSize");
			} else {
				$new_length = $downloadFileSize;
				header("Content-Length: " . $downloadFileSize);
			}

			$chunksize = 1*(1024*1024);
			$bytes_sent = 0;

			if(isset($_SERVER['HTTP_RANGE'])) {
				fseek($downloadFile, $range);
			}

			while(
				!feof($downloadFile) && 
				(!connection_aborted()) && 
				($bytes_sent < $new_length) 
			) {
				$buffer = fread($downloadFile, $chunksize);
				echo($buffer);
				flush();
				$bytes_sent += strlen($buffer);
			}

			@fclose($downloadFile);
			return;
		}
	}
}