<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Proxim\Configuration;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Mail;
use Proxim\MailTemplate;
use Proxim\Site\Site;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class Mailing extends Controller {
    public function mailing() {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
            return $this->response([
                "error" => true,
                "message" => "You don't have the right permission to access this"
            ]);
        }

        $do = ArrayUtils::get($params, 'do');

        switch( $do ) {
            case 'add_mail':
                $name = ArrayUtils::get($payload, 'name');
                if(!$name || !Validate::isName($name)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must enter a valid name"
                    ]);
                }

                $email = ArrayUtils::get($payload, 'email');
                if(!$email || !Validate::isEmail($email)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must enter a valid email"
                    ]);
                }

                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must select a valid site ID"
                    ]);
                }

                $sql = new DbQuery();
                $sql->select('m.`mail_id`');
                $sql->from('mailing_list', 'm');
                $sql->where('m.`email` = \'' . pSQL($email) . '\'');
                $sql->where('m.`site_id` = ' . (int) $site->id );
                $exists = (bool) Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
                
                if( $exists ) {
                    return $this->response([
                        "error" => true,
                        "message" => "This email already exists"
                    ]);
                }

                $success = Db::getInstance()->insert(
                    'mailing_list',
                    array(
                        'site_id' => $site->id,
                        'name' => $name,
                        'email' => $email,
                        'date_add' => DateUtils::now()
                    )
                );

                if(!$success) {
                    return $this->response([
                        "error" => true,
                        "message" => "Could not add this to mailing list"
                    ]);
                }

                $mailID = Db::getInstance()->Insert_ID();

                Hook::exec('activitylog', [
                    'object' => 'mailing_list',
                    'object_id' => $mailID,
                    'event' => 'add_mailing_list'
                ]);

                return $this->response([
                    'callback' => 'window.location.href="/mailing/edit/'.$mailID.'";'
                ]);
                break;

            case 'edit_mail':
                $mail_id = ArrayUtils::get($payload, 'mail_id');

                $sql = new DbQuery();
                $sql->select('m.*');
                $sql->from('mailing_list', 'm');
                $sql->where('mail_id = ' . (int) $mail_id );
                $mail = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow( $sql );

                if(!$mail) {
                    return $this->response([
                        "error" => true,
                        "message" => "This mail does not exist"
                    ]);
                }

                $name = ArrayUtils::get($payload, 'name');
                if(!$name || !Validate::isName($name)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must enter a valid name"
                    ]);
                }

                $email = ArrayUtils::get($payload, 'email');
                if(!$email || !Validate::isEmail($email)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must enter a valid email"
                    ]);
                }

                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must select a valid site ID"
                    ]);
                }

                if ( $mail['email'] != $email || $mail['site_id'] != $site->id ) {
                    $sql = new DbQuery();
                    $sql->select('m.`mail_id`');
                    $sql->from('mailing_list', 'm');
                    $sql->where('m.`email` = \'' . pSQL($email) . '\'');
                    $sql->where('m.`site_id` = ' . (int) $site->id );
                    $exists = (bool) Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
                    
                    if( $exists ) {
                        return $this->response([
                            "error" => true,
                            "message" => "This email already exists"
                        ]);
                    }
                }

                $success = Db::getInstance()->update(
                    'mailing_list',
                    array(
                        'site_id' => $site->id,
                        'name' => $name,
                        'email' => $email,
                        'date_add' => DateUtils::now()
                    ),
                    'mail_id = ' . (int) $mail['mail_id']
                );

                Hook::exec('activitylog', [
                    'object' => 'mailing_list',
                    'object_id' => $mail['mail_id'],
                    'event' => 'update_mailing_list'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "Email has been updated successfully"
                ]);
                break;

            case 'send_mail':
                $subject = ArrayUtils::get($payload, 'subject');
                if(!$subject || !Validate::isMailSubject($subject)) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must enter the email subject"
                    ]);
                }

                $message = ArrayUtils::get($payload, 'message'); 
                if(!$message) {
                    return $this->response([
                        "error" => true,
                        "message" => "You must enter the message subject"
                    ]);
                }

                $test_email = ArrayUtils::has($payload, 'test_email') ? true : false;
                $recipient = (int) ArrayUtils::get($payload, 'recipient');

                if( $test_email ) {
                    $mailParams = array(
                        'site_name' => Configuration::get('SITE_NAME'),
                        'email' => Configuration::get('SITE_EMAIL'),
                        'message' => $message
                    );

                    try {
                        Mail::Send(
                            'news_letter',
                            $subject,
                            $mailParams,
                            Configuration::get('SITE_EMAIL'),
                            Configuration::get('SITE_NAME')
                        );
                    } catch( \Exception $e ) {

                    }

                    return $this->response([
                        "success" => true,
                        "message" => "Email has been sent successfully"
                    ]);
                }

                switch( $recipient ) {

                    case '1':
                        $customer_ids = ArrayUtils::get($payload, 'customer_ids');
                        if(!$customer_ids) {
                            return $this->response([
                                "error" => true,
                                "message" => "Select the recepient"
                            ]);
                        }

                        foreach( $customer_ids as $customer_id ) {
                            $customer = new Customer( (int) $customer_id );
                            if(Validate::isLoadedObject($customer)) {
                                $mailParams = array(
                                    'username' => $customer->name,
                                    'email' => $customer->email,
                                    'message' => $message
                                );

                                try {
                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $customer->email,
                                        $customer->name,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        $customer->site_id
                                    );	
                                } catch( \Exception $e ) {
                                    
                                }
                                
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);

                        break;

                    case '2':
                        $writer_ids = ArrayUtils::get($payload, 'writer_ids', array());
                        if(!$writer_ids) {
                            return $this->response([
                                "error" => true,
                                "message" => "Select the recepient"
                            ]);
                        }

                        foreach( $writer_ids as $writer_id ) {
                            $writer = new Employee( (int) $writer_id );
                            if(Validate::isLoadedObject($writer)) {
                                $mailParams = array(
                                    'name' => $writer->first_name . " " . $writer->last_name,
                                    'email' => $writer->email,
                                    'message' => $message
                                );
                                
                                try {
                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $writer->email,
                                        $writer->first_name . " " . $writer->last_name
                                    );
                                } catch( \Exception $e ) {
                                    
                                }
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);
                        break;

                    case '3':
                        $mail_ids = ArrayUtils::get($payload, 'mail_ids', array());
                        if(!$mail_ids) {
                            return $this->response([
                                "error" => true,
                                "message" => "Select the recepient"
                            ]);
                        }

                        foreach( $mail_ids as $mail_id ) {
                            $sql = new DbQuery();
                            $sql->select('m.*');
                            $sql->from('mailing_list', 'm');
                            $sql->where('mail_id = ' . (int) $mail_id );
                            $mail = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow( $sql );

                            if($mail) {
                                $mailParams = array(
                                    'name' => $mail['name'],
                                    'email' => $mail['email'],
                                    'message' => $message
                                );

                                try {
                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $mail['email'],
                                        $mail['name'],
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        $mail['site_id']
                                    );
                                } catch( \Exception $e ) {
                                    
                                }
                                	
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);
                        break;

                    case '4':
                        $sql = new DbQuery();
                        $sql->select('c.customer_id, c.name, c.email, s.site_id');
                        $sql->from('customer', 'c');
                        $sql->innerJoin('site', 's', 's.site_id = c.site_id');
                        $sql->where('is_subscribed = 1');
                        $sql->where('is_anonymous = 0');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

                        foreach( $result as $customer ) {
                            if(Validate::isEmail($customer['email'])) {
                                $mailParams = array(
                                    'username' => $customer['name'],
                                    'email' => $customer['email'],
                                    'message' => $message
                                );

                                try {
                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $customer['email'],
                                        !empty($customer['name']) ? $customer['name'] : null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        $customer['site_id']
                                    );
                                } catch( \Exception $e ) {
                                    
                                }
                                	
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);
                        break;

                    case '5':
                        $sql = new DbQuery();
                        $sql->select('e.employee_id');
                        $sql->from('employee', 'e');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );
                        
                        foreach( $result as $writer ) {
                            $writer = new Employee( (int) $writer['employee_id'] );
                            if(Validate::isLoadedObject($writer)) {
                                $mailParams = array(
                                    'name' => $writer->first_name . " " . $writer->last_name,
                                    'email' => $writer->email,
                                    'message' => $message
                                );

                                try {
                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $writer->email,
                                        $writer->first_name . " " . $writer->last_name
                                    );	
                                } catch( \Exception $e ) {
                                    
                                }
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);

                        break;

                    case '6':
                        $sql = new DbQuery();
                        $sql->select('m.*');
                        $sql->from('mailing_list', 'm');
                        $mails = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );
                        
                        foreach( $mails as $mail ) {
                            if($mail) {
                                $mailParams = array(
                                    'name' => $mail['name'],
                                    'email' => $mail['email'],
                                    'message' => $message
                                );

                                try {

                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $mail['email'],
                                        $mail['name'],
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        $mail['site_id']
                                    );	

                                } catch( \Exception $e ) {
                                    
                                }
                                
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);

                        break;
                    
                    case '7':   
                        $site_ids = ArrayUtils::get($payload, 'site_ids', array());
                        if(!$site_ids) {
                            return $this->response([
                                "error" => true,
                                "message" => "Select at least one site"
                            ]);
                        }

                        $sql = new DbQuery();
                        $sql->select('c.customer_id');
                        $sql->from('customer', 'c');
                        $sql->where('site_id IN (' . implode(',', $site_ids) . ')' );
                        $sql->where('is_subscribed = 1');
                        $sql->where('is_anonymous = 0');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

                        foreach( $result as $customer ) {
                            $customer = new Customer( (int) $customer['customer_id'] );
                            if(Validate::isLoadedObject($customer)) {
                                $mailParams = array(
                                    'username' => $customer->name,
                                    'email' => $customer->email,
                                    'message' => $message
                                );

                                try {
                                    Mail::Send(
                                        'news_letter',
                                        $subject,
                                        $mailParams,
                                        $customer->email,
                                        $customer->name,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        null,
                                        $customer->site_id
                                    );
                                } catch( \Exception $e ) {
                                    
                                }
                                	
                            }
                        }

                        return $this->response([
                            "success" => true,
                            "message" => "Email has been sent successfully"
                        ]);
                        break;
                }
                
                break;

            case 'add_template':
                $template = ArrayUtils::get($payload, 'template');

                if(!$template) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select the mail template"
                    ]);
                }

                $mailTemplate = new MailTemplate();
                $mailTemplate->template = $template;
                $mailTemplate->add();

                return $this->response([
                    'callback' => 'window.location.href="/mailing/templates/edit?template='.$mailTemplate->template.'";'
                ]);
                break;

            case 'edit_template':
                $template_id = ArrayUtils::get($payload, 'template_id');
                $name = ArrayUtils::get($payload, 'name');
                $subject = ArrayUtils::get($payload, 'subject');
                $message = ArrayUtils::get($payload, 'message');
                $from_name = ArrayUtils::get($payload, 'from_name');
                $plaintext = (bool) ArrayUtils::has($payload, 'plaintext');
                $is_disabled = (bool) ArrayUtils::has($payload, 'is_disabled');

                $mailTemplate = new MailTemplate( (int) $template_id );
                if(!Validate::isLoadedObject($mailTemplate)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This template does not exist"
                    ]);
                }

                $mailTemplate->name = $name;
                $mailTemplate->subject = $subject;
                $mailTemplate->from_name = $from_name;
                $mailTemplate->plaintext = $plaintext;
                $mailTemplate->message = $message;
                $mailTemplate->is_disabled = $is_disabled;
                $mailTemplate->update();

                return $this->response([
                    "success" => true,
                    "message" => "Template settings updated successfully"
                ]);
                break;
        }
    }
}