<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Mail;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Hook;
use Proxim\MailTemplate;
use Proxim\Preference\WritingPrompt;
use Proxim\User\EmployeeFile;

class Onboarding extends Controller {
    public function application( $action = '' ) {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        switch($action) {
            case 'additional-info':
                $available_247 = ArrayUtils::has($payload, 'available_247');
                $available_urgent_orders = ArrayUtils::has($payload, 'available_urgent_orders');
                $about_me = ArrayUtils::get($payload, 'about_me');
                $facebook_profile = ArrayUtils::get($payload, 'facebook_profile');
                $linkedin_profile = ArrayUtils::get($payload, 'linkedin_profile');
                $referrer = ArrayUtils::get($payload, 'referrer');

                if(!$about_me) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please write something about yourself"
                    ]);
                }

                if(!$referrer) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select how you heard about our company"
                    ]);
                }

                $user->available_247 = (bool) $available_247;
                $user->available_urgent_orders = (bool) $available_urgent_orders;
                $user->about = $about_me;
                $user->facebook_profile = $facebook_profile;
                $user->linkedin_profile = $linkedin_profile;
                $user->referrer = $referrer;
                $user->update();

                $return['callback'] = "window.proxim.eventEmitter.emit('applicantAddInfoSuccess');";
                return $this->response($return);
                break;

            case 'personal-details':
                $first_name = ArrayUtils::get($payload, 'first_name');
                $middle_name = ArrayUtils::get($payload, 'middle_name');
                $last_name = ArrayUtils::get($payload, 'last_name');
                $gender = ArrayUtils::get($payload, 'gender');
                $phone = ArrayUtils::get($payload, 'phone');
                $date_of_birth = ArrayUtils::get($payload, 'date_of_birth');
                $country = ArrayUtils::get($payload, 'country');
                $city = ArrayUtils::get($payload, 'city');
                $timezone = ArrayUtils::get($payload, 'timezone');

                if ( !Validate::isName($first_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid first name"
                    ]);
                }
        
                if ( Tools::strlen($first_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your first name must be less than 20 characters long. Please try another"
                    ]);
                }
        
                if ( Tools::strlen($first_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your first name must be at least 3 characters long. Please try another"
                    ]);
                }
        
                if ( !Validate::isName($middle_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid middle name"
                    ]);
                }
        
                if ( Tools::strlen($middle_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your middle name must be less than 20 characters long. Please try another"
                    ]);
                }
        
                if ( Tools::strlen($middle_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your middle name must be at least 3 characters long. Please try another"
                    ]);
                }
        
                if ( !Validate::isName($last_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid last name"
                    ]);
                }
        
                if ( Tools::strlen($last_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your last name must be less than 20 characters long. Please try another"
                    ]);
                }

                if ( Tools::strlen($last_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your last name must be at least 3 characters long. Please try another"
                    ]);
                }
        
                if(!$phone || !Validate::isPhoneNumber($phone)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid phone number"
                    ]);
                }

                if(!$date_of_birth || !Validate::isDate($date_of_birth)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter your date of birth"
                    ]);
                }

                if(!$city || !Validate::isCityName($city)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter your city"
                    ]);
                }

                if(!$country || !Validate::isCountryName($country)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your county"
                    ]);
                }

                $user->first_name = $first_name; 
                $user->middle_name = $middle_name; 
                $user->last_name = $last_name; 
                $user->phone = $phone;
                $user->date_of_birth = $date_of_birth;
                $user->country = $country;
                $user->city = $city;
                $user->gender = $gender;
                $user->timezone = $timezone;
                $user->update();
                
                $return['callback'] = "window.proxim.eventEmitter.emit('applicantPersonalDetailsSuccess');";
                return $this->response($return);
                break;

            case 'qualifications':
                $native_language = ArrayUtils::get($payload, 'native_language');
                $other_languages = ArrayUtils::get($payload, 'other_languages', []);
                $highest_academic_degree = ArrayUtils::get($payload, 'highest_academic_degree');
                $academic_degree_major = ArrayUtils::get($payload, 'academic_degree_major');
                $proficient_disciplines = ArrayUtils::get($payload, 'proficient_disciplines', []);
                $software = ArrayUtils::get($payload, 'software', []);

                if(!$native_language) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your native language"
                    ]);
                }

                if(!$highest_academic_degree) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your highest academic degree"
                    ]);
                }

                if(!$academic_degree_major) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter your degree major"
                    ]);
                }

                if(!$proficient_disciplines && !is_array($proficient_disciplines)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the disciplines you are proficient in"
                    ]);
                }

                if(!$software && !is_array($software)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please indicate software you have access to and/or programming language you are expert in"
                    ]);
                }

                $user->native_language = $native_language;
                $user->other_languages = implode(',', $other_languages);
                $user->highest_academic_degree = $highest_academic_degree;
                $user->academic_degree_major = $academic_degree_major;
                $user->proficient_disciplines = implode(',', $proficient_disciplines);
                $user->software = implode(',', $software);
                $user->is_profile_completed = true;
                $user->update();

                return $this->modal("#modal-application-success"); 
                break;

            case 'guide':
                $user->is_guide_complete = true;
                $user->guide_completed_at = DateUtils::now();
                $user->update();

                return $this->modal("SUCCESS", "Amazing!", "You are now one step closer to joining our team."); 
                break;

            case 'writing-prompt':
                $writing_prompt_id = (int) ArrayUtils::get($payload, 'writing_prompt_id');
                $answer = ArrayUtils::get($payload, 'answer');

                $writing_prompt = new WritingPrompt( (int) $writing_prompt_id );
                if(!Validate::isLoadedObject($writing_prompt)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please choose one of the two questions"
                    ]);
                }

                if(!$answer || !Validate::isCleanHtml($answer)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please type in your answer"
                    ]);
                }

                if(str_word_count($answer) < 100) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your answer should be at least 100 words"
                    ]);
                }

                if(str_word_count($answer) > 150) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your answer should be less than 150 words"
                    ]);
                }

                $user->writing_prompt_question = $writing_prompt->title;
                $user->writing_prompt_answer = $answer;
                $user->is_writingprompt_complete = 1;
                $user->writingprompt_completed_at = DateUtils::now();
                $user->update();

                return $this->modal("SUCCESS", "Successful", "We have successfully received your answer."); 
                break;

            case 'upload':
                $handle = ArrayUtils::get($payload, 'handle');
                $files = ArrayUtils::get($payload, 'files', []);

                if(empty($files)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please upload a file"
                    ]);
                }

                foreach($files as $file) {
                    $writerFile = new EmployeeFile( (int) $file['id']);
                    if(Validate::isLoadedObject($writerFile)) {
                        $writerFile->employee_id = $user->id;
                        $writerFile->update();
                    }
                }

                switch($handle) {
                    case 'application-essay':
                        $user->is_essay_complete = true;
                        $user->essay_completed_at = DateUtils::now();
                        $user->update();

                        return $this->modal("SUCCESS", "Upload Successful", "Your essay was successfully submitted."); 
                        break;

                    case 'application-sample':
                        $user->is_samples_complete = true;
                        $user->samples_completed_at = DateUtils::now();
                        $user->update();

                        $word = 'sample';
                        $words = count($files) == 1 ? $word : $word  . 's';

                        return $this->modal("SUCCESS", "Upload Successful", "Your " . $words . " was successfully submitted."); 
                        break;

                    case 'application-cv':
                        $user->is_cv_complete = true;
                        $user->cv_completed_at = DateUtils::now();
                        $user->update();

                        return $this->modal("SUCCESS", "Upload Successful", "Your CV was successfully submitted."); 
                        break;

                    case 'application-degree':
                        $user->is_degree_cert_complete = true;
                        $user->degree_cert_completed_at = DateUtils::now();

                        $user->is_application_completed = true;
                        $user->update();

                        return $this->modal("SUCCESS", "Upload Successful", "Your degree certificate was uploaded successfully submitted."); 
                        break;
                }

                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }
}