<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use DateTime;
use DateTimeZone;
use Db;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Mail;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Site\Site;
use Proxim\Pager;
use Proxim\Notification;
use Proxim\PriceCalculator;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\MailTemplate;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Order\Order;
use Proxim\Order\Message;
use Proxim\Order\Bid;
use Proxim\Order\AdditionalService;
use Proxim\Order\DeliverySchedule;
use Proxim\Order\Revision;
use Proxim\Order\File;
use Proxim\Order\OrderEmployee;
use Proxim\Order\Rating;
use Proxim\Preference\ArticleWriting\ContentFeel;
use Proxim\Preference\ArticleWriting\EnglishType;
use Proxim\Preference\ArticleWriting\TargetAudience;
use Proxim\Preference\ArticleWriting\WritingStyle;
use Proxim\Preference\ArticleWriting\WritingTone;
use Proxim\Preference\Deadline;
use Proxim\Preference\Discipline;
use Proxim\Preference\ItemPricing;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Preference\UploadType;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;
use Proxim\Util\StringUtils;

class Orders extends Controller {

    public function showCustomersModal() {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;

        $customersSql = new DbQuery();
        $customersSql->select('c.customer_id, c.name, c.email, c.site_id');
        $customersSql->from('customer', 'c');
        $customers = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $customersSql );

        $smarty->assign([
            'base_uri' => $app->base_uri,
            'customers' => $customers
        ]);

        /* return */
        $return['template'] = $smarty->fetch("ajax.create-order.tpl");
        $return['callback'] = "$('#modal').modal('show'); $('.modal-content:last').html(response.template);";
        
        return $this->response($return);
    }

    public function createOrder(){
        global $globals;
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        $serviceType = (int) ArrayUtils::get($payload, 'serviceType');

        $new_order = true;
        if(ArrayUtils::has($payload, 'order_id')) {
            $order_id = ArrayUtils::get($payload, 'order_id');
            $order = new Order( (int) $order_id );
            if(!Validate::isLoadedObject($order)) {
                return $this->response([
                    'error' => true,
                    'message' => "This order does not exist"
                ]);
            }
            $new_order = false;
        } else {
            $order = new Order();
            $order->service_type = (int) $serviceType;
        }

        $changeDeadline = ArrayUtils::has($payload, 'changeDeadline');

        if(ArrayUtils::has($payload, 'customerId')) {
            $customerId = ArrayUtils::get($payload, 'customerId');

            $customer = new Customer( (int) $customerId );
            if(!Validate::isLoadedObject($customer)) {
                return $this->response([
                    "error" => true,
                    "message" => "Customer <strong>#$customerId</strong> does not exist",
                ]);
            }

            $order->customer_id = $customer->id;
            $order->site_id = $customer->site_id;
        } 

        switch($serviceType) {
            case Order::PROGRAMMING:
            case Order::CALCULATIONS:
                $payload = $serviceType == Order::PROGRAMMING ? ArrayUtils::get($payload, 'programming') : ArrayUtils::get($payload, 'calculations');
                
                $software = ArrayUtils::get($payload, 'software');
                $taskSizeId = (int) ArrayUtils::get($payload, 'taskSizeId');
                $paper_details = ArrayUtils::get($payload, 'paperDetails');

                $itemPricing = new ItemPricing( $taskSizeId );
                if(!Validate::isLoadedObject($itemPricing)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid task size"
                    ]);
                }

                /** Topic category */
                $topic_cat_id = (int) ArrayUtils::get($payload, 'topicCatId');
                $discipline = new Discipline( $topic_cat_id );
                if(!Validate::isLoadedObject($discipline)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid discipline"
                    ]);
                } 

                $isComplexAssignment = $discipline->is_complex_assignment;
                $topic_cat_option = $discipline->title;

                if($new_order || $changeDeadline) {
                    $tariff_id = (int) ArrayUtils::get($payload, 'tariffId');
                    $deadline = new Deadline( (int) $tariff_id );
                    if(!Validate::isLoadedObject($deadline)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Select a valid deadline",
                        ]);
                    } 
        
                    $order->tariff_id = $tariff_id;
                    $order->price_modifier = $deadline->price_modifier;
                    $order->hrs_customer = $deadline->hrs_customer;
                    $order->hrs_writer = $deadline->hrs_writer;
                    $order->customer_deadline = DateUtils::date(time() + ($deadline->hrs_customer*60*60));
                    $order->writer_deadline = DateUtils::date(time() + ($deadline->hrs_writer*60*60));
                }
        
                $serviceType = ArrayUtils::get($globals['serviceTypes'], $serviceType);

                $order->title = $serviceType['title'];
                $order->software = $software;
                $order->task_item_price = $itemPricing->price;
                $order->task_size_id = $itemPricing->task_size_id;
                $order->topic_category_id = $topic_cat_id;
                $order->topic_category_option = $topic_cat_option;
                $order->is_complex_assignment = $isComplexAssignment;
                $order->paper_details = $paper_details;

                $calc = $order->calculate();
                $baseCost = ArrayUtils::get($calc, 'totalCost');
                $servicesById = ArrayUtils::get($calc, 'servicesById');

                $order->raw_cost = $baseCost;
                $order->total = $baseCost; 
                $order->writer_pay = ArrayUtils::get($calc, 'writerCost');
                $order->editor_pay = ArrayUtils::get($calc, 'editorCost');
                break;

            case Order::ARTICLE_WRITING:
                $payload = ArrayUtils::get($payload, 'article');

                $title = ArrayUtils::get($payload, 'title');
                $keywords = ArrayUtils::get($payload, 'keywords');
                $words = (int) ArrayUtils::get($payload, 'words');
                $paper_details = ArrayUtils::get($payload, 'paperDetails');
                $writerCategoryId = (int) ArrayUtils::get($payload, 'writerCategory');

                $writerPreferences = ArrayUtils::get($globals, 'writerPreferences');
        
                $writerPercent = 0;
                if (ArrayUtils::has($writerPreferences, $writerCategoryId)) {
                    $writerPreference = ArrayUtils::get($writerPreferences, $writerCategoryId);
                    $writerPercent = ArrayUtils::get($writerPreference, 'percent', 0);
                }

                if(!$title) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter the title",
                    ]);
                } 

                $typeOfContentId = (int) ArrayUtils::get($payload, 'typeOfContent');
                $typeOfContent = new PaperType( $typeOfContentId );
                if(!Validate::isLoadedObject($typeOfContent)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid content type"
                    ]);
                } 

                $categoryId = (int) ArrayUtils::get($payload, 'categoryId');
                $discipline = new Discipline( $categoryId );
                if(!Validate::isLoadedObject($discipline)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid category"
                    ]);
                } 

                if ( $words < 50) {
                    return $this->response([
                        "error" => true,
                        "message" => "You need to order at least 50 words"
                    ]);
                }

                $englishTypeId = (int) ArrayUtils::get($payload, 'englishTypeId');
                $englishType = new EnglishType($englishTypeId);
                if(!Validate::isLoadedObject($englishType)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the english type"
                    ]);
                }

                $targetAudienceId = (int) ArrayUtils::get($payload, 'targetAudienceId');
                $targetAudience = new TargetAudience($targetAudienceId);
                if(!Validate::isLoadedObject($targetAudience)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the target audience"
                    ]);
                }

                $writingStyleId = ArrayUtils::get($payload, 'writingStyleId');
                $writingStyle = new WritingStyle($writingStyleId);
                if(!Validate::isLoadedObject($writingStyle)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the writing style"
                    ]);
                }

                $writingToneId = ArrayUtils::get($payload, 'writingToneId');
                $writingTone = new WritingTone($writingToneId);
                if(!Validate::isLoadedObject($writingTone)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the writing tone"
                    ]);
                }

                $contentFeelId = ArrayUtils::get($payload, 'contentFeelId');
                $contentFeel = new ContentFeel($contentFeelId);
                if(!Validate::isLoadedObject($contentFeel)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the content feel"
                    ]);
                }

                if($new_order || $changeDeadline) {
                    $tariff_id = (int) ArrayUtils::get($payload, 'tariffId');
                    $deadline = new Deadline( (int) $tariff_id );
                    if(!Validate::isLoadedObject($deadline)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Select a valid deadline",
                        ]);
                    } 
        
                    $order->tariff_id = $tariff_id;
                    $order->price_per_word = $deadline->price_per_page;
                    $order->hrs_customer = $deadline->hrs_customer;
                    $order->hrs_writer = $deadline->hrs_writer;
                    $order->customer_deadline = DateUtils::date(time() + ($deadline->hrs_customer*60*60));
                    $order->writer_deadline = DateUtils::date(time() + ($deadline->hrs_writer*60*60));
                }

                $order->title = $title;
                $order->keywords = $keywords;
                $order->paper_type_id = $typeOfContent->id;
                $order->paper_type_option = $typeOfContent->title;
                $order->topic_category_id = $discipline->id;
                $order->topic_category_option = $discipline->title;
                $order->is_complex_assignment = $discipline->is_complex_assignment;
                $order->paper_details = $paper_details;
                $order->words = $words;
                $order->writer_category_id = $writerCategoryId;
                $order->writer_percent = $writerPercent;
                $order->english_type_id = $englishType->id; 
                $order->target_audience_id = $targetAudience->id;
                $order->writing_style_id = $writingStyle->id;
                $order->writing_tone_id = $writingTone->id;
                $order->content_feel_id = $contentFeel->id;

                $calc = $order->calculate();
                $baseCost = ArrayUtils::get($calc, 'totalCost');
                $servicesById = ArrayUtils::get($calc, 'servicesById');

                $order->raw_cost = $baseCost;
                $order->total = $baseCost; 
                $order->writer_pay = ArrayUtils::get($calc, 'writerCost');
                $order->editor_pay = ArrayUtils::get($calc, 'editorCost');
                break;

            case Order::COMPLEX_ASSIGNMENT:
            default:
                $toptitle = ArrayUtils::get($payload, 'topic');
                if (!$toptitle) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter the title for your order"
                    ]);
                }
        
                $tracking_id = ArrayUtils::get($payload, 'tracking_id');
                if($tracking_id && Tools::strlen($tracking_id) < 3) {
                    return $this->response([
                        "error" => true,
                        "message" => "The tracking ID must be at least 3 characters long. Please try another"
                    ]);
                }
        
                /** Writer Category */
                $writer_cat_id = (int) ArrayUtils::get($payload, 'writerCategoryId'); 
                $writerPreferences = ArrayUtils::get($globals, 'writerPreferences');
        
                $writer_percent = 0;
                if (ArrayUtils::has($writerPreferences, $writer_cat_id)) {
                    $writer_preference = ArrayUtils::get($writerPreferences, $writer_cat_id);
                    $writer_percent = ArrayUtils::has($writer_preference, 'percent') ? (int) ArrayUtils::get($writer_preference, 'percent') : 0;
                }
        
                /** Pages / Charts / Slides */
                $charts = (int) ArrayUtils::get($payload, 'charts'); 
                $pages = (float) ArrayUtils::get($payload, 'pages'); 
                $slides = (int) ArrayUtils::get($payload, 'slides');
                $excelSheets = (int) ArrayUtils::get($payload, 'excelSheets');
                $min_sources = (int) ArrayUtils::get($payload, 'sources');
        
                // @todo both pages, charts, and slides can't be zero
                if ( $pages == 0 && $slides == 0 && $charts == 0 && $excelSheets == 0) {
                    return $this->response([
                        "error" => true,
                        "message" => "You need to order at least 1 page or 1 slide or 1 chart or 1 excel sheet"
                    ]);
                }
                
                /** Spacing */
                $spacing = ArrayUtils::get($payload, 'spacing'); 
                if ( !$spacing || !in_array($spacing, array(Order::SPACING_SINGLE, Order::SPACING_DOUBLE)) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid spacing"
                    ]);
                }
        
                /** Topic category */
                $topic_cat_id = (int) ArrayUtils::get($payload, 'topicCatId');
                $discipline = new Discipline( $topic_cat_id );
        
                if(Validate::isLoadedObject($discipline)) {
                    if($discipline->id == 52) {
                        $isComplexAssignment = 1;
                        $topic_cat_option = ArrayUtils::get($payload, 'topcatOption');
                    } else {
                        $isComplexAssignment = $discipline->is_complex_assignment;
                        $topic_cat_option = $discipline->title;
                    }
                } else {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid discipline"
                    ]);
                }
                
                /** Paper Type */
                $paper_type_id = (int) ArrayUtils::get($payload, 'paperTypeId');
                $paperType = new PaperType( $paper_type_id );
                if(Validate::isLoadedObject($paperType)) {
                    if($paperType->id == 64) {
                        $paper_type_option = ArrayUtils::get($payload, 'paperTypeOption');
                    } else {
                        $paper_type_option = $paperType->title;
                    }
                } else {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid paper type"
                    ]);
                }
        
                /** Paper Format */
                if(ArrayUtils::has($payload, 'paperFormat')) {
                    $paper_format_id = (int) ArrayUtils::get($payload, 'paperFormat');
        
                    $paperFormat = new PaperFormat( $paper_format_id  );
                    if(Validate::isLoadedObject($paperFormat)) {
                        if($paperFormat->id == 4) {
                            $paper_format_option = ArrayUtils::get($payload, 'paperFormatOption');
                        } else {
                            $paper_format_option = $paperFormat->title;
                        }
                    } else {
                        return $this->response([
                            "error" => true,
                            "message" => "Select a valid paper format"
                        ]);
                    }
        
                    $order->paper_format_id = $paper_format_id;
                    $order->paper_format_option = $paper_format_option;
                }
                
                if($new_order || $changeDeadline) {
                    $tariff_id = (int) ArrayUtils::get($payload, 'tariffId');
                    $deadline = new Deadline( (int) $tariff_id );
                    if(!Validate::isLoadedObject($deadline)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Select a valid deadline",
                        ]);
                    }
        
                    $order->tariff_id = $tariff_id;
                    $order->price_per_page = $deadline->price_per_page;
                    $order->hrs_customer = $deadline->hrs_customer;
                    $order->hrs_writer = $deadline->hrs_writer;
                    $order->customer_deadline = DateUtils::date(time() + ($deadline->hrs_customer*60*60));
                    $order->writer_deadline = DateUtils::date(time() + ($deadline->hrs_writer*60*60));
                }
        
                $paper_details = ArrayUtils::get($payload, 'paperDetails');
        
                $used_sources = ArrayUtils::has($payload, 'usedsources') ? true : false;
                $writer_samples = ArrayUtils::has($payload, 'samplesNeeded') ? true : false;
                $progressive_delivery = ArrayUtils::has($payload, 'progressiveDeliveryHidden') ? true : false;
                $expert_proofreading = ArrayUtils::has($payload, 'expertProofreading') ? true : false;
        
                if(ArrayUtils::has($payload, 'academicLevelId')) {
                    $academic_level_id = ArrayUtils::get($payload, 'academicLevelId');
                    $order->academic_level_id = $academic_level_id;
                }
        
                $order->title = $toptitle;
                $order->tracking_id = $tracking_id;
                $order->paper_type_id = $paper_type_id;
                $order->paper_type_option = $paper_type_option;
                $order->topic_category_id = $topic_cat_id;
                $order->topic_category_option = $topic_cat_option;
                $order->is_complex_assignment = $isComplexAssignment;
        
                $order->writer_category_id = $writer_cat_id;
                $order->writer_percent = $writer_percent;
        
                $order->sources = $min_sources;
                $order->pages = $pages;
                $order->charts = $charts;
                $order->slides = $slides;
                $order->excel_sheets = $excelSheets;
                $order->spacing = $spacing;
                $order->paper_details = $paper_details;
        
                /** Additional Services */
                $order->used_sources = $used_sources;
                $order->writer_samples = $writer_samples;
                $order->expert_proofreading = $expert_proofreading;
                $order->progressive_delivery = $progressive_delivery;

                $calc = $order->calculate();
                $baseCost = ArrayUtils::get($calc, 'rawCost');
                $servicesById = ArrayUtils::get($calc, 'servicesById');

                $order->raw_cost = $baseCost;
                $order->total = ArrayUtils::get($calc, 'totalCost'); 
                $order->writer_pay = ArrayUtils::get($calc, 'writerCost');
                $order->editor_pay = ArrayUtils::get($calc, 'editorCost');

                $serviceProvideSamples = $servicesById[PriceCalculator::SERVICES_IDS['PROVIDE_ME_SAMPLES']];
                if ($serviceProvideSamples['cost'] > 0) {
                    $order->writer_samples = true;
                } else {
                    $order->writer_samples = false;
                }

                $serviceUsedSources = $servicesById[PriceCalculator::SERVICES_IDS['USED_SOURCES']];
                if ($serviceUsedSources['cost'] > 0) {
                    $order->used_sources = true;
                } else {
                    $order->used_sources = false;
                }

                $serviceComplexAssignment = $servicesById[PriceCalculator::SERVICES_IDS['COMPLEX_ASSIGNMENT']];
                if ($serviceComplexAssignment['cost'] > 0) {
                    $order->is_complex_assignment = true;
                } else {
                    $order->is_complex_assignment = false;
                }

                $serviceExpertProofreading = $servicesById[PriceCalculator::SERVICES_IDS['EXPERT_PROOFREADING']];
                if ($serviceExpertProofreading['cost'] > 0) {
                    $order->expert_proofreading = true;
                } else {
                    $order->expert_proofreading = false;
                }

                $serviceProgressiveDelivery = $servicesById[PriceCalculator::SERVICES_IDS['PROGRESSIVE_DELIVERY']];
                if ($serviceProgressiveDelivery['cost'] > 0) {
                    $order->progressive_delivery = true;
                } else {
                    $order->progressive_delivery = false;
                }
                break;
        }

        Hook::exec('actionOrderAdd', array('order_id' => (int) $order->id, 'order' => $order));

        if($new_order) {
            $order->add();

            Hook::exec('activitylog', [
                'object' => 'order',
                'object_id' => $order->id,
                'event' => 'create_order'
            ]);

            return $this->response([
                'callback' => 'window.location.href="'.$app->base_uri.'/order/'.$order->id.'";'
            ]);	
        } else {
            $order->update();

            Hook::exec('activitylog', [
                'object' => 'order',
                'object_id' => $order->id,
                'event' => 'edit_order'
            ]);

            return $this->response([
                "success" => true,
                "message" => "Successfully updated"
            ]);
        }
    }

    public function order( $order_id, $do = null ) {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            return $this->response([
                "error" => true,
                "message" => "This order does not exist"
            ]);
        }

        switch( $do ) {
            # mark order as paid
            case 'mark-as-paid':
                if (!$order->is_paid) {
                    $customerDeadline = time() + ($order->hrs_customer*60*60);
                    $writerDeadline = time() + ($order->hrs_writer*60*60);

                    if ($order->progressive_delivery) {
                        $order->status_id = Order::NEW_PAID;
                    } else {
                        $order->status_id = Order::ORDER_PUBLISHED;
                    }

                    // $order->customer_deadline = DateUtils::date($customerDeadline);
                    // $order->writer_deadline = DateUtils::date($writerDeadline);

                    $order->total_paid = $order->total;
                    $order->is_paid = 1;
                    $order->paid_at = DateUtils::now();
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'mark_as_paid'
                    ]);
                }

                return $this->response([
                    'callback' => 'window.location.reload();'
                ]);
                break;
                
            # assign order
            case 'assign':
                $user_id = (int) ArrayUtils::get($payload, 'uid');
                $handle = ArrayUtils::get($payload, 'handle');

                $handle_user = new Employee( $user_id );
                if(!Validate::isLoadedObject($handle_user)) {
	                return $this->modal("ERROR", "Error", "The user does not exist");
                }

                $send_mail = true;

                switch($handle) {
                    case 'writer':
                        if($order->writer_id && $order->writer_assigned) {
                            $currentWriter = new Employee( (int) $order->writer_id );
                        }

                        $orderRules = $handle_user->getEmployeeOrderRules();

                        if($order->service_type == Order::ARTICLE_WRITING) { 
                            $calc = $order->calculate([
                                'writerCpwType' => $orderRules['cpw_type'],
                                'writerAmountCpw' => $orderRules['price_per_word']
                            ]);
                            $order->writer_pay = (float) ArrayUtils::get($calc, 'writerCost');
                        } elseif($order->service_type == Order::PROGRAMMING || $order->service_type == Order::CALCULATIONS ) {

                        } else {
                            $calc = $order->calculate([
                                'writerCppType' => $orderRules['cpp_type'],
                                'writerAmountCpp' => $orderRules['price_per_page']
                            ]);

                            $order->writer_pay = (float) ArrayUtils::get($calc, 'writerCost');
                        }
        
                        $order->status_id = Order::WRITER_ASSIGNED;
                        $order->writer_id = $handle_user->id;
                        $order->writer_assigned = 1;
                        $order->writer_assigned_at = DateUtils::now();
        
                        if(Configuration::get('CONFIRM_ORDER_ASSIGN')) {
                            $order->is_writer_confirmed = 0;
                        } else {
                            $order->is_writer_confirmed = 1;
                        }

                        $order->update();
        
                        $orderWriterId = $order->setCurrentWriter($handle_user->id);
                        $orderWriter = new OrderEmployee((int) $orderWriterId);
                        if(Validate::isLoadedObject($orderWriter)) {
                            $orderWriter->hrs_employee = $order->hrs_writer;
                            $orderWriter->employee_deadline = $order->writer_deadline;
                            $orderWriter->employee_pay = $order->writer_pay;

                            $orderWriter->is_auto_assigned = 1;
                            $orderWriter->is_employee_assigned = 1;
                            $orderWriter->employee_assigned_at = $order->writer_assigned_at;
                            
                            if($order->is_writer_confirmed) {
                                $orderWriter->is_employee_confirmed = $order->is_writer_confirmed;
                                $orderWriter->employee_confirmed_at = DateUtils::now();
                            }
                            
                            $orderWriter->update();
                        }

                        $sql = new DbQuery();
                        $sql->select('bid.bid_id');
                        $sql->from('order_bid', 'bid');
                        $sql->where('order_id = ' . (int) $order->id );
                        $sql->where('writer_id = ' . (int) $handle_user->id );
                        $bid_id = Db::getInstance(PROX_USE_SQL_SLAVE)->getValue( $sql );
                        
                        $bid = new Bid( (int) $bid_id );
                        if( Validate::isLoadedObject($bid) ) {
                            $bid->is_assigned = 1;
                            $bid->update();
                        }

                        if(!$order->ordermanager_assigned) {
                            $orderManagers = $order->getOrderManagers();

                            if(count($orderManagers) > 0) {
                                $orderManager = $orderManagers[0];
                                $employee = new Employee((int) $orderManager['employee_id']);
                                if(Validate::isLoadedObject($employee)) {
                                    $orderRules = $handle_user->getEmployeeOrderRules();
                                    $calc = $order->calculate([
                                        'orderManagerCppType' => $orderRules['cpp_type'],
                                        'orderManagerAmountCpp' => $orderRules['price_per_page']
                                    ]);
                                    if(ArrayUtils::has($calc, 'orderManagerCost')) {
                                        $order->ordermanager_pay = (float) ArrayUtils::get($calc, 'orderManagerCost');
                                    }

                                    $order->ordermanager_id = $employee->id;
                                    $order->ordermanager_assigned = 1;
                                    $order->ordermanager_assigned_at = DateUtils::now();
                                    $order->update();
                                }
                            }
                        }

                        if(isset($currentWriter) && Validate::isLoadedObject($currentWriter)) {
                            $order->is_writer_paid = 0;
                            $order->update();

                            $smarty->assign([
                                'order_id' => $order->id,
                                'first_name' => $currentWriter->first_name,
                                'last_name' => $currentWriter->last_name,
                            ]);
                            
                            $order_reassign_body = $smarty->fetch('messages/reassign_order.tpl');
        
                            $message = new Message();
                            $message->order_id = $order->id;
                            $message->sender_id = Employee::START_ID;
                            $message->from_department = DEPARTMENT_SUPPORT;
                            $message->to_department = DEPARTMENT_WRITER;
                            $message->subject = "Order #".$order->id.": Please stop working on the order";
                            $message->body = $order_reassign_body;
                            $message->is_verified = 1;
                            $message->verified_at = DateUtils::now();
                            $message->add();
        
                            try {
                                Mail::Send(
                                    MailTemplate::TEMPLATE_ORDER_REASSIGN,
                                    $message->subject,
                                    array(
                                        'order_id' => $order->id,
                                        'order_title' => $order->title,
                                        'order_url' => $app->base_uri . '/order/' . $order->id,
                                        'message' => Tools::nl2br($order_reassign_body),
                                        'username' => $currentWriter->first_name . ' ' . $currentWriter->last_name
                                    ),
                                    $currentWriter->email
                                );
                            } catch( \Exception $e ) {
                            }
                        }

                        break;

                    case 'editor':
                        if($order->editor_id && $order->editor_assigned) {
                            $currentEditor = new Employee( (int) $order->editor_id );
                        }

                        if($order->service_type == Order::ARTICLE_WRITING) {
                            $orderRules = $handle_user->getEmployeeOrderRules();
                            $calc = $order->calculate([
                                'editorCpwType' => $orderRules['cpp_type'],
                                'editorAmountCpw' => $orderRules['price_per_page']
                            ]);
                            $order->editor_pay = (float) ArrayUtils::get($calc, 'editorCost');
                        } elseif($order->service_type == Order::PROGRAMMING || $order->service_type == Order::CALCULATIONS ) {

                        } else {
                            $orderRules = $handle_user->getEmployeeOrderRules();
                            $calc = $order->calculate([
                                'editorCppType' => $orderRules['cpp_type'],
                                'editorAmountCpp' => $orderRules['price_per_page']
                            ]);
                            $order->editor_pay = (float) ArrayUtils::get($calc, 'editorCost');
                        }
        
                        $order->editor_id = $handle_user->id;
                        $order->editor_assigned = 1;
                        $order->editor_assigned_at = DateUtils::now();
        
                        if(Configuration::get('CONFIRM_ORDER_ASSIGN')) {
                            $order->is_editor_confirmed = 0;
                        } else {
                            $order->is_editor_confirmed = 1;
                        }
        
                        $order->update();

                        if(isset($currentEditor) && Validate::isLoadedObject($currentEditor)) {
                            $smarty->assign([
                                'order_id' => $order->id,
                                'first_name' => $currentEditor->first_name,
                                'last_name' => $currentEditor->last_name,
                            ]);
                            
                            $order_reassign_body = $smarty->fetch('messages/reassign_order.tpl');
        
                            $message = new Message();
                            $message->order_id = $order->id;
                            $message->sender_id = Employee::START_ID;
                            $message->from_department = DEPARTMENT_SUPPORT;
                            $message->to_department = DEPARTMENT_WRITER;
                            $message->subject = "Order #".$order->id.": Please stop working on the order";
                            $message->body = $order_reassign_body;
                            $message->is_verified = 1;
                            $message->verified_at = DateUtils::now();
                            $message->add();
        
                            try {
                                Mail::Send(
                                    MailTemplate::TEMPLATE_ORDER_REASSIGN,
                                    $message->subject,
                                    array(
                                        'order_id' => $order->id,
                                        'order_title' => $order->title,
                                        'order_url' => $app->base_uri . '/order/' . $order->id,
                                        'message' => Tools::nl2br($order_reassign_body),
                                        'username' => $currentEditor->first_name . ' ' . $currentEditor->last_name
                                    ),
                                    $currentEditor->email
                                );
                            } catch( \Exception $e ) {
                            }
                        }
                        break;

                    case 'ordermanager':
                        $orderRules = $handle_user->getEmployeeOrderRules();
                        $calc = $order->calculate([
                            'orderManagerCppType' => $orderRules['cpp_type'],
                            'orderManagerAmountCpp' => $orderRules['price_per_page']
                        ]);

                        if(ArrayUtils::has($calc, 'orderManagerCost')) {
                            $order->ordermanager_pay = (float) ArrayUtils::get($calc, 'orderManagerCost');
                        }

                        $order->ordermanager_id = $handle_user->id;
                        $order->ordermanager_assigned = 1;
                        $order->ordermanager_assigned_at = DateUtils::now();
                        $order->update();

                        $send_mail = false;
                        break;
                }

                Hook::exec('activitylog', [
                    'object' => 'order',
                    'object_id' => $order->id,
                    'event' => 'order_assign'
                ]);

                if($send_mail) {
                    $smarty->assign([
                        'order_id' => $order->id,
                        'first_name' => $handle_user->first_name,
                        'last_name' => $handle_user->last_name,
                    ]);
                    
                    $confirm_order_body = $smarty->fetch('messages/confirm_order.tpl');

                    $message = new Message();
                    $message->order_id = $order->id;
                    $message->sender_id = Employee::START_ID;
                    $message->from_department = DEPARTMENT_SUPPORT;
                    $message->to_department = DEPARTMENT_WRITER;
                    $message->subject = "Order #".$order->id.": Please confirm that you are working";
                    $message->body = $confirm_order_body;
                    $message->is_verified = 1;
                    $message->verified_at = DateUtils::now();
                    $message->add();

                    try {
                        Mail::Send(
                            MailTemplate::TEMPLATE_ORDER_ASSIGN,
                            $message->subject,
                            array(
                                'order_id' => $order->id,
                                'order_title' => $order->title,
                                'order_url' => $app->base_uri . '/order/' . $order->id,
                                'message' => Tools::nl2br($confirm_order_body),
                                'username' => $handle_user->first_name . ' ' . $handle_user->last_name
                            ),
                            $handle_user->email
                        );
                    } catch( \Exception $e ) {
                    }
                }

                return $this->response([
                    'callback' => 'window.location.reload();'
                ]);
                break;

            # take
            case 'take':

                if($user->is_writer) {
                    if($order->writer_id && $order->writer_assigned) {
                        return $this->modal("ERROR", "Error", "This order has been assigned to another writer");
                    }

                    $orderRules = $user->getEmployeeOrderRules();
                    $progress_orders = $user->order_stats['unconfirmed'] + $user->order_stats['progress'] + $user->order_stats['revision'];
                    $number_of_takes = $orderRules['takes'];

                    $can_take = false;
                    if($orderRules['takes'] == '-1') {
                        $can_take = true;
                    } else {
                        $can_take = ($progress_orders < $number_of_takes) ? true : false;
                    }

                    if(!$can_take) {
                        return $this->modal("ERROR", "Out of Takes", "You cannot take any more orders");
                    }

                    $orderRules = $user->getEmployeeOrderRules();

                    if($order->service_type == Order::ARTICLE_WRITING) { 
                        $calc = $order->calculate([
                            'writerCpwType' => $orderRules['cpw_type'],
                            'writerAmountCpw' => $orderRules['price_per_word']
                        ]);
                        $order->writer_pay = (float) ArrayUtils::get($calc, 'writerCost');
                    } elseif($order->service_type == Order::PROGRAMMING || $order->service_type == Order::CALCULATIONS ) {

                    } else {
                        $calc = $order->calculate([
                            'writerCppType' => $orderRules['cpp_type'],
                            'writerAmountCpp' => $orderRules['price_per_page']
                        ]);
                        $order->writer_pay = (float) ArrayUtils::get($calc, 'writerCost');
                    }

                    $order->status_id = Order::WRITER_ASSIGNED;
                    $order->writer_id = $user->id;
                    $order->writer_assigned = 1;
                    $order->writer_assigned_at = DateUtils::now();
                    $order->is_writer_confirmed = 1;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'writer_order_take'
                    ]);

                    return $this->modal("SUCCESS", "Successful", "You have successfuly been assigned order #".$order->id);
                }

                if($user->is_editor) {
                    if(!$order->editor_id && $order->editor_assigned) {
                        return $this->modal("ERROR", "Error", "This order has been assigned to another editor");
                    }
                    
                    if ( $order->status_id == Order::DONE ) {
                        $order->editor_id = $user->id;
                        $order->editor_assigned = 1;
                        $order->editor_assigned_at = DateUtils::now();
                        $order->is_editor_confirmed = 1;

                        if($order->service_type == Order::ARTICLE_WRITING) {
                            $orderRules = $user->getEmployeeOrderRules();
                            $calc = $order->calculate([
                                'editorCpwType' => $orderRules['cpp_type'],
                                'editorAmountCpw' => $orderRules['price_per_page']
                            ]);
                            $order->editor_pay = (float) ArrayUtils::get($calc, 'editorCost');
                        } elseif($order->service_type == Order::PROGRAMMING || $order->service_type == Order::CALCULATIONS ) {

                        } else {
                            $orderRules = $user->getEmployeeOrderRules();
                            $calc = $order->calculate([
                                'editorCppType' => $orderRules['cpp_type'],
                                'editorAmountCpp' => $orderRules['price_per_page']
                            ]);
                            $order->editor_pay = (float) ArrayUtils::get($calc, 'editorCost');
                        }
                        
                        $order->update();

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'editor_order_take'
                        ]);

                        return $this->modal("SUCCESS", "Successful", "You have successfuly been assigned order #".$order->id);
                    }
                }
                
                break;

            # apply 
            case 'apply':
                if (
                    $order->status_id != Order::ORDER_PUBLISHED &&
                    $order->status_id != Order::WRITER_ACCEPTED
                ) {
                    return $this->response([
                        'callback' => 'window.location.reload();'
                    ]);
                }

                $bid = new Bid();
                $bid->order_id = $order->id;
                $bid->writer_id = $user->id;
                $bid->bid_time = (int) $order->hrs_writer*60*60;
                $bid->amount = $order->writer_pay;
                $bid->add();

                $order->status_id = Order::WRITER_ACCEPTED;
                $order->update();

                Hook::exec('activitylog', [
                    'object' => 'order',
                    'object_id' => $order->id,
                    'event' => 'order_bid'
                ]);
                
                // notification
                $notification = new Notification();
                $notification->to_department = DEPARTMENT_SUPPORT;
                $notification->to_user_id = Employee::START_ID;
                $notification->from_department = DEPARTMENT_WRITER;
                $notification->from_user_id = $user->id;
                $notification->action = Notification::ACTION_ORDER;
                $notification->node_type = Notification::TYPE_BID_REQUEST;
                $notification->node_url = $order->id;
                $notification->send();

                return $this->modal("SUCCESS", "Successful Bid", "Your bid for order #".$order->id." was submitted successfuly." );
                break;

            # remove bid
            case 'remove_bid':
                $bid_id = ArrayUtils::get($payload, 'bid_id');

                $bid = new Bid( (int) $bid_id );
                if( Validate::isLoadedObject($bid) ) {
                    $bid->delete();
                }

                Hook::exec('activitylog', [
                    'object' => 'order',
                    'object_id' => $order->id,
                    'event' => 'remove_bid'
                ]);

                return $this->modal("SUCCESS", "Successful", "You have successfuly declined your bid for order #".$order->id);

                break;
            
            # confirm order
            case 'confirm':
                $do = ArrayUtils::get($payload, 'do');

                if ($do == "accept") {
                    $order->is_writer_confirmed = 1;
                    $order->update();

                    if(Validate::isLoadedObject($order->orderWriter)) {
                        $order->orderWriter->is_employee_confirmed = 1;
                        $order->orderWriter->employee_confirmed_at = DateUtils::now();
                        
                        $order->orderWriter->update();
                    }

                    Db::getInstance()->update(
                        'order_bid', 
                        array('is_assigned' => 1),
                        'order_id = ' . (int) $order->id . ' AND writer_id = ' . (int) $user->id
                    );

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'accept_request'
                    ]);

                } elseif ( $do == "decline" ) {
                    # delete bid
                    Db::getInstance()->delete('order_bid', 'order_id = ' . (int) $order->id . ' AND writer_id = ' . (int) $user->id );

                    $result = Db::getInstance()->executeS('SELECT order_id FROM ' . Db::prefix('order_bid') . ' WHERE order_id = ' . (int) $order->id );
                    if(count($result) == 0) {
                        $order->status_id = Order::ORDER_PUBLISHED;
                        $order->update();
                    } else {
                        $order->status_id = Order::WRITER_ACCEPTED;
                    }

                    $currentWriter = $order->getCurrentWriterFull();
                    $orderWriter = new OrderEmployee((int) $currentWriter['order_employee_id']); 
                    if(Validate::isLoadedObject($order->orderWriter)) {
                        $order->orderWriter->is_employee_declined = 1;
                        $order->orderWriter->employee_declined_at = DateUtils::now();
                        
                        $order->orderWriter->update();
                    }

                    $order->writer_id = null;
                    $order->writer_assigned = 0;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'decline_request'
                    ]);

                    return $this->response([
                        'callback' => 'history.back();'
                    ]);	
                }

                break;
                
            # complete order
            case 'complete':
                $hookBeforeCompleteResult = Hook::exec('actionOrderCompleted', ['order' => $order], null, true);

                if ($hookBeforeCompleteResult === null) {
                    $keepGoing = false;
                } else {
                    $keepGoing = array_reduce(
                        $hookBeforeCompleteResult,
                        function ($carry, $item) {
                            return ($item === false) ? false : $carry;
                        },
                        true
                    );
                }
        
                if (!$keepGoing) {
                    return true;
                }

                if ($user->is_writer) {
                    $orderRules = $user->getEmployeeOrderRules();

                    if($orderRules['auto_delivery']) {
                        $order->deliverOrderToCustomer();
                        
                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'auto_deliver_order'
                        ]);
                    } else {
                        $order->status_id = Order::DONE;
                        $order->completed_at = DateUtils::now();
                        $order->update();

                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_completed = 1;

                            if(DateUtils::hasPassed($order->writer_deadline)) {
                                $order->orderWriter->is_late = 1;
                            }
                            
                            $order->orderWriter->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'complete_order'
                        ]);
    
                        try {
                            $mailParams = array(
                                'order_id' => $order->id,
                                'site_name' => Configuration::get('SITE_NAME'),
                                'order_title' => $order->title,
                                'order_url' => $app->base_uri . '/order/' . $order->id
                            );
    
                            $receivers = array();
                            $receivers[] = Configuration::get('SITE_EMAIL');
                            $notification_emails = $app->getNotificationEmails();
                            $receivers = array_merge($receivers, $notification_emails);
    
                            foreach($receivers as $receiver_mail) {
                                Mail::send(
                                    MailTemplate::TEMPLATE_ORDER_WRITER_SUBMIT,
                                    'Order #' . $order->id .  ' has been completed',
                                    $mailParams,
                                    $receiver_mail
                                );  
                            }
                        } catch( \Exception $e ) {
                        }
    
                        try {
                            $receivers = array();
    
                            $sql = new DbQuery();
                            $sql->select('e.email');
                            $sql->from('employee', 'e');
                            $sql->where('employee_group = ' . (int) Employee::GROUP_EDITOR);
                            $sql->where('is_started = 1');
                            $sql->where('is_banned = 0');
                            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
    
                            foreach ($result as $editor) {
                                if( ArrayUtils::has($editor, 'email') && Validate::isEmail(ArrayUtils::get($editor, 'email')) ) {
                                    $receivers[] = ArrayUtils::get($editor, 'email');
                                }
                            }
    
                            foreach ($receivers as $receiver) {
                                Mail::Send(
                                    MailTemplate::TEMPLATE_ORDER_NEW_EDITOR,
                                    'New Order #' . $order->id . ' available',
                                    $mailParams,
                                    $receiver
                                );
                            }
                        } catch( \Exception $e ) {
                        }
                    }
                }

                return $this->modal("SUCCESS", "Order Completed", "Order has been completed successfuly");
                break;  
                    
            # deliver order
            case 'deliver':
                $order->deliverOrderToCustomer();

                Hook::exec('activitylog', [
                    'object' => 'order',
                    'object_id' => $order->id,
                    'event' => 'deliver_order'
                ]);

                return $this->modal("SUCCESS", "Order Delivered", "Order has been delivered to the customer successfuly");
                break;
                
            case 'pd_schedule':
                $pages = (float) ArrayUtils::get($payload, 'pages');
                $slides = (int) ArrayUtils::get($payload, 'slides');
                $charts = (int) ArrayUtils::get($payload, 'charts');
                $comments = ArrayUtils::get($payload, 'comments');
                $customer_deadline = ArrayUtils::get($payload, 'customer_deadline');
                $writer_deadline = ArrayUtils::get($payload, 'writer_deadline');

                // @todo both pages, charts, and slides can't be zero
                if ( $pages == 0 && $slides == 0 && $charts == 0) {
                    return $this->response([
                        "error" => true,
                        "message" => "You need at least 1 page or 1 slide or 1 chart"
                    ]);
                }

                if(!$customer_deadline) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter a vaild customer deadline"
                    ]);
                }

                if(!$writer_deadline) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter a vaild writer deadline"
                    ]);                        
                }

                if(!$comments) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter a comment"
                    ]);                        
                }

                $customerDeadline = DateUtils::date(strtotime($customer_deadline));
                $writerDeadline = DateUtils::date(strtotime($writer_deadline));

                $pdSchedule = new DeliverySchedule();
                $pdSchedule->order_id = $order->id;
                $pdSchedule->pages = $pages;
                $pdSchedule->slides = $slides;
                $pdSchedule->charts = $charts;
                $pdSchedule->comment = $comments;
                $pdSchedule->customer_deadline = $customerDeadline;
                $pdSchedule->writer_deadline = $writerDeadline;
                $pdSchedule->add();

                return $this->modal("SUCCESS", "Successful", "Progressive Schedule has been updated successfuly");
                break;

            case 'edit-cost':

                if ( ArrayUtils::has($payload, 'customer_pay') ) {
                    $customer_pay = ArrayUtils::get($payload, 'customer_pay');
                    if(!Validate::isPrice($customer_pay)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Enter a vaild amount"
                        ]);  
                    }
                    $order->total = $customer_pay;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'edit_customer_pay'
                    ]);

	                return $this->modal("SUCCESS", "Successful", "Customer Fee has been updated. Reload page to see changes.");
                }

                if ( ArrayUtils::has($payload, 'writer_pay') ) {
                    $writer_pay = ArrayUtils::get($payload, 'writer_pay');
                    if(!Validate::isPrice($writer_pay)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Enter a vaild amount"
                        ]);  
                    }

                    $order->writer_pay = $writer_pay;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'edit_writer_pay'
                    ]);

	                return $this->modal("SUCCESS", "Successful", "Writer Fee has been updated. Reload page to see changes.");
                }

                if ( ArrayUtils::has($payload, 'editor_pay') ) {
                    $editor_pay = ArrayUtils::get($payload, 'editor_pay');
                    if(!Validate::isPrice($editor_pay)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Enter a vaild amount"
                        ]);  
                    }
                    $order->editor_pay = $editor_pay;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'edit_editor_pay'
                    ]);

	                return $this->modal("SUCCESS", "Successful", "Editor Fee has been updated. Reload page to see changes.");
                }

                if ( ArrayUtils::has($payload, 'ordermanager_pay') ) {
                    $ordermanager_pay = ArrayUtils::get($payload, 'ordermanager_pay');
                    if(!Validate::isPrice($ordermanager_pay)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Enter a vaild amount"
                        ]);  
                    }
                    $order->ordermanager_pay = $ordermanager_pay;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'edit_ordermanager_pay'
                    ]);

	                return $this->modal("SUCCESS", "Successful", "Order Manager Fee has been updated. Reload page to see changes.");
                }

                return $this->modal("ERROR", "Not found", "Error 404. Resource not found");
                break;

            case 'edit-status':
                $status_id = ArrayUtils::get($payload, 'status_id');
                if( $status_id < 0 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid status"
                    ]);  
                }

                $order->status_id = $status_id;
                
                switch($status_id) {
                    case Order::ORDER_PUBLISHED:
                    case Order::NEW_PAID:

                        $order->markAsPaid();

                        if($order->writer_id) {
                            $order->writer_id = 0;
                            $order->writer_assigned = 0;
                            $order->is_writer_confirmed = 0;
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_order_published'
                        ]);

                        break;

                    case Order::WRITER_ASSIGNED:
                        $order->markAsPaid();

                        if(Configuration::get('CONFIRM_ORDER_ASSIGN')) {
                            $order->is_writer_confirmed = 0;
                        } else {
                            $order->is_writer_confirmed = 1;
                        }

                        $order->writer_assigned_at = DateUtils::now();

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_writer_assigned'
                        ]);
                        break;

                    case Order::DONE:
                        $order->markAsPaid();

                        $order->completed_at = DateUtils::now();

                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_completed = 1;
                            $order->orderWriter->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_done'
                        ]);
                        break;

                    case Order::DELIVERED:
                        $order->markAsPaid();

                        if(!$order->completed_at) {
                            $order->completed_at = DateUtils::now();
                        }
                        if(!$order->finished_at) {
                            $order->finished_at = DateUtils::now();
                        }

                        $order->delivered_at = DateUtils::now();
                        
                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_delivered = 1;
                            $order->orderWriter->update();
                        } 

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_delivered'
                        ]);
                        break;

                    case Order::FINISHED:
                        $order->markAsPaid();

                        if(!$order->completed_at) {
                            $order->completed_at = DateUtils::now();
                        }
                        if(!$order->delivered_at) {
                            $order->delivered_at = DateUtils::now();
                        }
                        if(!$order->finished_at) {
                            $order->finished_at = DateUtils::now();
                        }

                        $order->approved_at = DateUtils::now();
                        
                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_approved = 1;
                            $order->orderWriter->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_finished'
                        ]);
                        break;

                    case Order::CANCELLED:
                        $order->cancelled_at = DateUtils::now();

                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_cancelled = 1;
                            $order->orderWriter->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_cancelled'
                        ]);
                        break;
                    case Order::DISPUTE:
                        $order->cancelled_at = DateUtils::now();

                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_disputed = 1;
                            $order->orderWriter->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_dispute'
                        ]);
                        break;

                    case Order::REVISION:
                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_revised = 1;
                            $order->orderWriter->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'edit_status_revision'
                        ]);
                        break;
                }

                $order->update();

                $customer = $order->getCustomer();
                $customerSite = $customer->getCustomerSite();

                $order_url = Configuration::get('SITE_DOMAIN', $customerSite->id ) . '/dashboard/order/' . $order->id;

                if(Validate::isLoadedObject($customer) && !$customer->is_anonymous) {
                    try {
                        $mailParams = array(
                            'order_domain' => $customerSite->domain,
                            'order_url' => $order_url,
                            'order_id' => $order->id,
                            'order_title' => $order->title,
                            'username' => $customer->name
                        );
    
                        Mail::send(
                            MailTemplate::TEMPLATE_ORDER_STATUS_CHANGE,
                            "Order #" . $order->id . ": Order status has changed",
                            $mailParams,
                            $customer->email,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            $customer->site_id
                        );  
                    } catch( \Exception $e ) {
                        
                    }
                }

                Hook::exec('actionOrderStatusUpdate', array(
                    'order_id' => $order->id,
                    'newOrderStatus' => $order->status_id
                ));

                return $this->modal("SUCCESS", "Successful", "The order status has been changed successfuly");
                break;

            case 'deadline':
                $handle = ArrayUtils::get($payload, 'handle');
                $deadline = ArrayUtils::get($payload, 'deadline');
                
                if(!$deadline) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid deadline"
                    ]);
                }

                $selectedDeadline = new DateTime($deadline, new DateTimeZone($user->timezone));
                $selectedDeadline->setTimezone(new DateTimeZone("UTC"));
                $deadline = $selectedDeadline->format("Y-m-d H:i:s"); 


                if ($handle == "customer") {
                    $order->customer_deadline = $deadline;
                    $order->update();

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'edit_customer_deadline'
                    ]);
                } elseif ( $handle == "writer" ) {
                    $order->writer_deadline = $deadline;
                    $order->update();

                    if(Validate::isLoadedObject($order->orderWriter)) {
                        $order->orderWriter->employee_deadline = $deadline;
                        $order->orderWriter->update();
                    }

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'edit_writer_deadline'
                    ]);
                }

                return $this->modal("SUCCESS", "Successful", "Deadline has been extended. Reload page to see changes.");

                break;

            case 'additional-payment':
                $amount = (float) ArrayUtils::get($payload, 'amount');
                $pages = (float) ArrayUtils::get($payload, 'pages');
                $charts = (int) ArrayUtils::get($payload, 'charts');
                $slides = (int) ArrayUtils::get($payload, 'slides');
                $words = (int) ArrayUtils::get($payload, 'words');
                $comments = ArrayUtils::get($payload, 'comments');

                if($amount <= 0) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter the amount"
                    ]);
                }

                /** Additional Service **/
                $additionalService = new AdditionalService();
                $additionalService->order_id = $order->id;
                $additionalService->total = $amount;

                $additionalService->comment = $comments;

                if($pages) {
                    $additionalService->pages = $pages;
                }

                if($charts) {
                    $additionalService->charts = $charts;
                }

                if($slides) {
                    $additionalService->slides = $slides;
                }

                if($words) {
                    $additionalService->slides = $words;
                }

                if (!$additionalService->add()) {
                    return $this->response([
                        "error" => true,
                    ]);
                }

                Hook::exec('activitylog', [
                    'object' => 'add-payment',
                    'object_id' => $additionalService->id,
                    'event' => 'create_payment_link'
                ]);

                $payment_link = Configuration::get('SITE_DOMAIN', $order->site_id) . '/dashboard/add-payment/' . $additionalService->id;
                
                $modalData = [
                    'payment_link' => $payment_link
                ];

                return $this->modal("#modalAdditionalPaymentLink", Tools::jsonEncode($modalData)); 
                break;

            case 'reorder':

                $newOrder = $order->duplicate();

                if(Validate::isLoadedObject($newOrder)) {
                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $newOrder->id,
                        'event' => 'reorder'
                    ]);
    
                    return $this->response([
                        'callback' => 'window.location.href="/order/'. $newOrder->id .'";'
                    ]);
                } else {
                    return $this->modal("ERROR", "Re-order Unsuccessful", "This order could not be duplicated successful"); 
                }

                break;

            case 'duplicate':
                $duplicates = (int) ArrayUtils::get($payload, 'duplicates');
                if($duplicates == 0) {
                    return $this->response([
                        "error" => true,
                        "message" => "Duplicates should be at least 1"
                    ]);
                }

                for ($x = 1; $x <= $duplicates; $x++) {
                    $newOrder = $order->duplicate();
                }

                return $this->modal("SUCCESS", "Duplicate Successful", "This order has been duplicated successfuly"); 

                Hook::exec('activitylog', [
                    'object' => 'order',
                    'object_id' => $order->id,
                    'event' => 'duplicate_order'
                ]);

                break;

            case 'sendToRevision':
                try {
                    $revisionNote = ArrayUtils::get($payload, 'revision_note');
                    if(!$revisionNote) {
                        return $this->response([
                            "error" => true,
                            "message" => "Enter the revision note"
                        ]);
                    }

                    $order->status_id = Order::REVISION;
                    $order->update();

                    if(Validate::isLoadedObject($order->orderWriter)) {
                        $order->orderWriter->is_revised = 1;
                        $order->orderWriter->update();
                    }

                    Hook::exec('activitylog', [
                        'object' => 'order',
                        'object_id' => $order->id,
                        'event' => 'order_revision'
                    ]);

                    $smarty->assign([
                        'order_id' => $order->id,
                        'order_url' => Configuration::get('SITE_DOMAIN') . '/order/' . $order->id,
                        'subject' => "Order #" . $order->id . ": Please revise the order",
                        'body' => $revisionNote
                    ]);
            
                    $revision_body = $smarty->fetch('messages/order_revision.tpl');

                    /** Add Revision Note **/
                    $message = new Message();
                    $message->order_id = $order->id;
                    $message->sender_id = Employee::START_ID;
                    $message->from_department = DEPARTMENT_SUPPORT;
                    $message->to_department = DEPARTMENT_WRITER;
                    $message->subject = "Order #".$order->id.": Please revise the order";
                    $message->body = $revision_body;
                    $message->is_verified = true;
                    $message->verified_at = DateUtils::now();
                    $message->add();
                    
                    $writer = new Employee( (int) $order->writer_id );
                    if(Validate::isLoadedObject($writer)) {
                        Mail::Send(
                            MailTemplate::TEMPLATE_ORDER_REVISION,
                            $message->subject,
                            array(
                                'username' => $writer->first_name . ' ' . $writer->last_name,
                                'order_id' => $order->id,
                                'order_url' => Configuration::get('SITE_DOMAIN') . '/order/' . $order->id,
                                'body' => $revision_body
                            ),
                            $writer->email
                        ); 
                    }
                } catch( \Exception $e ) {
                }   
                break;
            
            default:
                return $this->modal("ERROR", "Not found", "Error 404. Resource not found");
                break;
        }

        return $this->response([
            'callback' => 'window.location.reload();'
        ]);
    }

    public function sendMessage() {
		global $globals;

        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        $order_id = (int) ArrayUtils::get($payload, 'order_id');
        $recepient = (int) ArrayUtils::get($payload, 'recepient');
        $subject = ArrayUtils::get($payload, 'subject');
        $body = ArrayUtils::get($payload, 'message');
        $send_sms = ArrayUtils::has($payload, 'send_sms');

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            return $this->response([
                "error" => true,
                "message" => "This order does not exist"
            ]);
        }

        $message = new Message();

        $send_email_to = array();

        if ( $user->is_admin || $user->is_sub_admin ) {
            $from_department = DEPARTMENT_SUPPORT;
            /** Verify Message */
            $message->is_verified = 1;
            $message->verified_at = DateUtils::now();
        } else {
            $from_department = DEPARTMENT_WRITER;
            $send_email_to[] = Configuration::get('SITE_EMAIL');
            $notification_emails = $app->getNotificationEmails();
            $send_email_to[] = array_merge($send_email_to, $notification_emails);
        }

        if ( $recepient == DEPARTMENT_WRITER ) {
            $writer = new Employee( (int) $order->writer_id );
            if(Validate::isLoadedObject($writer)) {
                $send_email_to[] = $writer->email;
            }
        } elseif ( $recepient == DEPARTMENT_CUSTOMER ) {
            $customer = new Customer( (int) $order->customer_id );
            if(Validate::isLoadedObject($customer)) {
                $send_email_to[] = $customer->email;
            }
        }

        if (!$body) {
            return $this->response([
                "error" => true,
                "message" => "Please enter your message"
            ]);
        }
        
        $message->order_id = $order->id;
        $message->sender_id = $user->id;
        $message->from_department = $from_department;
        $message->to_department = $recepient;
        $message->subject = $subject;
        $message->body = $body;
        $success = $message->add();

        if(!$success) {
			return $this->response([
				"error" => true
			]);
		}

        Hook::exec('activitylog', [
            'object' => 'message',
            'object_id' => $message->id,
            'event' => 'send_message'
        ]);

		$departments = ArrayUtils::get($globals, 'interlocutor_departments');

        try {
            if($subject) {
                $subject = "Order #" . $order->id . " - " . $subject;
            } else {
                $subject = "Order #" . $order->id . ": New message from " . $user->is_admin ? "Support" : "Writer";
            }

            $mailParams = array(
                'send_sms' => (bool) $send_sms,
                'to_department' => Tools::ucfirst($departments[$message->to_department]),
                'from_department' => Tools::ucfirst($departments[$message->from_department]),
                'order_id' => $order->id,
                'subject' => $subject,
                'text_message' => Tools::getStringClean($message->body),
                'message' => Tools::nl2br($message->body)
            );

			if( $message->to_department == DEPARTMENT_CUSTOMER ) {
                $customerSite = $customer->getCustomerSite();

                $mailParams['username'] = $customer->name ? $customer->name : 'Customer #' . $customer->id; 
                $mailParams['order_url'] = $customerSite->domain . '/dashboard/order/' . $order->id; 

                foreach ($send_email_to as $receiver) {
                    Mail::Send(
                        MailTemplate::TEMPLATE_ORDER_MESSAGE,
                        $subject,
                        $mailParams,
                        $receiver,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        $customer->site_id
                    );
                }	
            } else {
                $mailParams['username'] = $writer->first_name . ' ' . $writer->last_name; 
                $mailParams['order_url'] = Configuration::get('SITE_DOMAIN') . '/order/' . $order->id; 

                foreach ($send_email_to as $receiver) {
                    Mail::Send(
                        MailTemplate::TEMPLATE_ORDER_MESSAGE,
                        $subject,
                        $mailParams,
                        $receiver
                    );	
                }
            }
		} catch( \Exception $e ) {
        }

		return $this->modal("SUCCESS", "Message Sent", "Your message has been sent successfuly", $app->base_uri . '/order/' . $order->id . '/messages');
    }

    public function replyMessage( $message_id ) {
        $app = $this->app;
		$payload = $app->request->post();
        $user = $app->user;

        $message = new Message( (int) $message_id );
        if(!Validate::isLoadedObject($message)) {
            return $this->response([
                'error' => true,
                'message' => "This message does not exist"
            ]);
        }

        $order = new Order( (int) $message->order_id );
        if(!Validate::isLoadedObject($message)) {
            return $this->response([
                'error' => true,
                'message' => "This message does not have an existing order"
            ]);
        }

        $body = ArrayUtils::get($payload, 'reply');
        if (!$body) {
            return $this->response([
                "error" => true,
                "message" => "Please enter your message"
            ]);
        }

        $reply = new Message();

        $send_email_to = array();

        if ( $user->is_admin || $user->is_sub_admin ) {
            $from_department = DEPARTMENT_SUPPORT;
            $reply->is_verified = 1;
            $reply->verified_at = DateUtils::now();
		} else {
            $from_department = DEPARTMENT_WRITER;
            $send_email_to[] = Configuration::get('SITE_EMAIL');
            $notification_emails = $app->getNotificationEmails();
            $send_email_to[] = array_merge($send_email_to, $notification_emails);
        }
 
		$reply->order_id = $order->id;
        $reply->parent_id = $message->id;
        $reply->sender_id = $user->id;
		$reply->from_department = $from_department;
		$reply->to_department = $message->from_department;
		$reply->subject = $message->subject;
        $reply->body = $body;
        $success = $reply->add();
        
        if ( $reply->to_department == DEPARTMENT_WRITER ) {
            $writer = $order->getWriter();
            if(Validate::isLoadedObject($writer)) {
                $send_email_to[] = $writer->email;
            }
        } elseif ( $reply->to_department == DEPARTMENT_CUSTOMER ) {
            $customer = $order->getCustomer();
            if(Validate::isLoadedObject($customer)) {
                $send_email_to[] = $customer->email;
            }
        }

        if(!$success) {
			return $this->response([
				"error" => true
			]);
		}

        Hook::exec('activitylog', [
            'object' => 'message',
            'object_id' => $message->id,
            'event' => 'message_reply'
        ]);

        try {
            $subject = "Order #" . $order->id . ": New message from " . $user->is_admin ? "Support" : "Writer";

            $mailParams = array(
                'to_department' => $reply->to_department,
                'subject' => $subject,
                'orderId' => $order->id,
                'text_message' => Tools::getStringClean($reply->body),
                'message' => Tools::nl2br($reply->body)
            );

            if( $reply->to_department == DEPARTMENT_CUSTOMER) {
                if(!$customer->is_anonymous) {
                    $mailParams['username'] = $customer->name ? $customer->name : 'Customer #' . $customer->id; 

                    foreach ($send_email_to as $receiver) {
                        Mail::Send(
                            MailTemplate::TEMPLATE_ORDER_MESSAGE,
                            $subject,
                            $mailParams,
                            $receiver,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            $customer->site_id
                        );	
                    }
                }
                
            } else {
                $mailParams['name'] = $writer->first_name . ' ' . $writer->last_name;
                
                foreach ($send_email_to as $receiver) {
                    Mail::Send(
                        MailTemplate::TEMPLATE_ORDER_MESSAGE,
                        $subject,
                        $mailParams,
                        $receiver
                    );	
                }
            }
			
		} catch( \Exception $e ) {
        }

		return $this->modal("SUCCESS", "Message Sent", "Your message has been sent successfuly", $app->base_uri . '/order/'.$order->id.'/messages');
    }

    public function readMessage( $message_id ) {
        $app = $this->app;
        $user = $app->user;

        $message = new Message( (int) $message_id );
        if(Validate::isLoadedObject($message)) {
            if( $message->sender_id != $user->id ) {
                $message->is_viewed = 1;
                $message->viewed_at = DateUtils::now();
                $message->update();
            }
        }

        Hook::exec('activitylog', [
            'object' => 'message',
            'object_id' => $message->id,
            'event' => 'read_message'
        ]);

        return $this->response([
            'success' => true
        ]);
    }

    public function uploadFiles( $order_id ) {
        $app = $this->app;
        $payload = $app->request->post();
		$params = $app->request->get();
        $user = $app->user;
        
        $order = new Order( (int) $order_id );
		if (!Validate::isLoadedObject($order)) {
            return $this->response([
                'error' => true,
                'message' => "This order does not exist"
            ]);
        }
        
        $files = ArrayUtils::get($payload, 'files');
		if (!is_array($files) || empty($files)) {
            return $this->response([
                'error' => true,
                'message' => "Select the files to upload"
            ]);
        }

        $upload_for = ArrayUtils::get($params, "upload_for");
        if ( $upload_for == "customer" ) {
            $uploader_id = $order->customer_id;
        } elseif ( $upload_for == "writer" ) {
            $uploader_id = $order->writer_id;
        } elseif ( $upload_for == "editor" && $order->editor_id ) {
            $uploader_id = $order->editor_id;
        } elseif ( $upload_for == "support" ) {
            $uploader_id = Employee::START_ID;
        } else {
            $uploader_id = $user->id;
        }
        
		foreach ($files as $uploaded_file) {
            $file_id = (int) ArrayUtils::get($uploaded_file, 'id');
            $file_type_id = (int) ArrayUtils::get($uploaded_file, 'uploadTypeId');
            $file_type = new UploadType( $file_type_id );
            if(!Validate::isLoadedObject($file_type)) {
                return $this->response([
                    'error' => true,
                    'message' => "Select the file type"
                ]);
            }
            
            $file = new File( (int) $file_id );
            if ( Validate::isLoadedObject( $file) ) {
                $file->order_id = $order->id;
                $file->uploader_id = (int) $uploader_id;
                $file->file_type_id = $file_type->id;
                $file->file_description = $file_type->title;
                $file->update();
            }	
		}

        Hook::exec('activitylog', [
            'object' => 'file',
            'object_id' => $file->id,
            'event' => 'file_upload'
        ]);

        return $this->response([
            'callback' => 'window.location.href="/order/'.$order->id.'/files";'
        ]);	
    }   

    public function bulkActions() {
		$app = $this->app;
		$payload = $app->request->post();
		$user = $app->user;

		if(!$user->is_admin && !$user->is_sub_admin) {
            return $this->modal("ERROR", "Error", "You don't have the right permission to access this"); 
        }

		$action = ArrayUtils::get($payload, 'action');
		if ( !$action ) {
			return $this->response([
				"error" => true,
				"message" => "Select a valid action"
			]);
		}

		$orderIds = ArrayUtils::get($payload, 'orderIds');
		if ( !$orderIds ) {
			return $this->response([
				"error" => true,
				"message" => "Select at least one order"
			]);
		}

		$orderIds = Tools::jsonDecode($orderIds);

		switch ( $action ) {
			case 'status':
				$status_id = (int) ArrayUtils::get($payload, 'status_id');
				foreach( $orderIds as $order_id ) {
					$order = new Order( (int) $order_id );
					if ( Validate::isLoadedObject($order) ) {
                        $order->status_id = (int) $status_id;
                
                        switch($status_id) {
                            case Order::ORDER_PUBLISHED:
                                if(!$order->is_paid) {
                                    $order->is_piad = 1;
                                    $order->paid_at = DateUtils::now();
                                }

                                if($order->writer_id) {
                                    $order->writer_id = 0;
                                    $order->writer_assigned = 0;
                                    $order->is_writer_confirmed = 0;
                                }

                                break;

                            case Order::WRITER_ASSIGNED:
                                if(Configuration::get('CONFIRM_ORDER_ASSIGN')) {
                                    $order->is_writer_confirmed = 0;
                                } else {
                                    $order->is_writer_confirmed = 1;
                                }

                                $order->writer_assigned_at = DateUtils::now();
                                break;

                            case Order::DONE:
                                $order->completed_at = DateUtils::now();
                                break;

                            case Order::DELIVERED:
                                if(!$order->completed_at) {
                                    $order->completed_at = DateUtils::now();
                                }
                                if(!$order->finished_at) {
                                    $order->finished_at = DateUtils::now();
                                }

                                $order->delivered_at = DateUtils::now();
                                break;

                            case Order::FINISHED:
                                if(!$order->completed_at) {
                                    $order->completed_at = DateUtils::now();
                                }
                                if(!$order->delivered_at) {
                                    $order->delivered_at = DateUtils::now();
                                }
                                if(!$order->finished_at) {
                                    $order->finished_at = DateUtils::now();
                                }
                                $order->approved_at = DateUtils::now();

                                break;

                            case Order::CANCELLED:
                                $order->cancelled_at = DateUtils::now();
                                break;
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'order_status_changed'
                        ]);

						$order->update();
					}
				}
				break;
		
			case 'mark-paid':
				foreach( $orderIds as $order_id ) {
					$order = new Order( (int) $order_id );
					if ( Validate::isLoadedObject($order) ) {
						$order->is_paid = 1;
						$order->paid_at = DateUtils::now();
						$order->update();

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'order_marked_paid'
                        ]);
					}
				}
				break;

            case 'mark-unpaid':
                foreach( $orderIds as $order_id ) {
                    $order = new Order( (int) $order_id );
                    if ( Validate::isLoadedObject($order) ) {
                        $order->is_paid = 0;
                        $order->update();

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'order_marked_unpaid'
                        ]);
                    }
                }
                break;
			
			case 'writer-paid':
				foreach( $orderIds as $order_id ) {
					$order = new Order( (int) $order_id );
					if ( Validate::isLoadedObject($order) ) {
						$order->is_writer_paid = 1;
                        $order->writer_paid_at = DateUtils::now();
                        $order->update();

                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_employee_paid = 1;
                            $order->orderWriter->employee_paid_at = $order->writer_paid_at;
                            $order->orderWriter->update();
                        }
                        
                        $writer = new Employee( (int) $order->writer_id );
                        if(Validate::isLoadedObject($writer)) {
                            $writer->last_pay = $order->writer_pay;
                            $writer->total_paid = $writer->total_paid + $order->writer_pay;
                            $writer->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'writer_marked_paid'
                        ]);
					}
				}
				break;

            case 'writer-unpaid':
                foreach( $orderIds as $order_id ) {
                    $order = new Order( (int) $order_id );
                    if ( Validate::isLoadedObject($order) ) {
                        $order->is_writer_paid = 0;
                        $order->update();

                        if(Validate::isLoadedObject($order->orderWriter)) {
                            $order->orderWriter->is_employee_paid = 0;
                            $order->orderWriter->update();
                        }
                        
                        $writer = new Employee( (int) $order->writer_id );
                        if(Validate::isLoadedObject($writer)) {
                            $writer->last_pay = $writer->last_pay - $order->writer_pay;
                            $writer->total_paid = $writer->total_paid - $order->writer_pay;
                            $writer->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'writer_marked_unpaid'
                        ]);
                    }
                }
                break;

            case 'editor-paid':
                foreach( $orderIds as $order_id ) {
                    $order = new Order( (int) $order_id );
                    if ( Validate::isLoadedObject($order) ) {
                        $order->is_editor_paid = 1;
                        $order->editor_paid_at = DateUtils::now();
                        $order->update();
                        
                        $editor = new Employee( (int) $order->editor_id );
                        if(Validate::isLoadedObject($editor)) {
                            $editor->last_pay = $order->editor_pay;
                            $editor->total_paid = $editor->total_paid + $order->editor_pay;
                            $editor->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'editor_marked_paid'
                        ]);
                    }
                }
                break;

            case 'editor-unpaid':
                foreach( $orderIds as $order_id ) {
                    $order = new Order( (int) $order_id );
                    if ( Validate::isLoadedObject($order) ) {
                        $order->is_editor_paid = 0;
                        $order->update();
                        
                        $editor = new Employee( (int) $order->editor_id );
                        if(Validate::isLoadedObject($editor)) {
                            $editor->last_pay = $editor->last_pay - $order->editor_pay;
                            $editor->total_paid = $editor->total_paid - $order->editor_pay;
                            $editor->update();
                        }

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'editor_marked_unpaid'
                        ]);
                    }
                }
                break;

            case 'delete':
                foreach( $orderIds as $order_id ) {
					$order = new Order( (int) $order_id );
					if ( Validate::isLoadedObject($order) ) {
                        $order->delete();

                        Hook::exec('activitylog', [
                            'object' => 'order',
                            'object_id' => $order->id,
                            'event' => 'order_deleted'
                        ]);
					}
				}
				break;
		}

		return $this->modal('SUCCESS', "Successful", "Bulk update was successful");	
	}

    public function feedbackReply( $feedbackId ) {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        $rating = new Rating( (int) $feedbackId );
        if(!Validate::isLoadedObject($rating)) {
            return $this->modal("ERROR", "This feedback does not exist");
        }

        $order = new Order( (int) $rating->order_id );
        if(!Validate::isLoadedObject($order)) {
            return $this->modal("ERROR", "Reply not Sent", "A reply could not be sent to the customer.");
        }

        $customer = new Customer( (int) $order->customer_id );
        if(!Validate::isLoadedObject($customer)) {
            return $this->modal("ERROR", "Reply not Sent", "A reply could not be sent to the customer.");
        }

        $reply = ArrayUtils::get($payload, 'reply');
        if(!$reply) {
            return $this->response([
                "error" => true,
                "message" => "Please enter the reply"
            ]);
        }

        Hook::exec('activitylog', [
            'object' => 'feedback',
            'object_id' => $rating->id,
            'event' => 'feedback_reply'
        ]);

        try {
            $order_url = Configuration::get('SITE_DOMAIN', $order->site_id) . '/dashboard/order/' . $order->id;

            $mailParams = array(
                'order_url' => $order_url,
                'order_domain' => Configuration::get('SITE_DOMAIN', $order->site_id),
                'order_id' => $order->id,
                'order_title' => $order->title,
                'username' => $customer->name ? $customer->name : 'Customer #' . $customer->id,
                'customer_feedabck' => $rating->comment,    
                'support_reply' => $reply
            );

            if(!$customer->is_anonymous) {
                Mail::send(
                    MailTemplate::TEMPLATE_ORDER_FEEDBACK_REPLY,
                    'Thank you for your feedback',
                    $mailParams,
                    $customer->email,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    $customer->site_id
                );  
            }
        } catch( \Exception $e ) {
        }

        return $this->modal("SUCCESS", "Reply Sent", "Reply was successfuly sent to the customer");
    }
}