<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Proxim\Configuration;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\ArticleWriting\ContentFeel;
use Proxim\Preference\ArticleWriting\EnglishType;
use Proxim\Preference\ArticleWriting\WritingStyle;
use Proxim\Preference\ArticleWriting\WritingTone;
use Proxim\Preference\ArticleWriting\TargetAudience;
use Proxim\Preference\Deadline;
use Proxim\Preference\Deliverable;
use Proxim\Preference\Descriptor;
use Proxim\Preference\Discipline;
use Proxim\Preference\ItemPricing;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Preference\UploadType;
use Proxim\Site\Site;
use Proxim\Validate;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Util\Formatting;
use Proxim\Util\StringUtils;

class Preferences extends Controller {
    public function index() {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();
        $payload = $app->request->post();

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $do = ArrayUtils::get($params, 'do');

        switch($do) {
            case 'addAcademicDeadline':
            case 'editAcademicDeadline':
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $academic_level_id = (int) ArrayUtils::get($payload, 'academic_level_id');
                $name = ArrayUtils::get($payload, 'name');
                $price_per_page = (float) ArrayUtils::get($payload, 'price_per_page');
                $hrs_customer = (int) ArrayUtils::get($payload, 'hrs_customer');
                $hrs_writer = (int) ArrayUtils::get($payload, 'hrs_writer');
                $min_pages = (int) ArrayUtils::get($payload, 'min_pages');

                if(!$name || !Validate::isGenericName($name)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the name'
                    ]);	
                }

                if(!Validate::isPrice($price_per_page)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the price per page'
                    ]);	
                }

                if(!Validate::isInt($hrs_customer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Customer'
                    ]);	
                }

                if(!Validate::isInt($hrs_writer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Writer'
                    ]);	
                } 

                $update = false;
                if(ArrayUtils::has($payload, 'deadline_id')) {
                    $deadline_id = (int) ArrayUtils::get($payload, 'deadline_id');
                    $deadline = new Deadline( $deadline_id );
                    if(Validate::isLoadedObject($deadline)) {
                        $update = true;
                    }
                } else {
                    $deadline = new Deadline();
                    $deadline->site_id = $site_id;
                    $deadline->academic_level_id = $academic_level_id;
                }

                $deadline->is_academic_writing = true;
                $deadline->name = $name;
                $deadline->price_per_page = $price_per_page;
                $deadline->hrs_customer = $hrs_customer;
                $deadline->hrs_writer = $hrs_writer;
                $deadline->min_pages = $min_pages;

                if($update) {
                    $deadline->update();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Deadline updated successfully'
                    ]);	
                } else {
                    $deadline->add();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_add'
                    ]);

		            return $this->modal('SUCCESS', "Added successfully", "Deadline added successfully");	
                }

                break;

            case 'addProgrammingDeadline':
            case 'editProgrammingDeadline':
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $name = ArrayUtils::get($payload, 'name');
                $price_modifier = (float) ArrayUtils::get($payload, 'price_modifier');
                $hrs_customer = (int) ArrayUtils::get($payload, 'hrs_customer');
                $hrs_writer = (int) ArrayUtils::get($payload, 'hrs_writer');

                if(!$name || !Validate::isGenericName($name)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the name'
                    ]);	
                }

                if(!Validate::isPrice($price_modifier)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the price'
                    ]);	
                }

                if(!Validate::isInt($hrs_customer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Customer'
                    ]);	
                }

                if(!Validate::isInt($hrs_writer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Writer'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'deadline_id')) {
                    $deadline_id = (int) ArrayUtils::get($payload, 'deadline_id');
                    $deadline = new Deadline( $deadline_id );
                    if(Validate::isLoadedObject($deadline)) {
                        $update = true;
                    }
                } else {
                    $deadline = new Deadline();
                    $deadline->site_id = $site_id;
                }

                $deadline->is_programming = true;
                $deadline->name = $name;
                $deadline->price_modifier = $price_modifier;
                $deadline->hrs_customer = $hrs_customer;
                $deadline->hrs_writer = $hrs_writer;

                if($update) {
                    $deadline->update();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Deadline updated successfully'
                    ]);	
                } else {
                    $deadline->add();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Deadline added successfully");	
                }
                break;

            case 'addCalculationsDeadline':
            case 'editCalculationsDeadline':
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $name = ArrayUtils::get($payload, 'name');
                $price_modifier = (float) ArrayUtils::get($payload, 'price_modifier');
                $hrs_customer = (int) ArrayUtils::get($payload, 'hrs_customer');
                $hrs_writer = (int) ArrayUtils::get($payload, 'hrs_writer');

                if(!$name || !Validate::isGenericName($name)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the name'
                    ]);	
                }

                if(!Validate::isPrice($price_modifier)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the price'
                    ]);	
                }

                if(!Validate::isInt($hrs_customer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Customer'
                    ]);	
                }

                if(!Validate::isInt($hrs_writer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Writer'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'deadline_id')) {
                    $deadline_id = (int) ArrayUtils::get($payload, 'deadline_id');
                    $deadline = new Deadline( $deadline_id );
                    if(Validate::isLoadedObject($deadline)) {
                        $update = true;
                    }
                } else {
                    $deadline = new Deadline();
                    $deadline->site_id = $site_id;
                }

                $deadline->is_calculations = true;
                $deadline->name = $name;
                $deadline->price_modifier = $price_modifier;
                $deadline->hrs_customer = $hrs_customer;
                $deadline->hrs_writer = $hrs_writer;

                if($update) {
                    $deadline->update();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Deadline updated successfully'
                    ]);	
                } else {
                    $deadline->add();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Deadline added successfully");	
                }
                break;

            case 'addArticlesDeadline':
            case 'editArticlesDeadline':
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $name = ArrayUtils::get($payload, 'name');
                $price_per_word = (float) ArrayUtils::get($payload, 'price_per_word');
                $hrs_customer = (int) ArrayUtils::get($payload, 'hrs_customer');
                $hrs_writer = (int) ArrayUtils::get($payload, 'hrs_writer');

                if(!$name || !Validate::isGenericName($name)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the name'
                    ]);	
                }

                if(!Validate::isPrice($price_per_word)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the price per word'
                    ]);	
                }

                if(!Validate::isInt($hrs_customer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Customer'
                    ]);	
                }

                if(!Validate::isInt($hrs_writer)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the Hours Writer'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'deadline_id')) {
                    $deadline_id = (int) ArrayUtils::get($payload, 'deadline_id');
                    $deadline = new Deadline( $deadline_id );
                    if(Validate::isLoadedObject($deadline)) {
                        $update = true;
                    }
                } else {
                    $deadline = new Deadline();
                    $deadline->site_id = $site_id;
                }

                $deadline->is_article_writing = true;
                $deadline->name = $name;
                $deadline->price_per_page = $price_per_word;
                $deadline->hrs_customer = $hrs_customer;
                $deadline->hrs_writer = $hrs_writer;

                if($update) {
                    $deadline->update();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Deadline updated successfully'
                    ]);	
                } else {
                    $deadline->add();

                    Hook::exec('activitylog', [
                        'object' => 'deadline',
                        'object_id' => $deadline->id,
                        'event' => 'deadline_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Deadline added successfully");	
                }

                break;

            case 'addAcademicLevel':
            case 'editAcademicLevel':
                $title = ArrayUtils::get($payload, 'title');
                $short_title = ArrayUtils::get($payload, 'short_title');
                $degree = ArrayUtils::get($payload, 'degree');
                $site_ids = (array) ArrayUtils::get($payload, 'site_ids', array());

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                if($short_title && !Validate::isGenericName($short_title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter a valid short title'
                    ]);	
                }

                if($degree && !Validate::isGenericName($degree)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter a valid degree'
                    ]);	
                }

                if(!$short_title) {
                    $short_title = $degree;
                }

                if(!$degree) {
                    $degree = $degree . ' degree';
                }

                $update = false;
                if(ArrayUtils::has($payload, 'academic_level_id')) {
                    $academic_level_id = (int) ArrayUtils::get($payload, 'academic_level_id');
                    $academic_level = new AcademicLevel( $academic_level_id );
                    if(Validate::isLoadedObject($academic_level)) {
                        $update = true;
                    }
                } else {
                    $academic_level = new AcademicLevel();
                }

                $academic_level->title = $title;
                $academic_level->short_title = $short_title;
                $academic_level->degree = $degree;

                if($update) {
                    $academic_level->update();

                    Db::getInstance()->delete('academic_level_site', 'academic_level_id = ' . (int) $academic_level->id);

                    foreach($site_ids as $site_id) {
                        $site = new Site((int) $site_id);
                        if(Validate::isLoadedObject($site)) {
                            Db::getInstance()->insert('academic_level_site', [
                                'academic_level_id' => $academic_level->id,
                                'site_id' => $site->id
                            ], false, false, Db::ON_DUPLICATE_KEY);
                        }
                    }

                    Hook::exec('activitylog', [
                        'object' => 'academic_level',
                        'object_id' => $academic_level->id,
                        'event' => 'academic_level_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Academic level updated successfully'
                    ]);	
                } else {
                    $academic_level->add();

                    foreach($site_ids as $site_id) {
                        $site = new Site((int) $site_id);
                        if(Validate::isLoadedObject($site)) {
                            Db::getInstance()->insert('academic_level_site', [
                                'academic_level_id' => $academic_level->id,
                                'site_id' => $site->id
                            ], false, false, Db::ON_DUPLICATE_KEY);
                        }
                    }

                    Hook::exec('activitylog', [
                        'object' => 'academic_level',
                        'object_id' => $academic_level->id,
                        'event' => 'academic_level_add'
                    ]);

		            return $this->modal('SUCCESS', "Added successfully", "Academic level added successfully");	
                }
                break;
            
            case 'addPaperType':
            case 'editPaperType':
                $title = ArrayUtils::get($payload, 'title');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'paper_type_id')) {
                    $paper_type_id = (int) ArrayUtils::get($payload, 'paper_type_id');
                    $paper_type = new PaperType( $paper_type_id );
                    if(Validate::isLoadedObject($paper_type)) {
                        $update = true;
                    }
                } else {
                    $paper_type = new PaperType();
                }

                $paper_type->title = $title;
                $paper_type->is_academic_writing = ArrayUtils::has($payload, 'is_academic_writing') ? true : false;
                $paper_type->is_article_writing = ArrayUtils::has($payload, 'is_article_writing') ? true : false;

                if($update) {
                    $paper_type->update();

                    Hook::exec('activitylog', [
                        'object' => 'paper_type',
                        'object_id' => $paper_type->id,
                        'event' => 'paper_type_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Paper type updated successfully'
                    ]);	
                } else {
                    $paper_type->add();

                    Hook::exec('activitylog', [
                        'object' => 'paper_type',
                        'object_id' => $paper_type->id,
                        'event' => 'paper_type_add'
                    ]);

		            return $this->modal('SUCCESS', "Added successfully", "Paper type added successfully");	
                }
                break;

            case 'addPaperFormat':
            case 'editPaperFormat':
                $title = ArrayUtils::get($payload, 'title');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }
                
                $update = false;
                if(ArrayUtils::has($payload, 'paper_format_id')) {
                    $paper_format_id = (int) ArrayUtils::get($payload, 'paper_format_id');
                    $paper_format = new PaperFormat( $paper_format_id );
                    if(Validate::isLoadedObject($paper_format)) {
                        $update = true;
                    }
                } else {
                    $paper_format = new PaperFormat();
                }

                $paper_format->title = $title;

                if($update) {
                    $paper_format->update();

                    Hook::exec('activitylog', [
                        'object' => 'paper_format',
                        'object_id' => $paper_format->id,
                        'event' => 'paper_format_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Paper format updated successfully'
                    ]);	
                } else {
                    $paper_format->add();

                    Hook::exec('activitylog', [
                        'object' => 'paper_format',
                        'object_id' => $paper_format->id,
                        'event' => 'paper_format_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Paper format added successfully");	
                }

                break;

            case 'addUploadType':
            case 'editUploadType':
                $title = ArrayUtils::get($payload, 'title');
                $customer = ArrayUtils::has($payload, 'customer');
                $writer = ArrayUtils::has($payload, 'writer');
                $editor = ArrayUtils::has($payload, 'editor');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                if(!$customer && !$writer && !$editor) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select customer, writer or editor'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'upload_type_id')) {
                    $upload_type_id = (int) ArrayUtils::get($payload, 'upload_type_id');
                    $upload_type = new UploadType( $upload_type_id );
                    if(Validate::isLoadedObject($upload_type)) {
                        $update = true;
                    }
                } else {
                    $upload_type = new UploadType();
                }

                $upload_type->title = $title;
                $upload_type->customer = $customer;
                $upload_type->writer = $writer;
                $upload_type->editor = $editor;

                if($update) {
                    $upload_type->update();

                    Hook::exec('activitylog', [
                        'object' => 'upload_type',
                        'object_id' => $upload_type->id,
                        'event' => 'upload_type_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Upload type updated successfully'
                    ]);	
                } else {
                    $upload_type->add();

                    Hook::exec('activitylog', [
                        'object' => 'upload_type',
                        'object_id' => $upload_type->id,
                        'event' => 'upload_type_add'
                    ]);

		            return $this->modal('SUCCESS', "Added successfully", "Upload type added successfully");	
                }

                break;

            case 'addDiscipline':
            case 'editDiscipline':
                $title = ArrayUtils::get($payload, 'title');
                $is_complex_assignment = (bool) ArrayUtils::has($payload, 'is_complex_assignment');
                $is_academic_writing = (bool) ArrayUtils::has($payload, 'is_academic_writing');
                $is_programming = (bool) ArrayUtils::has($payload, 'is_programming');
                $is_calculations = (bool) ArrayUtils::has($payload, 'is_calculations');
                $is_article_writing = (bool) ArrayUtils::has($payload, 'is_article_writing');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'discipline_id')) {
                    $discipline_id = (int) ArrayUtils::get($payload, 'discipline_id');
                    $discipline = new Discipline( $discipline_id );
                    if(Validate::isLoadedObject($discipline)) {
                        $update = true;
                    }
                } else {
                    $discipline = new Discipline();
                }

                if(ArrayUtils::has($payload, 'discipline_group')) {
                    $discipline_group = (int) ArrayUtils::get($payload, 'discipline_group');
                    $discipline_group = new Discipline( $discipline_group );
                    if(Validate::isLoadedObject($discipline_group)) {
                        $discipline->group_id = $discipline_group->id;
                    }
                }

                $discipline->title = $title;
                $discipline->is_complex_assignment = $is_complex_assignment;
                $discipline->is_academic_writing = $is_academic_writing;
                $discipline->is_programming = $is_programming;
                $discipline->is_calculations = $is_calculations;
                $discipline->is_article_writing = $is_article_writing;

                if($update) {
                    $discipline->update();

                    Hook::exec('activitylog', [
                        'object' => 'discipline',
                        'object_id' => $discipline->id,
                        'event' => 'discipline_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Discipline updated successfully'
                    ]);	
                } else {
                    $discipline->add();

                    Hook::exec('activitylog', [
                        'object' => 'discipline',
                        'object_id' => $discipline->id,
                        'event' => 'discipline_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Discipline added successfully");	
                }

                break;

            case 'addCategory':
            case 'editCategory':
                $title = ArrayUtils::get($payload, 'title');
                $is_complex_assignment = (bool) ArrayUtils::has($payload, 'is_complex_assignment');

                if(!$title || !Validate::isGenericName($title)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'discipline_id')) {
                    $discipline_id = (int) ArrayUtils::get($payload, 'discipline_id');
                    $discipline = new Discipline( $discipline_id );
                    if(Validate::isLoadedObject($discipline)) {
                        $update = true;
                    }
                } else {
                    $discipline = new Discipline();
                }

                if(ArrayUtils::has($payload, 'discipline_group')) {
                    $discipline_group = (int) ArrayUtils::get($payload, 'discipline_group');
                    $discipline_group = new Discipline( $discipline_group );
                    if(Validate::isLoadedObject($discipline_group)) {
                        $discipline->group_id = $discipline_group->id;
                    }
                }

                $discipline->title = $title;
                $discipline->is_complex_assignment = $is_complex_assignment;
                $discipline->is_article_writing = 1;

                if($update) {
                    $discipline->update();

                    Hook::exec('activitylog', [
                        'object' => 'discipline',
                        'object_id' => $discipline->id,
                        'event' => 'discipline_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Category updated successfully'
                    ]);	
                } else {
                    $discipline->add();

                    Hook::exec('activitylog', [
                        'object' => 'discipline',
                        'object_id' => $discipline->id,
                        'event' => 'discipline_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Category added successfully");	
                }

                break;

            case 'addDeliverable':
            case 'editDeliverable':
                $name = ArrayUtils::get($payload, 'name');

                if(!$name || !Validate::isGenericName($name)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the name'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'deliverable_id')) {
                    $deliverable_id = (int) ArrayUtils::get($payload, 'deliverable_id');
                    $deliverable = new Deliverable( $deliverable_id );
                    if(Validate::isLoadedObject($deliverable)) {
                        $update = true;
                    }
                } else {
                    $deliverable = new Deliverable();
                }

                $deliverable->name = $name;

                if(ArrayUtils::has($payload, 'slug')) {
                    $slug = ArrayUtils::get($payload, 'slug');
                    $deliverable->slug = Formatting::sanitize_title_with_dashes($slug);
                } else {
                    $deliverable->slug = Formatting::sanitize_title_with_dashes($name);
                }

                if($update) {
                    $deliverable->update();

                    Hook::exec('activitylog', [
                        'object' => 'deliverable',
                        'object_id' => $deliverable->id,
                        'event' => 'deliverable_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Deliverable updated successfully'
                    ]);	
                } else {
                    $deliverable->add();

                    Hook::exec('activitylog', [
                        'object' => 'deliverable',
                        'object_id' => $deliverable->id,
                        'event' => 'deliverable_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Deliverable added successfully");	
                }
                
                break;

            case 'addDescriptor':
            case 'editDescriptor':
                $discipline_id = ArrayUtils::get($payload, 'discipline_id');
                $example = ArrayUtils::get($payload, 'example');
                $task_size_id = (int) ArrayUtils::get($payload, 'task_size');
                $deliverables = ArrayUtils::get($payload, 'deliverables');
                $is_programming = (bool) ArrayUtils::has($payload, 'is_programming');
                $is_calculations = (bool) ArrayUtils::has($payload, 'is_calculations');
                
                if(!$example || !Validate::isGenericName($example)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the example'
                    ]);	
                }

                if(!$task_size_id) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select the task size'
                    ]);	
                }

                if(!$deliverables || (is_array($deliverables) && empty($deliverables))) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select at least one deliverable'
                    ]);	
                }

                $discipline = new Discipline( (int) $discipline_id );
                if(!Validate::isLoadedObject($discipline)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select a valid discipline'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'descriptor_id')) {
                    $descriptor_id = ArrayUtils::get($payload, 'descriptor_id');
                    $descriptor = new Descriptor( (int) $descriptor_id );
                    if(Validate::isLoadedObject($descriptor)) {
                        $update = true;
                    }
                } else {
                    $descriptor = new Descriptor();
                    $descriptor->is_programming = $is_programming;
                    $descriptor->is_calculations = $is_calculations;
                }

                $descriptor->discipline_id = $discipline->id;
                $descriptor->task_size_id = $task_size_id;
                $descriptor->example = $example;

                if($update) {
                    $descriptor->update();

                    foreach($deliverables as $deliverable_id) {
                        $deliverable = new Deliverable((int) $deliverable_id);
                        if(Validate::isLoadedObject($deliverable)) {
                            $descriptor_deliverable_id = Db::getInstance()->getValue('SELECT dd.descriptor_deliverable_id FROM ' . Db::prefix('descriptor_deliverable') . ' AS dd WHERE dd.descriptor_id = ' . (int) $descriptor->id . ' AND dd.deliverable_id = ' . (int) $deliverable->id);
                            if($descriptor_deliverable_id) {
                                Db::getInstance()->update(
                                    'descriptor_deliverable',
                                    array(
                                        'descriptor_id' => (int) $descriptor->id,
                                        'deliverable_id' => $deliverable->id,
                                        'date_add' => DateUtils::now()
                                    ),
                                    'descriptor_deliverable_id = ' . (int) $descriptor_deliverable_id
                                );
                            } else {
                                Db::getInstance()->insert(
                                    'descriptor_deliverable',
                                    array(
                                        'descriptor_id' => (int) $descriptor->id,
                                        'deliverable_id' => (int) $deliverable->id,
                                        'date_add' => DateUtils::now()
                                    )
                                );
                            }
                        }
                    }

                    Hook::exec('activitylog', [
                        'object' => 'descriptor',
                        'object_id' => $descriptor->id,
                        'event' => 'descriptor_upd'
                    ]);
    
                    return $this->modal('SUCCESS', "Updated successfully", "Descriptor updated successfully");
                } else {
                    $descriptor->add();

                    foreach($deliverables as $deliverable_id) {
                        $deliverable = new Deliverable((int) $deliverable_id);
                        if(Validate::isLoadedObject($deliverable)) {
                            Db::getInstance()->insert(
                                'descriptor_deliverable',
                                array(
                                    'descriptor_id' => (int) $descriptor->id,
                                    'deliverable_id' => (int) $deliverable->id,
                                    'date_add' => DateUtils::now()
                                )
                            );
                        }
                    }

                    Hook::exec('activitylog', [
                        'object' => 'descriptor',
                        'object_id' => $descriptor->id,
                        'event' => 'descriptor_add'
                    ]);
    
                    return $this->modal('SUCCESS', "Added successfully", "Descriptor added successfully");	
                }
                
                break;

            case 'addItemPricing':
            case 'editItemPricing':
                $discipline_id = ArrayUtils::get($payload, 'discipline_id');
                $task_size_id = (int) ArrayUtils::get($payload, 'task_size');
                $price = (float) ArrayUtils::get($payload, 'price');
                $is_programming = (bool) ArrayUtils::has($payload, 'is_programming');
                $is_calculations = (bool) ArrayUtils::has($payload, 'is_calculations');

                if(!$discipline_id) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select the discipline'
                    ]);	
                }

                if(!$task_size_id) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please select the task size'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'item_pricing_id')) {
                    $item_pricing_id = (int) ArrayUtils::get($payload, 'item_pricing_id');
                    $item_pricing = new ItemPricing( $item_pricing_id );
                    if(Validate::isLoadedObject($item_pricing)) {
                        $update = true;
                    }
                } else {
                    $item_pricing = new ItemPricing();
                    $item_pricing->is_programming = $is_programming;
                    $item_pricing->is_calculations = $is_calculations;
                }

                $item_pricing->is_programming = $is_programming;
                $item_pricing->is_calculations = $is_calculations;

                $item_pricing->discipline_id = $discipline_id;
                $item_pricing->task_size_id = $task_size_id;
                $item_pricing->price = $price;

                if($update) {
                    $item_pricing->update();

                    Hook::exec('activitylog', [
                        'object' => 'item_pricing',
                        'object_id' => $item_pricing->id,
                        'event' => 'item_pricing_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Item pricing updated successfully'
                    ]);	
                } else {
                    $item_pricing->add();

                    Hook::exec('activitylog', [
                        'object' => 'item_pricing',
                        'object_id' => $item_pricing->id,
                        'event' => 'item_pricing_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Item pricing added successfully");	
                }
                break;

            case 'addEnglishType':
            case 'editEnglishType':
                $title = ArrayUtils::get($payload, 'title');

                if(!$title) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'englishTypeId')) {
                    $englishTypeId = (int) ArrayUtils::get($payload, 'englishTypeId');
                    $englishType = new EnglishType( $englishTypeId );
                    if(Validate::isLoadedObject($englishType)) {
                        $update = true;
                    }
                } else {
                    $englishType = new EnglishType();
                }

                $englishType->title = $title;

                if($update) {
                    $englishType->update();

                    Hook::exec('activitylog', [
                        'object' => 'english_type',
                        'object_id' => $englishType->id,
                        'event' => 'english_type_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'English Type updated successfully'
                    ]);	
                } else {
                    $englishType->add();

                    Hook::exec('activitylog', [
                        'object' => 'english_type',
                        'object_id' => $englishType->id,
                        'event' => 'english_type_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "English Type added successfully");	
                }

                break;

            case 'addTargetAudience':
            case 'editTargetAudience':
                $title = ArrayUtils::get($payload, 'title');

                if(!$title) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'targetAudienceId')) {
                    $targetAudienceId = (int) ArrayUtils::get($payload, 'targetAudienceId');
                    $targetAudience = new TargetAudience( $targetAudienceId );
                    if(Validate::isLoadedObject($targetAudience)) {
                        $update = true;
                    }
                } else {
                    $targetAudience = new TargetAudience();
                }

                $targetAudience->title = $title;

                if($update) {
                    $targetAudience->update();

                    Hook::exec('activitylog', [
                        'object' => 'target_audience',
                        'object_id' => $targetAudience->id,
                        'event' => 'target_audience_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Target Audience updated successfully'
                    ]);	
                } else {
                    $targetAudience->add();

                    Hook::exec('activitylog', [
                        'object' => 'target_audience',
                        'object_id' => $targetAudience->id,
                        'event' => 'target_audience_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Target Audience added successfully");	
                }

                break;

            case 'addWritingTone':
            case 'editWritingTone':
                $title = ArrayUtils::get($payload, 'title');
                $description = ArrayUtils::get($payload, 'description');

                if(!$title) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                if(!$description) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the description'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'writingToneId')) {
                    $writingToneId = (int) ArrayUtils::get($payload, 'writingToneId');
                    $writingTone = new WritingTone( $writingToneId );
                    if(Validate::isLoadedObject($writingTone)) {
                        $update = true;
                    }
                } else {
                    $writingTone = new WritingTone();
                }

                $writingTone->title = $title;
                $writingTone->description = $description;

                if($update) {
                    $writingTone->update();

                    Hook::exec('activitylog', [
                        'object' => 'writing_tone',
                        'object_id' => $writingTone->id,
                        'event' => 'writing_tone_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Writing Tone updated successfully'
                    ]);	
                } else {
                    $writingTone->add();

                    Hook::exec('activitylog', [
                        'object' => 'writing_tone',
                        'object_id' => $writingTone->id,
                        'event' => 'writing_tone_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Writing Tone added successfully");	
                }
                break;

            case 'addWritingStyle':
            case 'editWritingStyle':
                $title = ArrayUtils::get($payload, 'title');
                $description = ArrayUtils::get($payload, 'description');

                if(!$title) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                if(!$description) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the description'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'writingStyleId')) {
                    $writingStyleId = (int) ArrayUtils::get($payload, 'writingStyleId');
                    $writingStyle = new WritingStyle( $writingStyleId );
                    if(Validate::isLoadedObject($writingStyle)) {
                        $update = true;
                    }
                } else {
                    $writingStyle = new WritingStyle();
                }

                $writingStyle->title = $title;
                $writingStyle->description = $description;

                if($update) {
                    $writingStyle->update();

                    Hook::exec('activitylog', [
                        'object' => 'writing_style',
                        'object_id' => $writingStyle->id,
                        'event' => 'writing_style_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Writing Style updated successfully'
                    ]);	
                } else {
                    $writingStyle->add();

                    Hook::exec('activitylog', [
                        'object' => 'writing_style',
                        'object_id' => $writingStyle->id,
                        'event' => 'writing_style_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Writing Style added successfully");	
                }
                break;

            case 'addContentFeel':
            case 'editContentFeel':
                $title = ArrayUtils::get($payload, 'title');
                $description = ArrayUtils::get($payload, 'description');

                if(!$title) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the title'
                    ]);	
                }

                if(!$description) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Please enter the description'
                    ]);	
                }

                $update = false;
                if(ArrayUtils::has($payload, 'contentFeelId')) {
                    $contentFeelId = (int) ArrayUtils::get($payload, 'contentFeelId');
                    $contentFeel = new ContentFeel( $contentFeelId );
                    if(Validate::isLoadedObject($contentFeel)) {
                        $update = true;
                    }
                } else {
                    $contentFeel = new ContentFeel();
                }

                $contentFeel->title = $title;
                $contentFeel->description = $description;

                if($update) {
                    $contentFeel->update();

                    Hook::exec('activitylog', [
                        'object' => 'content_feel',
                        'object_id' => $contentFeel->id,
                        'event' => 'content_feel_upd'
                    ]);

                    return $this->response([
                        'success' => true,
                        'message' => 'Content Feel updated successfully'
                    ]);	
                } else {
                    $contentFeel->add();

                    Hook::exec('activitylog', [
                        'object' => 'content_feel',
                        'object_id' => $contentFeel->id,
                        'event' => 'content_feel_add'
                    ]);

                    return $this->modal('SUCCESS', "Added successfully", "Content Feel added successfully");	
                }
                break;

            default:
                return $this->response([
                    'error' => true,
                    'message' => 'Oops! URL Not found'
                ]);	
                break;
        }
    }

    public function survey() {
		$app = $this->app;
		$payload = $app->request->post();
		$do = ArrayUtils::get($payload, 'do', 'info');

		switch($do) {
			case 'info':
				$sql = new DbQuery();
				$sql->select('s.*');
				$sql->from('site', 's');
				$result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

				$sites = array();

				$sites[] = array(
					'site_id' => PROX_SITE_ID,
					'node' => 'admin',
					'name' => Configuration::get('SITE_NAME', PROX_SITE_ID),
					'email' => Configuration::get('SITE_EMAIL', PROX_SITE_ID),
					'domain' => Configuration::get('SITE_DOMAIN', PROX_SITE_ID),
				);

				foreach($result as $site) {
					$sites[] = array(
						'site_id' => $site['site_id'],
						'node' => 'client',
						'name' => $site['name'],
						'email' => $site['email'],
						'domain' => $site['domain'],
						'customers' => $site['customers'],
						'orders' => $site['orders'],
						'payments' => $site['payments']
					);
				}

				return $this->response([
					"success" => true,
					"sites" => $sites
				]);
				break;

			case 'zigby':
				Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("customer") . ' DROP `total_spent`');
				Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("customer") . ' DROP `last_activity`');
				Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("order") . ' DROP `parent_id`');
        		Db::getInstance()->Execute('ALTER TABLE ' . Db::prefix("order") . ' DROP `currency_to_usd_rate`');
				break;
		}
	}
}