<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Mail;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\Controller;
use Proxim\Order\Order;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Hook;
use Proxim\User\EmployeeFile;

class Profile extends Controller {

    public function setTheme()
    {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;
        $cookie = $app->cookie;

        $mode = ArrayUtils::get($payload, 'mode');
        $cookie->night_mode = ($mode == "night")? true : false;
        $cookie->write();
        
        return $this->response([
			"success" => true
		]);
    }

    public function editProfile() {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        $edit = ArrayUtils::get($params, 'edit');

        switch( $edit ) {
            case 'basic':
                $first_name = ArrayUtils::get($payload, 'first_name');
                $middle_name = ArrayUtils::get($payload, 'middle_name');
                $last_name = ArrayUtils::get($payload, 'last_name');
                $gender = ArrayUtils::get($payload, 'gender');
                $phone = ArrayUtils::get($payload, 'phone');
                $timezone = ArrayUtils::get($payload, 'timezone');
                $date_of_birth = ArrayUtils::get($payload, 'date_of_birth');
                $country = ArrayUtils::get($payload, 'country');
                $city = ArrayUtils::get($payload, 'city');

                $email_notifications = ArrayUtils::has($payload, 'email_notifications') ? true :false;
                 
                if ( !Validate::isName($first_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid first name"
                    ]);
                }
        
                if ( Tools::strlen($first_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your first name must be less than 20 characters long. Please try another"
                    ]);
                }
        
                if ( Tools::strlen($first_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your first name must be at least 3 characters long. Please try another"
                    ]);
                }
        
                if( $middle_name ) {
                    if ( !Validate::isName($middle_name) ) {
                        return $this->response([
                            "error" => true,
                            "message" => "Enter a valid middle name"
                        ]);
                    }
            
                    if ( Tools::strlen($middle_name) > 20 ) {
                        return $this->response([
                            "error" => true,
                            "message" => "Your middle name must be less than 20 characters long. Please try another"
                        ]);
                    }
            
                    if ( Tools::strlen($middle_name) < 3 ) {
                        return $this->response([
                            "error" => true,
                            "message" => "Your middle name must be at least 3 characters long. Please try another"
                        ]);
                    }
                }
        
                if ( !Validate::isName($last_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid last name"
                    ]);
                }
        
                if ( Tools::strlen($last_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your last name must be less than 20 characters long. Please try another"
                    ]);
                }

                if ( Tools::strlen($last_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your last name must be at least 3 characters long. Please try another"
                    ]);
                }
        
                if($phone && !Validate::isPhoneNumber($phone)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid phone number"
                    ]);
                }

                if ( ArrayUtils::has($payload, 'status') ) {
                    $status = (int) ArrayUtils::get($payload, 'status');
                    $user->employee_status = $status; 
                }

                $user->first_name = $first_name; 
                $user->middle_name = $middle_name; 
                $user->last_name = $last_name; 
                $user->phone = $phone;
                $user->gender = $gender;
                $user->date_of_birth = $date_of_birth;
                $user->country = $country;
                $user->city = $city;

                if(ArrayUtils::has($payload, 'timezone')) {
                    $user->timezone = $timezone;
                }

                $user->email_notifications = $email_notifications;
                $user->update();

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $user->id,
                    'event' => 'basic_settings'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "Your profile has been updated"
                ]);

                break;

            case 'qualifications':
                $native_language = ArrayUtils::get($payload, 'native_language');
                $other_languages = ArrayUtils::get($payload, 'other_languages', []);
                $highest_academic_degree = ArrayUtils::get($payload, 'highest_academic_degree');
                $academic_degree_major = ArrayUtils::get($payload, 'academic_degree_major');
                $proficient_disciplines = ArrayUtils::get($payload, 'proficient_disciplines', []);
                $software = ArrayUtils::get($payload, 'software', []);

                if(!$native_language) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your native language"
                    ]);
                }

                if(!$highest_academic_degree) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your highest academic degree"
                    ]);
                }

                if(!$academic_degree_major) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter your degree major"
                    ]);
                }

                if(!$proficient_disciplines && !is_array($proficient_disciplines)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the disciplines you are proficient in"
                    ]);
                }

                if(!$software && !is_array($software)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please indicate software you have access to and/or programming language you are expert in"
                    ]);
                }

                $user->native_language = $native_language;
                $user->other_languages = implode(',', $other_languages);
                $user->highest_academic_degree = $highest_academic_degree;
                $user->academic_degree_major = $academic_degree_major;
                $user->proficient_disciplines = implode(',', $proficient_disciplines);
                $user->software = implode(',', $software);
                $user->is_profile_completed = true;
                $user->update();

                return $this->response([
                    "success" => true,
                    "message" => "Qualifications updated successfully"
                ]);
                break;

            case 'about':
                    $about_me = ArrayUtils::get($payload, 'about_me');
                    $facebook_profile = ArrayUtils::get($payload, 'facebook_profile');
                    $linkedin_profile = ArrayUtils::get($payload, 'linkedin_profile');
                    
                    $user->about = $about_me;
                    $user->facebook_profile = $facebook_profile;
                    $user->linkedin_profile = $linkedin_profile;
                    
                    $user->update();

                    Hook::exec('activitylog', [
                        'object' => 'employee',
                        'object_id' => $user->id,
                        'event' => 'edit_employee_account'
                    ]);
    
                    return $this->response([
                        'success' => true,
                        'message' => 'Acount updated successfully'
                    ]);
                break;

            case 'working_status':
                $employee_status = (int) ArrayUtils::get($payload, 'employee_status');
                $user->employee_status = $employee_status;

                $user->available_247 = ArrayUtils::has($payload, 'available_247') ? true : false;
                $user->available_urgent_orders = ArrayUtils::has($payload, 'available_urgent_orders') ? true : false;

                $user->update();

                return $this->response([
                    "success" => true,
                    "message" => "Status updated successfully"
                ]);
                
                $user->update();

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $user->id,
                    'event' => 'about'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "Your about has been updated"
                ]);
                break;

            case 'qualifications':
                $native_language = ArrayUtils::get($payload, 'native_language');
                $other_languages = ArrayUtils::get($payload, 'other_languages', []);
                $highest_academic_degree = ArrayUtils::get($payload, 'highest_academic_degree');
                $academic_degree_major = ArrayUtils::get($payload, 'academic_degree_major');
                $proficient_disciplines = ArrayUtils::get($payload, 'proficient_disciplines', []);
                $software = ArrayUtils::get($payload, 'software', []);

                if(!$native_language) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your native language"
                    ]);
                }

                if(!$highest_academic_degree) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select your highest academic degree"
                    ]);
                }

                if(!$academic_degree_major) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter your degree major"
                    ]);
                }

                if(!$proficient_disciplines && !is_array($proficient_disciplines)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please select the disciplines you are proficient in"
                    ]);
                }

                if(!$software && !is_array($software)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please indicate software you have access to and/or programming language you are expert in"
                    ]);
                }

                $user->native_language = $native_language;
                $user->other_languages = implode(',', $other_languages);
                $user->highest_academic_degree = $highest_academic_degree;
                $user->academic_degree_major = $academic_degree_major;
                $user->proficient_disciplines = implode(',', $proficient_disciplines);
                $user->software = implode(',', $software);
                $user->is_profile_completed = true;
                $user->update();

                return $this->response([
                    "success" => true,
                    "message" => "Your qualifications have been updated"
                ]);
                break;

            case 'timezone':
                $timezone = ArrayUtils::get($payload, 'timezone');
                $user->timezone = $timezone;
                $user->update();

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $user->id,
                    'event' => 'timezone_settings'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "Your timezone has been updated"
                ]);
                break;

            case 'email':
                $email = ArrayUtils::get($payload, 'email');
                    
                if ( !$email || !Validate::isEmail($email) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid email address"
                    ]);
                }
                
                if( $user->email != $email ) {
                    $user_exists = $user->getByEmail($email);
                    if (  Validate::isLoadedObject($user_exists) ) {
                        return $this->response([
                            "error" => true,
                            "message" => "Sorry, it looks like <strong>".$email."</strong> belongs to an existing account"
                        ]);
                    }
                }

                $user->email = $email;
                $user->update();

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $user->id,
                    'event' => 'email_settings'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "Your email has been updated"
                ]);
                break;

            case 'password':
                $current = ArrayUtils::get($payload, 'current');
                $new = ArrayUtils::get($payload, 'new');
                $confirm = ArrayUtils::get($payload, 'confirm');
                    
                /* validate all fields */
                if(!$current || !$new || !$confirm ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'You must fill in all of the fields'
                    ]);
                }

                $crypto = new Hashing();
                /* validate current password */
                if ( !$crypto->checkHash($current, $user->password) ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Your old password is incorrect'
                    ]);
                }
                
                /* validate new password */
                if($new != $confirm ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Your passwords do not match'
                    ]);
                }

                if( !Validate::isPasswd($new) ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'New password must be at least 6 characters long. Please try another'
                    ]);
                }

                $user->password = $crypto->hash(
                    $new,
                    PROX_COOKIE_KEY
                );
                $user->update();

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $user->id,
                    'event' => 'password_settings'
                ]);
                
                return $this->response([
                    "success" => true,
                    "message" => "Your password has been updated"
                ]);
                break;

            case 'notifications_sound':
                $notifications_sound = (bool) ArrayUtils::get($payload, 'notifications_sound');
                $user->notification_sound = $notifications_sound;
                $user->update();

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $user->id,
                    'event' => 'notifications_sound_settings'
                ]);

                return $this->response([
                    "success" => true
                ]);
                break;

            case 'upload':
                $files = ArrayUtils::get($payload, 'files');

                foreach ($files as $uploadedFile) {
                    $file_id = (int) ArrayUtils::get($uploadedFile, 'id');
                    $employeeFile = new EmployeeFile( $file_id );
                    if(Validate::isLoadedObject($employeeFile)) {
                        $employeeFile->employee_id = $user->id;
                        $employeeFile->uploader_id = $user->id;
                        $employeeFile->update();
                    }
                }

		        return $this->modal('SUCCESS', "Upload Successful", "The files have been uploaded successfuly");	
                break;
                
            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }

    public function removePicture() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        $handle = ArrayUtils::get($payload, 'handle');

        switch( $handle ) {
            case 'picture-user':
                $user->avatar = null;
                $user->update();

                /* return */
                return $this->response([
                    'file' => $user->getPicture(null, $user->gender)
                ]);
			    break;
        }
    }

    public function session() {
        $app = $this->app;
        $payload = $app->request->post();

        $handle = ArrayUtils::get($payload, 'handle');

        switch( $handle ) {
            case 'session':
                $sid = ArrayUtils::get($payload, 'id');
                Db::getInstance()->delete('user_session', 'session_id = ' . (int) $sid );

                return $this->response([
                    'callback' => 'window.location.reload();'
                ]);	
                break;

            case 'sessions':
                if(ArrayUtils::has($payload, 'uid')) {
                    $uid = ArrayUtils::get($payload, 'uid');
                    $user = new Employee( (int) $uid );
                    if(Validate::isLoadedObject($user)) {
                        Db::getInstance()->delete('user_session', 'user_id = ' . (int) $user->id );
                    }

                } else {
                    $user = $app->user;
                    Db::getInstance()->delete('user_session', 'user_id = ' . (int) $user->id );
                }

                return $this->response([
                    'callback' => 'window.location.reload();'
                ]);	
			    break;
        }
    }

    public function requestPayment() {
        $app = $this->app;
        $user = $app->user;

        $sql = new DbQuery();
        $sql->select('o.order_id');
        $sql->from('order', 'o');

        if ( $user->is_writer ) {
            $sql->where('o.`writer_id` = ' . (int) $user->id );
            $sql->where('o.`is_writer_paid` = 0' );
        } elseif ( $user->is_editor ) {
            $sql->where('o.`editor_id` = ' . (int) $user->id );
            $sql->where('o.`is_editor_paid` = 0' );
        } elseif ( $user->is_sub_admin ) {
            $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
            $sql->where('o.`is_ordermanager_paid` = 0' );
        }

        $statuses = array( Order::FINISHED );

        if( Configuration::get('PAY_DELIVERED_ORDERS') ) {
            $statuses[] = Order::DELIVERED;
        } 

        $statuses = implode(',', $statuses);
        $sql->where("o.`status_id` IN ( $statuses )");
        $sql->leftJoin('payment_request_order', 'pro', 'pro.order_id = o.order_id AND pro.user_id = ' . (int) $user->id );
        $sql->where( 'pro.order_id IS NULL' );
        $sql->orderBy( 'o.finished_at DESC' );
        $orders = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $total_pay = $completed_orders_pay = $done_orders_pay = $employee_credit = $employee_fine = 0;
        $orders_id = array();

        foreach( $orders as $order ) {
            $order = new Order( (int) $order['order_id'] );
            if( Validate::isLoadedObject($order) ) {
                if($order->status_id == Order::FINISHED) {
                    if($user->is_sub_admin) {
                        $completed_orders_pay += $order->ordermanager_pay;
                    } elseif($user->is_editor) {
                        $completed_orders_pay += $order->editor_pay;
                    } elseif($user->is_writer) {
                        $completed_orders_pay += $order->writer_pay;
                    }
                } elseif ( $order->status_id == Order::DELIVERED ) {
                    if($user->is_sub_admin) {
                        $done_orders_pay += $order->ordermanager_pay;
                    } elseif($user->is_editor) {
                        $done_orders_pay += $order->editor_pay;
                    } elseif($user->is_writer) {
                        $done_orders_pay += $order->writer_pay;
                    }
                }

                $orders_id[] = $order->id;
            }
        }

        $total_pay = $completed_orders_pay+$done_orders_pay;

        $additionalPaymentsByModule = Hook::exec('actionGetEmployeeAdditionalPayments', array(
            'employee_id' => $user->id,
            'requested' => false,
            'resolved' => false
        ), null, true);

        if(is_array($additionalPaymentsByModule) && !empty($additionalPaymentsByModule)) {
            foreach($additionalPaymentsByModule as $moduleName => $additionalPayments) {
                foreach($additionalPayments as $additionalPayment) {
                    if($additionalPayment['type'] == 'credit') {
                        $employee_credit += (float) $additionalPayment['amount'];
                    } elseif($additionalPayment['type'] == 'fine') {
                        $employee_fine += (float) $additionalPayment['amount'];
                    }
                }
            }
        }

        $total_pay = ($total_pay + $employee_credit) - $employee_fine;

        if($total_pay <= 0) {
	        return $this->modal("ERROR", "Payment Request Failed", "You don't have enough funds to send a payment request");
        }

        $now = DateUtils::now();
        $data = array(
            'user_id' => (int) $user->id,
            'total_done_pay' => $done_orders_pay,
            'total_credit' => $employee_credit,
            'total_fine' => $employee_fine,
            'total_completed_pay' => $completed_orders_pay,
            'total_pay' => $total_pay,
            'date_add' => $now,
            'date_upd' => $now,
        );

        $result = Db::getInstance()->insert('payment_request', $data, true);

        if(!$result) {
	        return $this->modal("ERROR", "Payment Request Failed", "Your payment request could not be sent at the moment. Try again later");
        }

        $payment_request_id = Db::getInstance()->Insert_ID();

        foreach( $orders_id as $order_id ) {
            $orders_request[] = array(
                'payment_request_id' => $payment_request_id,
                'user_id' => $user->id,
                'order_id' => $order_id
            );
        }

        Db::getInstance()->insert('payment_request_order', $orders_request, true);

        Hook::exec('actionUpdateRequestedAdditionalPayments', array(
            'employee_id' => $user->id,
            'requested' => true
        ));

	    return $this->modal("SUCCESS", "Payment Request Sent", "Your payment request has been sent");
    }
}
