<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Exception;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Site\Site;
use Proxim\Coupon;
use Proxim\Currency;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Order\Order;
use Proxim\Order\File;
use Proxim\Order\Message;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Hook;
use Proxim\Order\Rating;
use Proxim\Preference\Deadline;
use Proxim\Preference\Descriptor;
use Proxim\Preference\ItemPricing;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Preference\UploadType;
use Proxim\Uploader;
use Proxim\User\EmployeeFile;

class Users extends Controller {
    public function customers() {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $do = ArrayUtils::get($params, 'do');

        switch( $do ) {
            case 'add':
                $customer = new Customer();
                $name = ArrayUtils::get($payload, 'name');
                $email = ArrayUtils::get($payload, 'email');
                $password = ArrayUtils::get($payload, 'password');
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
        
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Select a valid site'
                    ]);
                }
        
                if ( !$email ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Email is required'
                    ]);
                }
        
                if(!validate::isEmail($email)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Enter a vlid Email address'
                    ]);
                }
        
                $exists = $customer->getByEmail( $email, null, $site->id );
                if ( Validate::isLoadedObject($exists) ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'E-mail is already registered at ' . $site->name 
                    ]);
                }
        
                /** Name Validation */
                if ( $name && !Validate::isName($name) ) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Enter a valid name'
                    ]);
                }
        
                if ( !Validate::isPasswd($password)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Password should be more than six characters'
                    ]);
                }
        
                $customer->google_analytics_id = Tools::randomGen(32);
                $customer->name = $name; 
                $customer->email = $email; 
                /** United Sates of America */
                $customer->country_id = (int) 840;
                $customer->site_id = $site->id;
                $customer->reg_date = DateUtils::now();
                
                $crypto = new Hashing();
                $customer->password = $crypto->hash(
                    $password,
                    PROX_COOKIE_KEY
                );
        
                $success = $customer->add(true, true);
                if ( !$success ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Sorry, there was a problem creating an account"
                    ]);	
                }

                Hook::exec('activitylog', [
                    'object' => 'customer',
                    'object_id' => $customer->id,
                    'event' => 'add_customer'
                ]);
        
                return $this->response([
                    'callback' => "window.location.href='/customers/edit/$customer->id';"
                ]);	

                break;

            case 'edit':
                $customer_id = ArrayUtils::get($payload, 'customer_id');

                $customer = new Customer( (int) $customer_id );
                if(!Validate::isLoadedObject($customer)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This customer does not exist"
                    ]);
                }
        
                $name = ArrayUtils::get($payload, 'name');
                $email = ArrayUtils::get($payload, 'email');
                $phone = ArrayUtils::get($payload, 'phone');
                $password = ArrayUtils::get($payload, 'password');
                $site_id = (int) ArrayUtils::get($payload, 'site_id');
                $total_spent = (float) ArrayUtils::get($payload, 'total_spent');
        
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Select a valid site'
                    ]);
                }

                if(!$customer->is_anonymous) {
                    if ( !$email ) {
                        return $this->response([
                            'error' => true,
                            'message' => 'Email is required'
                        ]);
                    }
            
                    if(!validate::isEmail($email)) {
                        return $this->response([
                            'error' => true,
                            'message' => 'Enter a vlid Email address'
                        ]);
                    }
            
                    if( $customer->email != $email ) {
                        $exists = $customer->getByEmail( $email, null, $site_id );
                        if ( Validate::isLoadedObject($exists) ) {
                            return $this->response([
                                'error' => true,
                                'message' => 'E-mail is already registered at ' . $site->name 
                            ]);
                        }
                    }
            
                    /** Name Validation */
                    if ( $name && !Validate::isName($name) ) {
                        return $this->response([
                            'error' => true,
                            'message' => 'Enter a valid name'
                        ]);
                    }
                    
                    $customer->name = $name; 
                    $customer->phone = $phone;
                    $customer->email = $email; 
                }
        
                if($total_spent && !Validate::isPrice($total_spent)) {
                    return $this->response([
                        'error' => true,
                        'message' => 'Enter a valid amount for total spent'
                    ]);
                }
        
                if ( $password ) {
                    if ( !Validate::isPasswd($password)) {
                        return $this->response([
                            'error' => true,
                            'message' => 'Password should be more than six characters'
                        ]);
                    }
        
                    $crypto = new Hashing();
                    $customer->password = $crypto->hash(
                        $password,
                        PROX_COOKIE_KEY
                    );
                }
        
                $customer->total_spent = $total_spent;
                $customer->site_id = $site_id; 
        
                $customer->update();

                Hook::exec('activitylog', [
                    'object' => 'customer',
                    'object_id' => $customer->id,
                    'event' => 'edit_customer'
                ]);
        
                return $this->response([
                    'success' => true,
                    'message' => 'Profile updated successfully'
                ]);
                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }

    public function employees( $do ) {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        switch( $do ) {
            case 'add':
                $employee_group = (int) ArrayUtils::get($payload, 'employee_group');
                $first_name = ArrayUtils::get($payload, 'first_name');
                $last_name = ArrayUtils::get($payload, 'last_name');
                $email = ArrayUtils::get($payload, 'email');
                $password = ArrayUtils::get($payload, 'password');
                $gender = ArrayUtils::get($payload, 'gender');

                if (!$employee_group) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid employee group"
                    ]);
                }

                if ( $first_name && !Validate::isName($first_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid first name"
                    ]);
                }

                if ( Tools::strlen($first_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your first name must be less than 20 characters long. Please try another"
                    ]);
                }
                
                if ( Tools::strlen($first_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your first name must be at least 3 characters long. Please try another"
                    ]);
                }
        
                if ( $last_name && !Validate::isName($last_name) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid last name"
                    ]);
                }
        
                if ( Tools::strlen($last_name) > 20 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your last name must be less than 20 characters long. Please try another"
                    ]);
                }
                if ( Tools::strlen($last_name) < 3 ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Your last name must be at least 3 characters long. Please try another"
                    ]);
                }

                if ( !$email || !Validate::isEmail($email) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid email address"
                    ]);
                }

                $exists = $user->getByEmail($email);
                if (  Validate::isLoadedObject($exists) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Sorry, it looks like <strong>".$email."</strong> belongs to an existing account"
                    ]);
                }

                if ( !$gender || !in_array($gender, array('male', 'female')) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Select a valid gender"
                    ]);
                }

                if ( !$password || !Validate::isPasswd($password) ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Password must be long than 6 characters"
                    ]);
                }

                $employee = new Employee();
                $employee->employee_group = $employee_group;
                $employee->email = $email;
                $employee->first_name = $first_name; 
                $employee->last_name = $last_name; 
                
                $crypto = new Hashing();
                $employee->password = $crypto->hash(
                    $password,
                    PROX_COOKIE_KEY
                );
                $employee->gender = $gender; 
                $employee->policy_accepted = 1;
                $employee->policy_accepted_at = DateUtils::now();
                $employee->reg_date = DateUtils::now();
                $success = $employee->add(true, true);
                if ( !$success ) {
                    return $this->response([
                        "error" => true,
                        "message" => "Sorry, there was a problem creating an account"
                    ]);	
                }

                Hook::exec('actionEmployeeAdded', ['employee' => $employee]);

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $employee->id,
                    'event' => 'add_employee'
                ]);
                
                return $this->response([
                    'callback' => 'window.location.href="/employees/edit/'.$employee->id.'";'
                ]);	
                break;

            case 'edit':
                $edit = ArrayUtils::get($params, 'edit');
                $employee_id = ArrayUtils::get($payload, 'employee_id');
        
                $employee = new Employee( (int) $employee_id );
                if(!Validate::isLoadedObject($employee)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This employee does not exist"
                    ]);
                }    
                
                Hook::exec('actionEmployeeEdited', ['employee' => $employee]);

                switch( $edit ) {
                    case 'account':
                        $writer_category = (int) ArrayUtils::get($payload, 'writer_category');
                        $academic_level = (int) ArrayUtils::get($payload, 'academic_level');
                        $password = ArrayUtils::get($payload, 'password');
                        $email_notifications = ArrayUtils::has($payload, 'email_notifications') ? true : false;
                        $started = ArrayUtils::has($payload, 'started') ? true : false;
                        $banned = ArrayUtils::has($payload, 'banned') ? true : false;
                        $number_of_takes = (int) ArrayUtils::get($payload, 'number_of_takes');

                        if(ArrayUtils::has($payload, 'email')) {
                            $email = ArrayUtils::get($payload, 'email');

                            if ( !$email ) {
                                return $this->response([
                                    'error' => true,
                                    'message' => 'Email is required'
                                ]);
                            }
                    
                            if(!validate::isEmail($email)) {
                                return $this->response([
                                    'error' => true,
                                    'message' => 'Enter a vlid Email address'
                                ]);
                            }
                    
                            if( $employee->email != $email ) {
                                $exists = $employee->getByEmail( $email );
                                if ( Validate::isLoadedObject($exists) ) {
                                    return $this->response([
                                        "error" => true,
                                        "message" => "Sorry, it looks like <strong>".$email."</strong> belongs to an existing account"
                                    ]);
                                }
                            }

                            $employee->email = $email;
                        }
        
                        if ($employee->id == Employee::START_ID && !$started ) {
                            return $this->response([
                                "error" => true,
                                "message" => "You cant ban or deactivate this user"
                            ]);
                        }
        
                        if ($employee->id == Employee::START_ID && $banned) {
                            return $this->response([
                                "error" => true,
                                "message" => "You cant ban or deactivate this user"
                            ]);
                        }

                        if(ArrayUtils::has($payload, 'employee_group')) {
                            $employee_group = (int) ArrayUtils::get($payload, 'employee_group');
                            if ($employee->id == Employee::START_ID && $employee->employee_group != $employee_group) {
                                return $this->response([
                                    "error" => true,
                                    "message" => "You cant change the Employee Group of this user"
                                ]);
                            }
                            
                            $employee->employee_group = $employee_group;
                        }
        
        
                        if ( $writer_category ) {
                            $employee->writer_category = $writer_category;
                        }
        
                        if ( $password ) {
                            if ( !Validate::isPasswd($password)) {
                                return $this->response([
                                    'error' => true,
                                    'message' => 'Password should be more than six characters'
                                ]);
                            }
                
                            $crypto = new Hashing();
                            $employee->password = $crypto->hash(
                                $password,
                                PROX_COOKIE_KEY
                            );
                        }

                        $employee->academic_level = $academic_level;
                        $employee->push_notifications = $email_notifications;
                        $employee->is_started = $started;
                        $employee->is_banned = $banned;
                        $employee->number_of_takes = $number_of_takes;
                        
                        $employee->update();

                        Hook::exec('activitylog', [
                            'object' => 'employee',
                            'object_id' => $employee->id,
                            'event' => 'edit_employee_account'
                        ]);
        
                        return $this->response([
                            'success' => true,
                            'message' => 'Acount updated successfully'
                        ]);
                        break;

                    case 'employee_cpp':
                        $cpp_type = ArrayUtils::get($payload, 'cpp_type');
                        $cpp_amount = (float) ArrayUtils::get($payload, 'cpp_amount');
                        $cpw_type = ArrayUtils::get($payload, 'cpw_type');
                        $cpw_amount = (float) ArrayUtils::get($payload, 'cpw_amount');

                        if($cpp_type == "system") $cpp_amount = 0;
                        if($cpw_type == "system") $cpw_amount = 0;

                        # cpp
                        $employee->cpp_type = $cpp_type;
                        $employee->cpp_amount = $cpp_amount;
                        # cpw
                        $employee->cpw_type = $cpw_type;
                        $employee->cpw_amount = $cpw_amount;
                        $employee->update();

                        Hook::exec('activitylog', [
                            'object' => 'employee',
                            'object_id' => $employee->id,
                            'event' => 'edit_employee_pay'
                        ]);
        
                        return $this->response([
                            'success' => true,
                            'message' => 'Employee pay updated successfully'
                        ]);
                        break;
        
                    case 'profile':
                        $first_name = ArrayUtils::get($payload, 'first_name');
                        $middle_name = ArrayUtils::get($payload, 'middle_name');
                        $last_name = ArrayUtils::get($payload, 'last_name');
                        $gender = ArrayUtils::get($payload, 'gender');
                        $phone = ArrayUtils::get($payload, 'phone'); 
                        $date_of_birth = ArrayUtils::get($payload, 'date_of_birth');
                        $country = ArrayUtils::get($payload, 'country');
                        $city = ArrayUtils::get($payload, 'city');
        
                        if ( $first_name && !Validate::isName($first_name) ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Enter a valid first name"
                            ]);
                        }
                        if ( Tools::strlen($first_name) > 20 ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Your first name must be less than 20 characters long. Please try another"
                            ]);
                        }
                        if ( Tools::strlen($first_name) < 3 ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Your first name must be at least 3 characters long. Please try another"
                            ]);
                        }
        
                        if ( $last_name && !Validate::isName($last_name) ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Enter a valid last name"
                            ]);
                        }
        
                        if ( Tools::strlen($last_name) > 20 ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Your last name must be less than 20 characters long. Please try another"
                            ]);
                        }
        
                        if ( Tools::strlen($last_name) < 3 ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Your last name must be at least 3 characters long. Please try another"
                            ]);
                        }
        
                        if ($middle_name) {
                            if ( $middle_name && !Validate::isName($middle_name) ) {
                                return $this->response([
                                    "error" => true,
                                    "message" => "Enter a valid middle name"
                                ]);
                            }
            
                            if ( Tools::strlen($middle_name) > 20 ) {
                                return $this->response([
                                    "error" => true,
                                    "message" => "Your middle name must be less than 20 characters long. Please try another"
                                ]);
                            }
                            
                            if ( Tools::strlen($middle_name) < 3 ) {
                                return $this->response([
                                    "error" => true,
                                    "message" => "Your middle name must be at least 3 characters long. Please try another"
                                ]);
                            }
                        }
        
                        if ( !$gender || !in_array($gender, array('male', 'female')) ) {
                            return $this->response([
                                "error" => true,
                                "message" => "Select a valid gender"
                            ]);
                        }

                        if($date_of_birth && !Validate::isDate($date_of_birth)) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the date of birth"
                            ]);
                        }

                        if($city && !Validate::isCityName($city)) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter the city"
                            ]);
                        }
        
                        if($country && !Validate::isCountryName($country)) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please select the county"
                            ]);
                        }
        
                        $employee->first_name = $first_name; 
                        $employee->last_name = $last_name; 
                        $employee->middle_name = $middle_name; 
                        $employee->phone = $phone; 
                        $employee->gender = $gender; 
                        $employee->date_of_birth = $date_of_birth;
                        $employee->country = $country;
                        $employee->city = $city;
                        $employee->update();

                        Hook::exec('activitylog', [
                            'object' => 'employee',
                            'object_id' => $employee->id,
                            'event' => 'edit_employee_profile'
                        ]);
        
                        return $this->response([
                            'success' => true,
                            'message' => 'Profile updated successfully'
                        ]);
                        break;

                    case 'email':
                        $email = ArrayUtils::get($payload, 'email');

                        if ( !$email ) {
                            return $this->response([
                                'error' => true,
                                'message' => 'Email is required'
                            ]);
                        }

                        if(!validate::isEmail($email)) {
                            return $this->response([
                                'error' => true,
                                'message' => 'Enter a vlid Email address'
                            ]);
                        }
                
                        if( $employee->email != $email ) {
                            $exists = $employee->getByEmail( $email );
                            if ( Validate::isLoadedObject($exists) ) {
                                return $this->response([
                                    "error" => true,
                                    "message" => "Sorry, it looks like <strong>".$email."</strong> belongs to an existing account"
                                ]);
                            }
                        }

                        $employee->email = $email;
                        $employee->update();

                        Hook::exec('activitylog', [
                            'object' => 'employee',
                            'object_id' => $employee->id,
                            'event' => 'edit_employee_email'
                        ]);
        
                        return $this->response([
                            'success' => true,
                            'message' => 'Email updated successfully'
                        ]);

                        break;

                    case 'about': 
                        $facebook_profile = ArrayUtils::get($payload, 'facebook_profile');
                        $linkedin_profile = ArrayUtils::get($payload, 'linkedin_profile');
                        $about_me = ArrayUtils::get($payload, 'about_me');

                        $employee->facebook_profile = $facebook_profile;
                        $employee->linkedin_profile = $linkedin_profile;
                        $employee->about = $about_me;
                        $employee->update();
        
                        Hook::exec('activitylog', [
                            'object' => 'employee',
                            'object_id' => $employee->id,
                            'event' => 'about'
                        ]);
        
                        return $this->response([
                            "success" => true,
                            "message" => "About updated successfully"
                        ]);
                        break;
        
                    case 'qualifications':
                        $native_language = ArrayUtils::get($payload, 'native_language');
                        $other_languages = ArrayUtils::get($payload, 'other_languages', []);
                        $highest_academic_degree = ArrayUtils::get($payload, 'highest_academic_degree');
                        $academic_degree_major = ArrayUtils::get($payload, 'academic_degree_major');
                        $proficient_disciplines = ArrayUtils::get($payload, 'proficient_disciplines', []);
                        $software = ArrayUtils::get($payload, 'software', []);
        
                        if(!$native_language) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please select your native language"
                            ]);
                        }
        
                        if(!$highest_academic_degree) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please select your highest academic degree"
                            ]);
                        }
        
                        if(!$academic_degree_major) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please enter your degree major"
                            ]);
                        }
        
                        if(!$proficient_disciplines && !is_array($proficient_disciplines)) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please select the disciplines you are proficient in"
                            ]);
                        }

                        if(!$software && !is_array($software)) {
                            return $this->response([
                                "error" => true,
                                "message" => "Please indicate software you have access to and/or programming language you are expert in"
                            ]);
                        }
        
                        $employee->native_language = $native_language;
                        $employee->other_languages = implode(',', $other_languages);
                        $employee->highest_academic_degree = $highest_academic_degree;
                        $employee->academic_degree_major = $academic_degree_major;
                        $employee->proficient_disciplines = implode(',', $proficient_disciplines);
                        $employee->software = implode(',', $software);
                        $employee->is_profile_completed = true;
                        $employee->update();
        
                        return $this->response([
                            "success" => true,
                            "message" => "Qualifications updated successfully"
                        ]);
                        break;

                    case 'dashboard_access':
                        $dashboard_access = ArrayUtils::get($payload, 'dashboard_access');

                        # flip array values to keys
                        if(ArrayUtils::has($payload, 'dashboard_access')) {
                            $dashboard_access = array_flip($dashboard_access);
                            $employee->has_academic_access = ArrayUtils::has($dashboard_access, 'academic') ? true : false;
                            $employee->has_article_access = ArrayUtils::has($dashboard_access, 'article') ? true : false;
                            $employee->has_programming_access = ArrayUtils::has($dashboard_access, 'programming') ? true : false;
                            $employee->has_calculations_access = ArrayUtils::has($dashboard_access, 'calculations') ? true : false;
                        }

                        $employee->update();
        
                        return $this->response([
                            "success" => true,
                            "message" => "Order access updated successfully"
                        ]);
                        break;

                    case 'working_status':
                        $employee_status = (int) ArrayUtils::get($payload, 'employee_status');
                        $employee->employee_status = $employee_status;

                        $employee->available_247 = ArrayUtils::has($payload, 'available_247') ? true : false;
                        $employee->available_urgent_orders = ArrayUtils::has($payload, 'available_urgent_orders') ? true : false;

                        $employee->update();
        
                        return $this->response([
                            "success" => true,
                            "message" => "Status updated successfully"
                        ]);
                        break;

                    case 'timezone':
                        $timezone = ArrayUtils::get($payload, 'timezone');
                        $employee->timezone = $timezone;
                        $employee->update();

                        Hook::exec('activitylog', [
                            'object' => 'employee',
                            'object_id' => $employee->id,
                            'event' => 'edit_employee_timezone'
                        ]);

                        return $this->response([
                            "success" => true,
                            "message" => "Timezone updated successfully"
                        ]);
                        break;

                    case 'password':
                        $password = ArrayUtils::get($payload, 'password');

                        if ( !$password ) {
                            return $this->response([
                                'error' => true,
                                'message' => 'Please enter the password'
                            ]);
                        }

                        if ( !Validate::isPasswd($password)) {
                            return $this->response([
                                'error' => true,
                                'message' => 'Password should be more than six characters'
                            ]);
                        }
            
                        $crypto = new Hashing();
                        $employee->password = $crypto->hash(
                            $password,
                            PROX_COOKIE_KEY
                        );
                        $employee->last_passwd_gen = DateUtils::now();
                        $employee->update();

                        return $this->response([
                            "success" => true,
                            "message" => "Password updated successfully"
                        ]);
                        break;

                    default:
                        return $this->response([
                            "error" => true,
                            "message" => "Error 404. Resource not found"
                        ]);
                        break;
                }

                break;
            
            case 'upload':
                $employee_id = ArrayUtils::get($payload, 'id');

                $employee = new Employee( (int) $employee_id );
                if(!Validate::isLoadedObject($employee)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This employee does not exist"
                    ]);
                }    

                $files = ArrayUtils::get($payload, 'files');
                foreach ($files as $uploadedFile) {
                    $file_id = (int) ArrayUtils::get($uploadedFile, 'id');
                    $employeeFile = new EmployeeFile( $file_id );
                    if(Validate::isLoadedObject($employeeFile)) {
                        $employeeFile->employee_id = $employee->id;
                        $employeeFile->uploader_id = $user->id;
                        $employeeFile->update();
                    }
                }

                return $this->modal('SUCCESS', "Upload Successful", "The files have been uploaded successfuly");	
                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }

    public function generateResetLink() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        if(!$user->is_admin) {
            return $this->modal("ERROR", "Error", "You don't have the right permission to access this"); 
        }

        $id = ArrayUtils::get($payload, 'id');
        $handle = ArrayUtils::get($payload, 'handle');

        switch( $handle ) {
            case 'customer':
                $customer = new Customer( (int) $id );
                if(!Validate::isLoadedObject($customer)) {
                    return $this->modal("ERROR", "Error", "This customer does not exist"); 
                }

                $customer->stampResetPasswordToken();
                $customer->update();

                $site = new Site( (int) $customer->site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->modal("ERROR", "Error", "This customer's site not exist"); 
                }

                Hook::exec('activitylog', [
                    'object' => 'customer',
                    'object_id' => $customer->id,
                    'event' => 'password_reset_link'
                ]);

                $reset_link = $site->domain . '/dashboard/reset-password/' . $customer->reset_password_token;

                $modalData = array(
                    'password_reset_link' => $reset_link
                );

                return $this->modal("#modalPasswordResetLink", Tools::jsonEncode($modalData)); 
                break;

            case 'employee':
                $employee = new Employee( (int) $id );
                if(!Validate::isLoadedObject($employee)) {
                    return $this->modal("ERROR", "Error", "This user does not exist"); 
                }

                $employee->stampResetPasswordToken();
                $employee->update();

                $reset_link = Configuration::get('SITE_DOMAIN') . '/reset/' . $employee->reset_password_token;

                $modalData = array(
                    'password_reset_link' => $reset_link
                );

                Hook::exec('activitylog', [
                    'object' => 'employee',
                    'object_id' => $employee->id,
                    'event' => 'password_reset_link'
                ]);

                return $this->modal("#modalPasswordResetLink", Tools::jsonEncode($modalData));  
                break;
        }

        return $this->response([
            'callback' => 'window.location.reload();'
        ]);
    }
}