<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Addon\Theme\ThemeManager;
use Proxim\Addon\Theme\ThemeManagerBuilder;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Pager;
use Proxim\Currency;
use Proxim\Coupon;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Admin extends Controller {
    public function settings() {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "System Settings"
        ]);

        $formValues = Configuration::getMultiple(
            [
                'SITE_NAME',
                'SITE_EMAIL',
                'SITE_DOMAIN',
                'SITE_EMAIL_LOGO',
                'SITE_BRAND_LOGO',
                'SITE_FAVICON',
                'NOTIFICATION_EMAILS',
                'REGISTRATION_ENABLED',
                'COOKIE_CONSENT_ENABLED',
                'ARTICLE_WRITING_ENABLED',
                'CSS_CUSTOMIZE',
                'HEADER_COLOR',
                'BUTTON_PRIMARY',
                'MENU_BACKGROUND',
                'CUSTOM_CSS',
                'SEND_MAIL',
                'MAIL_METHOD',
                'MAIL_SERVER',
                'MAIL_USER',
                'MAIL_PASSWD',
                'MAIL_SMTP_ENCRYPTION',
                'MAIL_SMTP_PORT',
                'PWA_ENABLED',
                'PWA_APP_ICON',
                'PWA_APP_NAME',
                'PWA_SHORT_NAME',
                'PWA_THEME_COLOR',
                'PWA_BACKGROUND_COLOR'
            ]
        );
        
        $formValues['SITE_ID'] = PROX_SITE_ID;
        $formValues['COOKIE_LIFETIME'] = Configuration::get('COOKIE_LIFETIME', PROX_SITE_ID, 480);

        $smarty->assign([
            'view' => 'system',
            'sub_view' => 'settings',
            'formValues' => $formValues
        ]);

        $this->setTemplate('dashboard/system/settings');
        $this->display(); 
    }   

    public function ordersPaymentsSettings( $_tab = 'general' ) {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "System Settings › Orders & Payments"
        ]);

        switch($_tab) {
            case 'order-statuses':
                $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
                $order_statuses = array();

                $sql = new DbQuery();
                $sql->select('os.*');
                $sql->from('order_state', 'os');
                $sql->orderBy('c.coupon_id DESC');

                $result = Coupon::getCoupons( false, false );
                $total_coupons = count($result);

                if( $total_coupons > 0) {
                    $params['total_items'] = $total_coupons;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/coupons?page=%s';

                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    
                    $sql .= $limit_query;

                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                    foreach ( $result as $coupon ) {
                        $coupon = new Coupon( (int) $coupon['coupon_id'] );
                        if(Validate::isLoadedObject($coupon)) {
                            $coupons[] = $coupon;
                        } 
                    }

                    $app->smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $smarty->assign([
                    'total_coupons' => $total_coupons,
                    'coupons' => $coupons,
                ]);
                break;
        }

        $formValues = Configuration::getMultiple(
            [
                'ORDER_MANAGER_ENABLED',
                'ORDER_MANAGER_PUBLIC',
                'NOTIFICATION_EMAILS',
                'PAY_DELIVERED_ORDERS',
                'ORDER_AUTOAPPROVE_TIME',
                'WRITER_PAYMENT_REQUEST_LIMIT',
                'EDITOR_PAYMENT_REQUEST_LIMIT',
                'CSS_CUSTOMIZE',
                'HEADER_COLOR',
                'BUTTON_PRIMARY',
                'MENU_BACKGROUND',
                'CUSTOM_CSS',
                'WRITER_CPP_TYPE',
                'WRITER_AMOUNT_CPP',
                'WRITER_PERCENTAGE_CPP',
                'WRITER_CPW_TYPE',
                'WRITER_AMOUNT_CPW',
                'WRITER_PERCENTAGE_CPW',
                'WRITER_W_CATEGORY_COST',
                'WRITER_COMPLEX_ASSIGNMENT_COST',
                'WRITER_USED_SOURCES_COST',
                'WRITER_PROGRESSIVE_DELIVERY_COST',
                'WRITER_PROVIDE_ME_SAMPLES_COST',
                'WRITER_EXPERT_PROOFREADING_COST',
                'WRITER_VIP_SUPPORT_COST',
                'WRITER_PLAGIARISM_REPORT_COST',
                'WRITER_DRAFT_OUTLINE_COST',
                'WRITER_PREFFERED_WRITER_COST',
                'EDITOR_CPP_TYPE',
                'EDITOR_AMOUNT_CPP',
                'EDITOR_PERCENTAGE_CPP',
                'EDITOR_CPW_TYPE',
                'EDITOR_AMOUNT_CPW',
                'EDITOR_PERCENTAGE_CPW',
                'EDITOR_W_CATEGORY_COST',
                'EDITOR_COMPLEX_ASSIGNMENT_COST',
                'EDITOR_USED_SOURCES_COST',
                'EDITOR_PROGRESSIVE_DELIVERY_COST',
                'EDITOR_PROVIDE_ME_SAMPLES_COST',
                'EDITOR_EXPERT_PROOFREADING_COST',
                'EDITOR_VIP_SUPPORT_COST',
                'EDITOR_PLAGIARISM_REPORT_COST',
                'EDITOR_DRAFT_OUTLINE_COST',
                'ORDER_MANAGER_CPP_TYPE',
                'ORDER_MANAGER_AMOUNT_CPP',
                'ORDER_MANAGER_PERCENTAGE_CPP',
                'ORDER_AUTOINCREMENT',
                'CONFIRM_ORDER_ASSIGN',
                'AUTOASSIGN_REQUESTED_WRITER',
                'WORDS_PER_PAGE',
                'ATTACH_COMPLETED_FILES',
                'EDITOR_ACCOUNT_ENABLED',
                'SUBADMIN_HIDE_CLIENT_COST',
                'HIGHLIGHT_ORDER_DEADLINE',
                'DEADLINE_DUE_HOURS',
                'WRITER_PAY_CURRENCY'
            ]
        );

        $formValues['SITE_ID'] = PROX_SITE_ID;
        $formValues['NEW_ORDER_STATUS'] = Configuration::get('NEW_ORDER_STATUS', null, Order::ORDER_PUBLISHED);

        $formValues['SUBADMIN_CLIENT_COST'] = Configuration::get('SUBADMIN_CLIENT_COST', null, true);
        $formValues['SUBADMIN_WRITER_INFO'] = Configuration::get('SUBADMIN_WRITER_INFO', null, true);


        $system_payment_dates = explode(',', Configuration::get('PAYMENT_DATES'));
        $payment_dates = array();

        for ($i=1; $i <= 31; $i++) { 
            $payment_dates[] = array(
                'day' => $i,
                'checked' => in_array($i, $system_payment_dates) ? true : false
            );
        }

        $currencies = Currency::getCurrencies( false );

        $smarty->assign([
            'view' => 'system',
            'sub_view' => 'orders-payments',
            '_tab' => $_tab,
            'formValues' => $formValues,
            'currencies' => $currencies,
            'payment_dates' => $payment_dates
        ]);

        $this->setTemplate('dashboard/system/orders-payments');
        $this->display(); 
    }

    public function uploads() {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "System Settings › Uploads"
        ]);

        $formValues = Configuration::getMultiple([
            'SHARED_FOLDER_PERMISSIONS',
            'INTERNAL_FILE_STORAGE',
            'ALLOWED_EXTENSIONS',
            'S3_ENABLED',
            'S3_REGION',
            'S3_KEY',
            'S3_SECRET',
            'S3_BUCKET',
            'DIGITALOCEAN_ENABLED',
            'DIGITALOCEAN_KEY',
            'DIGITALOCEAN_SECRET',
            'DIGITALOCEAN_SPACE_NAME',
            'DIGITALOCEAN_SPACE_REGION'
        ]);

        $smarty->assign([
            'view' => 'system',
            'sub_view' => 'uploads',
            'max_upload_size' => ini_get('upload_max_filesize'),
            'formValues' => $formValues,
        ]);

        $this->setTemplate('dashboard/system/uploads');
        $this->display();  
    }

    public function themes() {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "System Settings › Themes"
        ]);

        $themes = (new ThemeManagerBuilder($app, Db::getInstance()))
            ->buildRepository()
            ->getList();

        $smarty->assign([
            'view' => 'system',
            'sub_view' => 'themes',
            'themes' => $themes
        ]);

        $this->setTemplate('dashboard/system/themes');
        $this->display();  
    }

    public function payments() {
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Payments"
        ]);

        $smarty->assign([
            'view' => 'system',
            'sub_view' => 'payments',
        ]);

        $this->setTemplate('dashboard/system/payments');
        $this->display(); 
    }

    public function currency( $sub_view = "currency", $currency_id = null ) {
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Currency"
        ]);
 
        switch( $sub_view ) {
            case 'currency':
                $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
                $currencies = array();

                $result = Currency::getCurrencies( false );
                $total_currencies = count($result);

                if( $total_currencies > 0) {
                    $params['total_items'] = $total_currencies;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/system/currency?page=%s';

                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('
                        SELECT c.*
                        FROM `' . DB_PREFIX . 'currency` c '.
                        $limit_query
                    );

                    foreach ( $result as $currency ) {
                        $currency = new Currency( (int) $currency['currency_id'] );
                        if(Validate::isLoadedObject($currency)) {
                            $currencies[] = (array) $currency;
                        }
                    }

                    $app->smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $smarty->assign([
                    'total_currencies' => $total_currencies,
                    'currencies' => $currencies,
                ]);
                break;
            
            case 'add':

                break;
            case 'edit':
                
                $currency = new Currency( (int) $currency_id );
                if(!Validate::isLoadedObject($currency)) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $smarty->assign([
                    'currency' => (array) $currency,
                ]);

                break;
            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'system',
            'sub_view' => $sub_view
        ]);

        $this->setTemplate('dashboard/system/currency');
        $this->display();
    }
}