<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Pager;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\MailTemplate;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\User\Employee;
use Proxim\User\Customer;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;
use Proxim\Util\StringUtils;

class Mailing extends Controller {

    public function mails() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $this->setVars([
            'page_title' => "Mailing › Mails"
        ]);

        $mails = array();

        $sql = new DbQuery();
        $sql->select('m.*');
        $sql->from('mailing_list', 'm');
        $sql->orderBy('m.mail_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_mails = count($result);

        if( $total_mails > 0) {
            $params['total_items'] = $total_mails;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/mailing/mails?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
            foreach( $result as $mail ) {
                $mail['site_name'] = Configuration::get('SITE_NAME', (int) $mail['site_id']);
                $mail['date'] = DateUtils::convertToISOFormat( $mail['date_add'] );
                $mails[] = $mail;
            }


            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'mailing',
            'sub_view' => 'mails',
            'total_mails' => $total_mails,
            'mails' => $mails
        ]);

        $this->setTemplate('dashboard/mailing/mailing');
        $this->display(); 
    }

    public function sendMail() {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Mailing › Send"
        ]);

        $sql = new DbQuery();
        $sql->select('s.*');
        $sql->from('site', 's');
        $sites = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

        $sql = new DbQuery();
        $sql->select('e.*');
        $sql->from('employee', 'e');
        $writers = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

        $customers = array();
        $sql = new DbQuery();
        $sql->select('c.customer_id');
        $sql->from('customer', 'c');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

        foreach ( $result as $customer ) {
            $customer = new Customer( (int) $customer['customer_id'] );
            if(Validate::isLoadedObject($customer)) {
                $customers[] = (array) $customer;
            }
        }

        $customers = ArrayUtils::groupByKey( $customers, 'site_name' );

        $sql = new DbQuery();
        $sql->select('m.*');
        $sql->from('mailing_list', 'm');
        $mailing_list = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

        // mail sites
        $mail_sites = Site::getSites();

        $smarty->assign([
            'view' => 'mailing',
            'sub_view' => 'send',
            'customers' => $customers,
            'writers' => $writers,
            'mailing_list' => $mailing_list,
            'mail_sites' => $mail_sites,
            'sites' => $sites
        ]);

        $this->setTemplate('dashboard/mailing/mailing');
        $this->display(); 
    }

    public function addMail() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Mailing › Add"
        ]);

        $sql = new DbQuery();
        $sql->select('s.*');
        $sql->from('site', 's');
        $sites = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

        $smarty->assign([
            'view' => 'mailing',
            'sub_view' => 'add',
            'sites' => $sites
        ]);

        $this->setTemplate('dashboard/mailing/mailing');
        $this->display(); 
    }

    public function editMail($mailId) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        $sql = new DbQuery();
        $sql->select('m.*');
        $sql->from('mailing_list', 'm');
        $sql->where('mail_id = ' . (int) $mailId );
        $mail = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow( $sql );

        if(!$mail) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Mailing › Edit › " . $mail['mail_id']
        ]);

        $sql = new DbQuery();
        $sql->select('s.*');
        $sql->from('site', 's');
        $sites = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

        $smarty->assign([
            'view' => 'mailing',
            'sub_view' => 'edit',
            'mail' => $mail,
            'sites' => $sites
        ]);
        
        $this->setTemplate('dashboard/mailing/mailing');
        $this->display(); 
    }

    public function templates( $tab = 'templates' ) {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin || !Module::isInstalled('mail_templates')) {
            $this->setTemplate('404');
            $this->display();
        }

        switch($tab) {
            case 'templates':
                $this->setVars([
                    'page_title' => "Mailing › Templates"
                ]);

                $sql = new DbQuery();
                $sql->select('mt.*');
                $sql->from('mail_template', 'mt');

                $orderTemplatesSql = clone $sql;
                $customerTemplatesSql = clone $sql;
                $employeeTemplatesSql = clone $sql; 

                $orderMailTemplates = array(
                    MailTemplate::TEMPLATE_ORDER_PAYMENT,
                    MailTemplate::TEMPLATE_ORDER_NEW_AVAILABLE,
                    MailTemplate::TEMPLATE_ORDER_FILES,
                    MailTemplate::TEMPLATE_ORDER_ASSIGN,
                    MailTemplate::TEMPLATE_ORDER_WRITER_SUBMIT,
                    MailTemplate::TEMPLATE_ORDER_NEW_EDITOR,
                    MailTemplate::TEMPLATE_ORDER_COMPLETE,
                    MailTemplate::TEMPLATE_ORDER_STATUS_CHANGE,
                    MailTemplate::TEMPLATE_ORDER_REVISION,
                    MailTemplate::TEMPLATE_ORDER_MESSAGE,
                    MailTemplate::TEMPLATE_ORDER_FEEDBACK_REPLY,
                    MailTemplate::TEMPLATE_ORDER_PAY_REMINDER,
                    MailTemplate::TEMPLATE_ORDER_FILE_DELETE,
                    MailTemplate::TEMPLATE_ORDER_REFUND_REQUEST
                );
                $orderTemplatesSql->where("mt.template IN ('" . implode("', '", $orderMailTemplates) . "')");

                $customerMailTemplates = array(
                    MailTemplate::TEMPLATE_CUSTOMER_PASSWORD_CHANGE
                );
                $customerTemplatesSql->where("mt.template IN ('" . implode("', '", $customerMailTemplates) . "')");

                $employeeMailTemplates = array(
                    MailTemplate::TEMPLATE_EMPLOYEE_PASSWORD_CHANGE,
                    MailTemplate::TEMPLATE_EMPLOYEE_TWO_FACTOR
                );
                $employeeTemplatesSql->where("mt.template IN ('" . implode("', '", $employeeMailTemplates) . "')");

                $orderTemplatesResult = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($orderTemplatesSql);
                $customerTemplatesResult = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($customerTemplatesSql);
                $employeeTemplatesResult = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($employeeTemplatesSql);

                $orderTemplates = $customerTemplates = $employeeTemplates = array();
                foreach($orderTemplatesResult as $mailTemplate) {
                    $mailTemplate['name'] = MailTemplate::getTitleByTemplateName($mailTemplate['template']);
                    $orderTemplates[$mailTemplate['template']] = $mailTemplate;
                }


                foreach($customerTemplatesResult as $mailTemplate) {
                    $mailTemplate['name'] = MailTemplate::getTitleByTemplateName($mailTemplate['template']);
                    $customerTemplates[$mailTemplate['template']] = $mailTemplate;
                }

                foreach($employeeTemplatesResult as $mailTemplate) {
                    $mailTemplate['name'] = MailTemplate::getTitleByTemplateName($mailTemplate['template']);
                    $employeeTemplates[$mailTemplate['template']] = $mailTemplate;
                }
 
                $smarty->assign([
                    'orderTemplates' => $orderTemplates,
                    'customerTemplates' => $customerTemplates,
                    'employeeTemplates' => $employeeTemplates
                ]);
                break;

            case 'add':
                $this->setVars([
                    'page_title' => "Mailing › Add Template"
                ]);

                $sql = new DbQuery();
                $sql->select('mt.*');
                $sql->from('mail_template', 'mt');
                $emailTemplatesResult = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $emailTemplates = array();
                foreach($emailTemplatesResult as $mailTemplate) {
                    $mailTemplate['name'] = MailTemplate::getTitleByTemplateName($mailTemplate['template']);
                    $emailTemplates[$mailTemplate['template']] = $mailTemplate;
                }

                $smarty->assign([
                    'emailTemplates' => $emailTemplates
                ]);
                break;

            case 'edit':
                $template = ArrayUtils::get($params, 'template');
                $mailTemplateId = MailTemplate::getIdByTemplateName( $template );
                $mailTemplate = new MailTemplate( (int) $mailTemplateId );
                if(!Validate::isLoadedObject($mailTemplate)) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $this->setVars([
                    'page_title' => "Mailing › Edit Template"
                ]);

                $smarty->assign([
                    'mailTemplate' => (array) $mailTemplate
                ]);
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'mailing',
            'sub_view' => 'templates',
            '_tab' => $tab
        ]);

        $this->setTemplate('dashboard/mailing/mailing');
        $this->display(); 
    }
}