<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Pager;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Currency;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\Deadline;
use Proxim\Presenter\Deadline\DeadlinePresenter;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Validate;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class MySites extends Controller {

    public function viewSites() {
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "My Sites"
        ]);

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $sites = array();

        $sql = new DbQuery();
        $sql->select('site.site_id');
        $sql->from('site', 'site');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $total_sites = count($result);
        if( $total_sites > 0) {
            $params['total_items'] = $total_sites;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = '/my-sites?page=%s';

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('
                SELECT site.site_id
                FROM `' . DB_PREFIX . 'site` site ' .
                $limit_query
            );

            foreach( $result as $site ) {
                $site = new Site( (int) $site['site_id'] );
                if( Validate::isLoadedObject($site) ) {
                    $sites[] = (array) $site;
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'my-sites',
            'sub_view' => 'sites',
            'total_sites' => $total_sites,
            'sites' => $sites,
        ]);

        $this->setTemplate('dashboard/my-sites/my-sites');
        $this->display(); 
    }

    public function addSite() {
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Sites › Add"
        ]);

        $smarty->assign([
            'view' => 'my-sites',
            'sub_view' => "add"
        ]);

        $this->setTemplate('dashboard/my-sites/add');
        $this->display(); 
    }

    public function editSite( $site_id ) {
        global $globals;

        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $site = new Site( (int) $site_id );
        if(!Validate::isLoadedObject($site)) {
            $this->setTemplate('404');
            $this->display();
        }

        $app->site = $site;

        $this->setVars([
            'page_title' => "Sites › General Settings"
        ]);

        $currencies = Currency::getCurrencies( false );
        $paymentModules = PaymentModule::getInstalledPaymentModules();

        $paymentMethods = array();
        foreach($paymentModules as $paymentModule) {
            $paymentModule = Module::getInstanceByName($paymentModule['name']);
            if(Validate::isLoadedObject($paymentModule)) {
                if (@filemtime(PROX_DIR_MODULE . $paymentModule->name . DIRECTORY_SEPARATOR . 'logo.png')) {
                    $paymentModuleLogo = $app->modules_uri . '/' . $paymentModule->name . '/logo.png';
                } elseif( @filemtime(PROX_DIR_MODULE . $paymentModule->name . DIRECTORY_SEPARATOR . 'logo.jpg')) {
                    $paymentModuleLogo = $app->modules_uri . '/' . $paymentModule->name . '/logo.jpg';
                } 

                $paymentMethods[] = array(
                    'name' => $paymentModule->name,
                    'logo' => $paymentModuleLogo,
                    'displayName' => $paymentModule->displayName,
                    'description' => $paymentModule->description
                );
            }
        }

        $formValues = Configuration::getMultiple(
            [
                'SITE_NAME',
                'SITE_EMAIL',
                'SITE_DOMAIN',
                'SITE_EMAIL_LOGO',
                'SITE_BRAND_LOGO',
                'SITE_FAVICON',
                'PAYMENT_METHOD_DEFAULT',
                'AUTH_REQUIRE_NAME',
                'AUTH_REQUIRE_PHONE',
                'UK_FORM',
                'SHARED_AUTH',
                'PWA_ENABLED',
                'PWA_APP_ICON',
                'PWA_APP_NAME',
                'PWA_SHORT_NAME',
                'PWA_THEME_COLOR',
                'PWA_BACKGROUND_COLOR',
                'ANONYMOUS_ENABLED',
                'TAWK_CHAT_CODE',
                'SIGNUP_DEFAULT',
                'MAIL_METHOD',
                'MAIL_SERVER',
                'MAIL_USER',
                'MAIL_PASSWD',
                'MAIL_SMTP_ENCRYPTION',
                'MAIL_SMTP_PORT',
                'DEFAULT_COURSELEVEL',
                'URGENT_DEADLINE_MAX_PAGES',
                'CURRENCY_DEFAULT',
                'DEFAULT_SERVICE_TYPE_ID',
                'ACADEMIC_WRITING_ENABLED',
                'ARTICLE_WRITING_ENABLED',
                'CALCULATIONS_ENABLED',
                'PROGRAMMING_ENABLED'
            ],
            (int) $site->id
        );

        $formValues['SITE_ID'] = (int) $site->id;

        $formValues['CPP_PERCENTAGE'] = (float) Configuration::get('CPP_PERCENTAGE', $site->id, 100);


        $default_deadlines = (string) Configuration::get('DEFAULT_DEADLINES', $site->id);
        $default_deadlines = explode(',', $default_deadlines);

        // tariffs
        $sql = new DbQuery();
        $sql->select('d.deadline_id');
        $sql->from('deadline', 'd');
        $sql->where('d.site_id = ' . (int) $site->id );
        $sql->where('d.is_academic_writing = 1' );
        $sql->orderBy('d.hrs_customer ASC');
        $result = Db::getInstance()->executeS($sql);

        $academicLevels = ArrayUtils::get($globals, 'academicLevels');

        $tariffs = $deadlines = array();
        foreach($result as $deadline) {
            $tariff = new Deadline( (int) $deadline['deadline_id']);
            if(Validate::isLoadedObject($tariff)) {
                $tariff->selected = in_array($tariff->id, $default_deadlines) ? true : false;
                $deadlines[] = (array) $tariff;
            }
        }

        $deadlines = ArrayUtils::groupByKey($deadlines, 'academic_level_id');

        foreach($deadlines as $academic_level_id => $deadline_items) {
            if(ArrayUtils::has($academicLevels, $academic_level_id)) {
                $academicLevel = ArrayUtils::get($academicLevels, $academic_level_id);
                $academic_level_title = ArrayUtils::get($academicLevel, 'title');

                $tariffs[$academic_level_id] = array(
                    'id' => $academic_level_id,
                    'title' => $academic_level_title,
                    'deadlines' => $deadline_items
                );
            }
        }

        ksort($tariffs);
        
        $objectPresenter = new ObjectPresenter();

        $smarty->assign([
            'view' => 'my-sites',
            'sub_view' => 'edit',
            'tariffs' => $tariffs,
            'formValues' => $formValues,
            'currencies' => $currencies,
            'paymentMethods' => $paymentMethods,
            'site' => $objectPresenter->present($site)
        ]);

        $this->setTemplate('dashboard/my-sites/edit');
        $this->display(); 
    }

    public function preferences($site_id) {
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $site = new Site( (int) $site_id );
        if(!Validate::isLoadedObject($site)) {
            $this->setTemplate('404');
            $this->display();
        }

        $app->site = $site;

        $this->setVars([
            'page_title' => "Sites › Preferences"
        ]);

        $tab = ArrayUtils::get($params, 'tab', 'features');

        switch($tab) {
            case 'features':
                $formValues = Configuration::getMultiple(
                    [
                        'ACADEMIC_WRITING_ENABLED',
                        'PROGRAMMING_ENABLED',
                        'CALCULATIONS_ENABLED',
                        'ARTICLE_WRITING_ENABLED',
                        'RESUME_ENABLED',
                        'ORDER_FORM_COUPONS',
                        'DISCOUNTS_ENABLED',
                        'INQUIRY_ENABLED',
                        'REQUEST_REFUND_ALLOWED'
                    ],
                    (int) $site->id
                );
                
                $smarty->assign(array(
                    'formValues' => $formValues
                ));
                break;

            case 'add-services':
                $formValues = Configuration::getMultiple(
                    [
                        'WRITER_SAMPLES_ENABLED',
                        'PROGRESSIVE_DELIVERY_ENABLED',
                        'EXPERT_PROOFREADING_ENABLED',
                        'COPY_OF_SOURCES_ENABLED',
                        'VIP_SUPPORT_ENABLED',
                        'PLAGIARISM_REPORT_ENABLED',
                        'DRAFT_OUTLINE_ENABLED'
                    ],
                    (int) $site->id
                );
                
                $smarty->assign(array(
                    'formValues' => $formValues
                ));
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $objectPresenter = new ObjectPresenter();
        
        $smarty->assign([
            'view' => 'my-sites',
            'sub_view' => 'preferences',
            'tab' => $tab,
            'site' => $objectPresenter->present($site)
        ]);

        $this->setTemplate('dashboard/my-sites/preferences');
        $this->display(); 
    }

    public function deadlines($site_id) {
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $site = new Site( (int) $site_id );
        if(!Validate::isLoadedObject($site)) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Sites › Deadlines"
        ]);

        $app->site = $site;

        $tab = ArrayUtils::get($params, 'tab', 'academic');

        switch($tab) {
            case 'academic':
                $academicLevels = AcademicLevel::getAcademicLevels($site->id);

                $academicLevelId = ArrayUtils::get($params, 'academicLevel', count($academicLevels) ? $academicLevels[0]['id']: LEVEL_HIGHSCHOOL);
        
                $sql = new DbQuery();
                $sql->select('d.deadline_id');
                $sql->from('deadline', 'd');
                $sql->where('d.is_academic_writing = 1');
                $sql->where('d.site_id = ' . (int) $site->id );
                if($academicLevelId) {
                    $sql->where('d.academic_level_id = ' . (int) $academicLevelId );
                }
                $sql->orderBy('d.hrs_customer ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $presenter = new DeadlinePresenter();
                $deadlines = array();
                foreach($result as $deadline) {
                    $deadline = new Deadline($deadline['deadline_id']);
                    if(Validate::isLoadedObject($deadline)) {
                        $deadline = $presenter->present( $deadline );
                        $deadlines[] = array(
                            'id' => $deadline->getId(),
                            'name' => $deadline->getName(),
                            'academic_level' => $deadline->getAcademicLevel(),
                            'price_per_page' => $deadline->getPricePerPage(),
                            'hrs' => $deadline->getHrs(),
                            'hrs_writer' => $deadline->getHrsWriter(),
                            'min_pages' => $deadline->getMinPages(),
                        );
                    }
                }

                $smarty->assign([
                    'academic_level_selected' => $academicLevelId,
                    'deadlines' => $deadlines,
                    'academic_levels' => $academicLevels
                ]);
                break;

            case 'programming':
                $sql = new DbQuery();
                $sql->select('d.deadline_id');
                $sql->from('deadline', 'd');
                $sql->where('d.is_programming = 1');
                $sql->where('d.site_id = ' . (int) $site->id );
                $sql->orderBy('d.hrs_customer ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $presenter = new DeadlinePresenter();
                $deadlines = array();
                foreach($result as $deadline) {
                    $deadline = new Deadline($deadline['deadline_id']);
                    if(Validate::isLoadedObject($deadline)) {
                        $deadline = $presenter->present( $deadline );
                        $deadlines[] = array(
                            'id' => $deadline->getId(),
                            'name' => $deadline->getName(),
                            'price_modifier' => $deadline->getPriceModifier(),
                            'hrs' => $deadline->getHrs(),
                            'hrs_writer' => $deadline->getHrsWriter()
                        );
                    }
                }

                $smarty->assign([
                    'deadlines' => $deadlines
                ]);
                break;

            case 'calculations':
                $sql = new DbQuery();
                $sql->select('d.deadline_id');
                $sql->from('deadline', 'd');
                $sql->where('d.is_calculations = 1');
                $sql->where('d.site_id = ' . (int) $site->id );
                $sql->orderBy('d.hrs_customer ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $presenter = new DeadlinePresenter();
                $deadlines = array();
                foreach($result as $deadline) {
                    $deadline = new Deadline($deadline['deadline_id']);
                    if(Validate::isLoadedObject($deadline)) {
                        $deadline = $presenter->present( $deadline );
                        $deadlines[] = array(
                            'id' => $deadline->getId(),
                            'name' => $deadline->getName(),
                            'price_modifier' => $deadline->getPriceModifier(),
                            'hrs' => $deadline->getHrs(),
                            'hrs_writer' => $deadline->getHrsWriter()
                        );
                    }
                }

                $smarty->assign([
                    'deadlines' => $deadlines
                ]);
                break;

            case 'articles':
                $sql = new DbQuery();
                $sql->select('d.deadline_id');
                $sql->from('deadline', 'd');
                $sql->where('d.is_article_writing = 1');
                $sql->where('d.site_id = ' . (int) $site->id );
                $sql->orderBy('d.hrs_customer ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $presenter = new DeadlinePresenter();
                $deadlines = array();
                foreach($result as $deadline) {
                    $deadline = new Deadline($deadline['deadline_id']);
                    if(Validate::isLoadedObject($deadline)) {
                        $deadline = $presenter->present( $deadline );
                        $deadlines[] = array(
                            'id' => $deadline->getId(),
                            'name' => $deadline->getName(),
                            'price_per_word' => $deadline->getPricePerWord(),
                            'hrs' => $deadline->getHrs(),
                            'hrs_writer' => $deadline->getHrsWriter(),
                            'min_pages' => $deadline->getMinPages(),
                        );
                    }
                }

                $smarty->assign([
                    'deadlines' => $deadlines,
                ]);
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }
        

        $objectPresenter = new ObjectPresenter();


        $smarty->assign([
            'view' => 'my-sites',
            'sub_view' => 'deadlines',
            'tab' => $tab,
            'site' => $objectPresenter->present($site)
        ]);

        $this->setTemplate('dashboard/my-sites/deadlines');
        $this->display(); 
    }
}