<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use ArrayAccess;
use Db;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Pager;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Preference\Deadline;
use Proxim\Preference\Deliverable;
use Proxim\Preference\Descriptor;
use Proxim\Preference\Discipline;
use Proxim\Preference\ItemPricing;
use Proxim\Presenter\Deadline\DeadlinePresenter;
use Proxim\Site\Site;
use Proxim\Validate;
use Proxim\Util\ArrayUtils;

class Preferences extends Controller {

    public function general( $sub_view = 'disciplines' ) {
        $app = $this->app; 
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display(); 
        }

        $selected_page = ArrayUtils::get($params, 'page', 1);

        switch($sub_view) {
            case 'disciplines':
                $this->setVars([
                    'page_title' => "Preferences › Disciplines"
                ]);
        
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_disciplines = count($result);
        
                $disciplines = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/general/disciplines?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($result as $discipline) {
                        $disciplines[] = array(
                            'discipline_id' => (int) $discipline['discipline_id'],
                            'group_id' => (int) $discipline['group_id'],
                            'title' => $discipline['title'],
                            'is_group' => (bool) $discipline['is_group'],
                            'is_complex_assignment' => (bool) $discipline['is_complex_assignment'],
                            'is_academic_writing' => (bool) $discipline['is_academic_writing'],
                            'is_calculations' => (bool) $discipline['is_calculations'],
                            'is_programming' => (bool) $discipline['is_programming'],
                            'is_article_writing' => (bool) $discipline['is_article_writing']
                        );
                    } 
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $disciplineGroups = Discipline::getDisciplinesByGroup();
        
                $smarty->assign([
                    'controller' => 'disciplines',
                    'disciplines' => $disciplines,
                    'disciplineGroups' => $disciplineGroups
                ]);
                break;

            case 'upload-types':
                $this->setVars([
                    'page_title' => "Preferences › Upload Types"
                ]);
        
                $sql = new DbQuery();
                $sql->select('ft.*');
                $sql->from('upload_type', 'ft');
                $sql->orderBy('ft.upload_type_id ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_paper_types = count($result);
        
                $upload_types = array();
                if( $total_paper_types > 0) {
                    $params['total_items'] = $total_paper_types;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/general/upload-types?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
        
                    $upload_types = array();
                    foreach($result as $upload_type) {
                        //@todo for mustache.min if statement
                        $upload_type['customer'] = (bool) $upload_type['customer'];
                        $upload_type['writer'] = (bool) $upload_type['writer'];
                        $upload_type['editor'] = (bool) $upload_type['editor'];
                        $upload_types[] = $upload_type;
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $smarty->assign([
                    'controller' => 'uploadTypes',
                    'uploadTypes' => $upload_types
                ]);
                break;
                
            case 'deliverables':

                if(!Module::isInstalled('programming') && !Module::isInstalled('calculations')) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $this->setVars([
                    'page_title' => "Preferences › Deliverables"
                ]);
        
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('deliverable', 'd');
                $sql->orderBy('d.deliverable_id DESC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_deliverables = count($result);
        
                $deliverables = array();
                if( $total_deliverables > 0) {
                    $params['total_items'] = $total_deliverables;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/general/deliverables?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $deliverables = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'controller' => 'deliverables',
                    'deliverables' => $deliverables
                ]);
                break;
                
            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'preferences',
            'sub_view' => 'general'
        ]);

        $this->setTemplate('dashboard/preferences/general/customize');
        $this->display(); 
    }

    public function academicWriting( $sub_view = 'paper-types' ) {
        $app = $this->app; 
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::get($params, 'page', 1);

        switch($sub_view) {
            case 'paper-types':
                $this->setVars([
                    'page_title' => "Preferences › Paper Types"
                ]);
        
                $sql = new DbQuery();
                $sql->select('pt.*');
                $sql->from('paper_type', 'pt');
                $sql->where('pt.is_academic_writing = 1');
                $sql->orderBy('pt.paper_type_id ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_paper_types = count($result);
        
                $paper_types = array();
                if( $total_paper_types > 0) {
                    $params['total_items'] = $total_paper_types;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/academic/paper-types?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $paper_types = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $smarty->assign([
                    'controller' => 'paperTypes',
                    'paper_types' => $paper_types
                ]);
                break;

            case 'disciplines':
                $this->setVars([
                    'page_title' => "Preferences › Disciplines"
                ]);
        
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_academic_writing = 1');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_disciplines = count($result);
        
                $disciplines = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/academic/disciplines?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($result as $discipline) {
                        $disciplines[] = array(
                            'discipline_id' => (int) $discipline['discipline_id'],
                            'group_id' => (int) $discipline['group_id'],
                            'title' => $discipline['title'],
                            'is_group' => (bool) $discipline['is_group'],
                            'is_complex_assignment' => (bool) $discipline['is_complex_assignment'],
                            'is_academic_writing' => (bool) $discipline['is_academic_writing'],
                            'is_calculations' => (bool) $discipline['is_calculations'],
                            'is_programming' => (bool) $discipline['is_programming'],
                            'is_article_writing' => (bool) $discipline['is_article_writing']
                        );
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $disciplineGroups = Discipline::getDisciplinesByGroup();
        
                $smarty->assign([
                    'controller' => 'disciplines',
                    'disciplines' => $disciplines,
                    'disciplineGroups' => $disciplineGroups
                ]);
                break;

            case 'paper-formats':
                $this->setVars([
                    'page_title' => "Preferences › Paper Formats"
                ]);
        
                $sql = new DbQuery();
                $sql->select('pf.*');
                $sql->from('paper_format', 'pf');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_paper_formats = count($result);
        
                $paper_formats = array();
                if( $total_paper_formats > 0) {
                    $params['total_items'] = $total_paper_formats;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/academic/paper-formats?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $paper_formats = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $smarty->assign([
                    'controller' => 'paperFormats',
                    'paper_formats' => $paper_formats
                ]);
                break;
            
            case 'academic-levels':
                $this->setVars([
                    'page_title' => "Preferences › Academic Levels"
                ]);
        
                $sql = new DbQuery();
                $sql->select('al.*,  GROUP_CONCAT(als.site_id) AS site_ids');
                $sql->from('academic_level', 'al');
                $sql->leftJoin('academic_level_site', 'als', 'als.academic_level_id = al.academic_level_id');
                $sql->groupBy('al.academic_level_id');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_academic_levels = count($result);
        
                $academic_levels = array();
                if( $total_academic_levels > 0) {
                    $params['total_items'] = $total_academic_levels;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/academic/academic-levels?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);

                    $academic_levels = array();
                    foreach ($result as $academic_level) {
                        $site_ids = explode(',', $academic_level['site_ids']);
                        if(is_array($site_ids) && !empty($site_ids)) {
                            $academic_level_sites = array();

                            foreach($site_ids as $site_id) {
                                $site = new Site((int) $site_id);
                                if(Validate::isLoadedObject($site)) {
                                    $academic_level_sites[] = [
                                        'id' => $site->id,
                                        'name' => $site->name
                                    ];
                                }
                            }

                            $academic_level['sites'] = $academic_level_sites;
                        }

                        $academic_level['site_ids'] = $site_ids;

                        $academic_levels[] = $academic_level;
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $sites = Site::getSites();
        
                $smarty->assign([
                    'controller' => 'academicLevels',
                    'sites' => $sites,
                    'academic_levels' => $academic_levels
                ]);
                break;
                
            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'preferences',
            'sub_view' => 'academic'
        ]);

        $this->setTemplate('dashboard/preferences/academic/customize');
        $this->display(); 
    }

    public function programming($sub_view = 'disciplines') {
        global $globals;
        $app = $this->app; 
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin || !Module::isInstalled('programming')) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::get($params, 'page', 1);

        switch($sub_view) {
            case 'disciplines':
                $this->setVars([
                    'page_title' => "Preferences › Disciplines"
                ]);
        
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_programming = 1');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_disciplines = count($result);
        
                $disciplines = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/programming/disciplines?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($result as $discipline) {
                        $disciplines[] = array(
                            'discipline_id' => (int) $discipline['discipline_id'],
                            'group_id' => (int) $discipline['group_id'],
                            'title' => $discipline['title'],
                            'is_group' => (bool) $discipline['is_group'],
                            'is_complex_assignment' => (bool) $discipline['is_complex_assignment'],
                            'is_academic_writing' => (bool) $discipline['is_academic_writing'],
                            'is_calculations' => (bool) $discipline['is_calculations'],
                            'is_programming' => (bool) $discipline['is_programming'],
                            'is_article_writing' => (bool) $discipline['is_article_writing']
                        );
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $disciplineGroups = Discipline::getDisciplinesByGroup();
        
                $smarty->assign([
                    'controller' => 'disciplines',
                    'disciplines' => $disciplines,
                    'disciplineGroups' => $disciplineGroups
                ]);
                break;

            case 'descriptors':
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_programming = 1');
                $sql->orderBy('d.title ASC');
                $disciplines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_disciplines = count($disciplines);

                $descriptorsByDiscipline = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/programming/descriptors?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $disciplines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($disciplines as $discipline) {
                        $sql = new DbQuery();
                        $sql->select('d.descriptor_id, d.discipline_id, d.task_size_id, d.example');
                        $sql->from('descriptor', 'd');
                        $sql->where('d.is_programming = 1');
                        $sql->where('d.discipline_id = ' . (int) $discipline['discipline_id']);
                        $sql->orderBy('d.task_size_id ASC');
                        $descriptors = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                        $total_descriptors = count($descriptors);

                        foreach($descriptors as $descriptor) {
                            $descriptor = new Descriptor( (int) $descriptor['descriptor_id'] );
                            if(Validate::isLoadedObject($descriptor)) {
                                $taskSize = ArrayUtils::get($globals['taskSizes'], $descriptor->task_size_id);
                                $deliverables = $descriptor->getDeliverables();
                               
                                $descriptorsByDiscipline[] = array(
                                    'descriptor_id' => $descriptor->id,
                                    'discipline_title' => $discipline['title'],
                                    'discipline_id' => $descriptor->discipline_id,
                                    'task_size_id' => $descriptor->task_size_id,
                                    'task_size_title' => $taskSize['title'],
                                    'example' => $descriptor->example,
                                    'deliverables' => $deliverables
                                );
                            }
                        }
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $deliverables = Deliverable::getDeliverables();

                $this->setVars([
                    'page_title' => "Preferences › Descriptors"
                ]);

                $descriptorsByDiscipline = ArrayUtils::groupByKey($descriptorsByDiscipline, 'discipline_id');

                $smarty->assign([
                    'controller' => 'descriptors',
                    'descriptorsByDiscipline' => $descriptorsByDiscipline,
                    'taskSizes' => $globals['taskSizes'],
                    'disciplines' => $disciplines,
                    'deliverables' => $deliverables
                ]);
                break;

            case 'pricing':
                $this->setVars([
                    'page_title' => "Preferences › Pricing"
                ]);

                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_programming = 1');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_disciplines = count($result);

                $disciplines = $itemPricesByDiscipline = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/programming/pricing?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $disciplines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($disciplines as $discipline) {
                        $sql = new DbQuery();
                        $sql->select('ip.item_pricing_id');
                        $sql->from('item_pricing', 'ip');
                        $sql->where('ip.discipline_id = ' . (int) $discipline['discipline_id'] );
                        $sql->orderBy('ip.task_size_id ASC');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                        foreach($result as $itemPricing) {
                            $itemPricing = new ItemPricing( (int) $itemPricing['item_pricing_id'] );
                            if(Validate::isLoadedObject($itemPricing)) {
                                $taskSize = ArrayUtils::get($globals['taskSizes'], $itemPricing->task_size_id);

                                $itemPricesByDiscipline[] = array(
                                    'item_pricing_id' => (int) $itemPricing->id,
                                    'discipline_id' => (int) $discipline['discipline_id'],
                                    'discipline_title' => $discipline['title'],
                                    'task_size_id' => (int) $itemPricing->task_size_id,
                                    'task_size_title' => $taskSize['title'],
                                    'price' => (float) $itemPricing->price,
                                    'priceFormatted' => formatPrice($itemPricing->price)
                                );
                            }
                        }
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);

                }
                
                $itemPricesByDiscipline = ArrayUtils::groupByKey($itemPricesByDiscipline, 'discipline_id');

                $smarty->assign([
                    'itemPricesByDiscipline' => $itemPricesByDiscipline,
                    'taskSizes' => $globals['taskSizes'],
                    'disciplines' => $disciplines,
                    'controller' => 'pricing'
                ]);
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'preferences',
            'sub_view' => 'programming'
        ]);

        $this->setTemplate('dashboard/preferences/programming/customize');
        $this->display(); 
    }

    public function calculations( $sub_view = 'disciplines' ) {
        global $globals;

        $app = $this->app; 
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin || !Module::isInstalled('calculations')) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::get($params, 'page', 1);

        switch($sub_view) {
            case 'disciplines':
                $this->setVars([
                    'page_title' => "Preferences › Disciplines"
                ]);
        
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_calculations = 1');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_disciplines = count($result);
        
                $disciplines = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/calculations/disciplines?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($result as $discipline) {
                        $disciplines[] = array(
                            'discipline_id' => (int) $discipline['discipline_id'],
                            'group_id' => (int) $discipline['group_id'],
                            'title' => $discipline['title'],
                            'is_group' => (bool) $discipline['is_group'],
                            'is_complex_assignment' => (bool) $discipline['is_complex_assignment'],
                            'is_academic_writing' => (bool) $discipline['is_academic_writing'],
                            'is_calculations' => (bool) $discipline['is_calculations'],
                            'is_programming' => (bool) $discipline['is_programming'],
                            'is_article_writing' => (bool) $discipline['is_article_writing']
                        );
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $disciplineGroups = Discipline::getDisciplinesByGroup();
        
                $smarty->assign([
                    'controller' => 'disciplines',
                    'disciplines' => $disciplines,
                    'disciplineGroups' => $disciplineGroups
                ]);
                break;

            case 'descriptors':
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_calculations = 1');
                $sql->orderBy('d.title ASC');
                $disciplines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_disciplines = count($disciplines);

                $descriptorsByDiscipline = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/calculations/descriptors?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $disciplines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($disciplines as $discipline) {
                        $sql = new DbQuery();
                        $sql->select('d.descriptor_id, d.discipline_id, d.task_size_id, d.example');
                        $sql->from('descriptor', 'd');
                        $sql->where('d.discipline_id = ' . (int) $discipline['discipline_id']);
                        $sql->where('d.is_calculations = 1');
                        $sql->orderBy('d.task_size_id ASC');
                        $descriptors = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                        $total_descriptors = count($descriptors);

                        foreach($descriptors as $descriptor) {
                            $descriptor = new Descriptor( (int) $descriptor['descriptor_id'] );
                            if(Validate::isLoadedObject($descriptor)) {
                                $taskSize = ArrayUtils::get($globals['taskSizes'], $descriptor->task_size_id);
                                $deliverables = $descriptor->getDeliverables();
                                
                                $descriptorsByDiscipline[] = array(
                                    'descriptor_id' => $descriptor->id,
                                    'discipline_title' => $discipline['title'],
                                    'discipline_id' => $descriptor->discipline_id,
                                    'task_size_id' => $descriptor->task_size_id,
                                    'task_size_title' => $taskSize['title'],
                                    'example' => $descriptor->example,
                                    'deliverables' => $deliverables
                                );
                            }
                        }
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $deliverables = Deliverable::getDeliverables();

                $this->setVars([
                    'page_title' => "Preferences › Descriptors"
                ]);

                $descriptorsByDiscipline = ArrayUtils::groupByKey($descriptorsByDiscipline, 'discipline_id');

                $smarty->assign([
                    'controller' => 'descriptors',
                    'descriptorsByDiscipline' => $descriptorsByDiscipline,
                    'taskSizes' => $globals['taskSizes'],
                    'disciplines' => $disciplines,
                    'deliverables' => $deliverables
                ]);
                break;

            case 'pricing':
                $this->setVars([
                    'page_title' => "Preferences › Pricing"
                ]);

                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_calculations = 1');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_disciplines = count($result);

                $disciplines = $itemPricesByDiscipline = array();
                if( $total_disciplines > 0) {
                    $params['total_items'] = $total_disciplines;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/calculations/pricing?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $disciplines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($disciplines as $discipline) {
                        $sql = new DbQuery();
                        $sql->select('ip.item_pricing_id');
                        $sql->from('item_pricing', 'ip');
                        $sql->where('ip.discipline_id = ' . (int) $discipline['discipline_id'] );
                        $sql->orderBy('ip.task_size_id ASC');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                        foreach($result as $itemPricing) {
                            $itemPricing = new ItemPricing( (int) $itemPricing['item_pricing_id'] );
                            if(Validate::isLoadedObject($itemPricing)) {
                                $taskSize = ArrayUtils::get($globals['taskSizes'], $itemPricing->task_size_id);

                                $itemPricesByDiscipline[] = array(
                                    'item_pricing_id' => (int) $itemPricing->id,
                                    'discipline_id' => (int) $discipline['discipline_id'],
                                    'discipline_title' => $discipline['title'],
                                    'task_size_id' => (int) $itemPricing->task_size_id,
                                    'task_size_title' => $taskSize['title'],
                                    'price' => (float) $itemPricing->price,
                                    'priceFormatted' => formatPrice($itemPricing->price)
                                );
                            }
                        }
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);

                }
                
                $itemPricesByDiscipline = ArrayUtils::groupByKey($itemPricesByDiscipline, 'discipline_id');

                $smarty->assign([
                    'itemPricesByDiscipline' => $itemPricesByDiscipline,
                    'taskSizes' => $globals['taskSizes'],
                    'disciplines' => $disciplines,
                    'controller' => 'pricing'
                ]);
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'preferences',
            'sub_view' => 'calculations'
        ]);

        $this->setTemplate('dashboard/preferences/calculations/customize');
        $this->display(); 
    }

    public function articleWriting( $sub_view = 'content-types' ) {
        $app = $this->app; 
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        if(!$user->is_admin || !Module::isInstalled('articlewriting')) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::get($params, 'page', 1);

        switch($sub_view) {
            case 'content-types':
                $this->setVars([
                    'page_title' => "Preferences › Paper Types"
                ]);
        
                $sql = new DbQuery();
                $sql->select('pt.*');
                $sql->from('paper_type', 'pt');
                $sql->where('pt.is_article_writing = 1');
                $sql->orderBy('pt.paper_type_id ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_content_types = count($result);
        
                $contentTypes = array();
                if( $total_content_types > 0) {
                    $params['total_items'] = $total_content_types;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/articles/content-types?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $contentTypes = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $smarty->assign([
                    'controller' => 'contentTypes',
                    'contentTypes' => $contentTypes
                ]);
                break;

            case 'categories':
                $this->setVars([
                    'page_title' => "Preferences › Categories"
                ]);
        
                $sql = new DbQuery();
                $sql->select('d.*');
                $sql->from('discipline', 'd');
                $sql->where('d.is_article_writing = 1');
                $sql->orderBy('d.title ASC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        
                $total_categories = count($result);
        
                $categories = array();
                if( $total_categories > 0) {
                    $params['total_items'] = $total_categories;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/preferences/articles/categories?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
                    foreach($result as $category) {
                        $categories[] = array(
                            'discipline_id' => (int) $category['discipline_id'],
                            'group_id' => (int) $category['group_id'],
                            'title' => $category['title'],
                            'is_group' => (bool) $category['is_group'],
                            'is_complex_assignment' => (bool) $category['is_complex_assignment'],
                            'is_academic_writing' => (bool) $category['is_academic_writing'],
                            'is_calculations' => (bool) $category['is_calculations'],
                            'is_programming' => (bool) $category['is_programming'],
                            'is_article_writing' => (bool) $category['is_article_writing']
                        );
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
        
                $categoryGroups = Discipline::getDisciplinesByGroup( Order::ARTICLE_WRITING );

                $smarty->assign([
                    'controller' => 'categories',
                    'categories' => $categories,
                    'categoryGroups' => $categoryGroups
                ]);
                break;

            case 'other':
                $this->setVars([
                    'page_title' => "Preferences › Other"
                ]);

                # englishTypes
                $englishTypes = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT et.* FROM '. Db::prefix('english_type') . ' AS et ORDER BY et.title ASC');

                #targetAudiences
                $targetAudiences = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT ta.* FROM '. Db::prefix('target_audience') . ' AS ta ORDER BY ta.title ASC');

                #writingTones
                $writingTones = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT wt.* FROM '. Db::prefix('writing_tone') . ' AS wt ORDER BY wt.title ASC');

                #writingStyles
                $writingStyles = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT ws.* FROM '. Db::prefix('writing_style') . ' AS ws ORDER BY ws.title ASC');

                #contentFeels
                $contentFeels = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT cf.* FROM '. Db::prefix('content_feel') . ' AS cf ORDER BY cf.title ASC');

                $smarty->assign([
                    'englishTypes' => $englishTypes,
                    'targetAudiences' => $targetAudiences,
                    'writingTones' => $writingTones,
                    'writingStyles' => $writingStyles,
                    'contentFeels' => $contentFeels,
                    'controller' => 'other'
                ]);
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'view' => 'preferences',
            'sub_view' => 'articles'
        ]);

        $this->setTemplate('dashboard/preferences/articles/customize');
        $this->display(); 
    }
}