<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Pager;
use Proxim\Currency;
use Proxim\Coupon;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Csv\CSV;
use Proxim\Csv\CsvResponse;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\Discipline;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\User\EmployeeFile;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Users extends Controller {
    public function customers( $sub_view = "", $customer_id = null ) {
        global $globals;
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $query = ArrayUtils::has($params, 'query') ? ArrayUtils::get($params, 'query') : false;
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $this->setVars([
            'page_title' => "Customers"
        ]);

        switch( $sub_view ) {
            case '':
                $sql = new DbQuery();
                $sql->select('c.customer_id');
                $sql->from('customer', 'c');

                if($query) {
                    $sql->where('
                        c.customer_id LIKE \'%' . pSQL($query) . '%\' OR
                        c.name LIKE \'%' . pSQL($query) . '%\' OR
                        c.email LIKE \'%' . pSQL($query) . '%\' OR 
                        c.site_id LIKE \'%' . pSQL($query) . '%\'
                    ');
                }

                $sql->orderBy('c.customer_id DESC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
            
                $customers = array();
                $total_customers = count($result);
        
                if( $total_customers > 0) {
                    $params['total_items'] = $total_customers;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/customers?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);
        
                    foreach( $result as $customer ) {
                        $customer = new Customer( (int) $customer['customer_id'] );
                        if( Validate::isLoadedObject($customer) ) {
                            $customers[] = (array) $customer;
                        }
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'total_customers' => $total_customers,
                    'customers' => $customers,
                ]);
                break;  

            case 'add':

                $sites = new DbQuery();
                $sites->select('s.*');
                $sites->from('site', 's');
                $sites = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sites );
        
                $smarty->assign([
                    'sites' => $sites
                ]);

                break;
            
            case 'edit':
                $customer = new Customer( (int) $customer_id );
                if( !Validate::isLoadedObject($customer) ) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $completedOrders = Db::getInstance()->getValue('SELECT COUNT(order_id) FROM ' . Db::prefix('order') . ' WHERE is_paid = 1 AND customer_id = ' . (int) $customer->id );
		        $customer->completed_orders = toFixed($completedOrders, 0);

                $isWalletEnabled = $isReferralEnabled = false;
                if(Module::isInstalled('wallet')) {
                    $isWalletEnabled = true;

                    $walletModule = Module::getInstanceByName('wallet');
                    $customerWallet = $walletModule->getWalletByCustomerId( $customer->id );
                    $smarty->assign('customerWallet', $customerWallet);
                }

                if(Module::isInstalled('rewards_program')) {
                    $isReferralEnabled = true;

                    $invitedFriends = Db::getInstance()->getValue('SELECT COUNT(customer_id) FROM ' . Db::prefix('customer') . ' WHERE referrer_id = ' . (int) $customer->id );
                    $customer->friends_invited = toFixed($invitedFriends, 0);
                }

                $sites = new DbQuery();
                $sites->select('s.*');
                $sites->from('site', 's');
                $sites = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sites );
                
                $site = new Site( (int) $customer->site_id );

                $smarty->assign([
                    'isReferralEnabled' => $isReferralEnabled,
                    'isWalletEnabled' => $isWalletEnabled,
                    'customer' => (array) $customer,
                    'customer_site' => (array) $site,
                    'sites' => $sites
                ]);

                break;

            case 'export':
                $sql = new DbQuery();
                $sql->select('c.*');
                $sql->from('customer', 'c');
                $sql->where('c.is_anonymous = 0');
                $sql->orderBy('c.customer_id DESC');
                $customers = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $data = [];

                // phone number parsing
                $phoneUtil = PhoneNumberUtil::getInstance();
                $countriesByName = $globals['countries'];
                $countriesByCode = array();
                foreach($countriesByName as $countryCode => $country) {
                    $country['countryCode'] = $countryCode;
                    $countriesByCode[$country['code']] = $country;
                }
        
                foreach ($customers as $customer) {
                    if(!$customer['country_id']) {
                        $customer['country_id'] = '840';
                    }
    
                    $customerPhone = $customer['phone'];
                    $country = ArrayUtils::get($countriesByCode, $customer['country_id'], array());
                    if( is_array($country) && ArrayUtils::has($country, 'countryCode')) {
                        $phoneCode = ArrayUtils::get($country, 'countryCode');
                        try {
                            $phoneNumberProto = $phoneUtil->parse($customer['phone'], strtoupper($phoneCode) );
                            $customerPhone = $phoneUtil->format($phoneNumberProto, PhoneNumberFormat::INTERNATIONAL);
                        } catch (\libphonenumber\NumberParseException $e) {
                        }
                    }

                    $data[] = (object) [
                        'customer_id' => $customer['customer_id'],
                        'name' => $customer['name'],
                        'email' => $customer['email'],
                        'phone' => $customerPhone,
                        'website' => Configuration::get('SITE_NAME', $customer['site_id']),
                        'total_spent' => formatPrice($customer['total_spent']),
                        'registered' => DateUtils::date(strtotime($customer['reg_date']))
                    ];
                }

                $csv = new CSV($data, 'customers_' . date('Y-m-d_His') . '.csv', ',');
                return $csv->export();
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'query' => $query,
            'view' => 'users',
            'sub_view' => "customers",
            'section' => $sub_view,
        ]);

        $this->setTemplate('dashboard/users/customers');
        $this->display(); 
    }

    public function employees( $sub_view = "", $employee_id = null ) {
        global $globals;
        $app = $this->app;
        $params = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        if( !$user->is_admin && !$user->is_sub_admin ) {
            $this->setTemplate('404');
            $this->display();
        }
 
        if($user->is_sub_admin && !Configuration::get('SUBADMIN_WRITER_INFO')) {
            $this->setTemplate('404');
            $this->display();
        }

        $_tab = ArrayUtils::get($params, '_tab', 'info');
        $query = ArrayUtils::has($params, 'query') ? ArrayUtils::get($params, 'query') : false;
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $this->setVars([
            'page_title' => "Employees"
        ]); 

        $countries = $globals['countries'];

        switch( $sub_view ) {
            case 'writers':
            case 'editors':
            case 'sub-admins':
            case 'administrators':
                $sql = new DbQuery();
                $sql->select('e.employee_id');
                $sql->from('employee', 'e');

                if ( $sub_view == "writers" ) {
                    $sql->where('employee_group = ' . (int) Employee::GROUP_WRITER);
                } elseif ( $sub_view == "editors" ) {
                    $sql->where('employee_group = ' . (int) Employee::GROUP_EDITOR);
                } elseif ( $sub_view == "sub-admins" ) {
                    $sql->where('employee_group = ' . (int) Employee::GROUP_SUB_ADMIN);
                } elseif ( $sub_view == "administrators" ) {
                    $sql->where('employee_group = ' . (int) Employee::GROUP_ADMIN);
                }

                if($query) {
                    $sql->where('
                        e.employee_id LIKE \'%' . pSQL($query) . '%\' OR
                        e.first_name LIKE \'%' . pSQL($query) . '%\' OR
                        e.last_name LIKE \'%' . pSQL($query) . '%\' OR
                        e.email LIKE \'%' . pSQL($query) . '%\'
                    ');
                }

                $sql->orderBy('e.employee_id DESC');
                
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
            
                $employees = array();
                $total_employees = count($result);
        
                if( $total_employees > 0) {
                    $params['total_items'] = $total_employees;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/employees/'.$sub_view.'?page=%s';
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
        
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);
        
                    foreach( $result as $employee ) {
                        $employee = new Employee( (int) $employee['employee_id'] );
                        if( Validate::isLoadedObject($employee) ) {
                            $employees[] = (array) $employee;
                        }
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'total_employees' => $total_employees,
                    'employees' => $employees,
                ]);
                break;  

            case 'add':

                break;
            
            case 'edit': 
                $employee = new Employee( (int) $employee_id );
                if( !Validate::isLoadedObject($employee) ) {
                    $this->setTemplate('404');
                    $this->display();
                }

                if($employee->is_admin && !$user->is_admin) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $user_country = ArrayUtils::get($countries, $employee->country);

                $chart = $insights = array();
                $months = array();

                for ($i = 0; $i <= 11; $i++) {
                    $months[$i] = array(
                        'y' => date("Y", strtotime( date( 'Y-m-01' )." -$i months")),
                        'month' => date("M-Y", strtotime( date( 'Y-m-01' )." -$i months")),
                        'm' => date("m", strtotime( date( 'Y-m-01' )." -$i months"))
                    );
                }

                if($employee->is_writer || $employee->is_editor) {

                    $chart['totalIncome'] = $chart['totalOrders'] = 0;

                    foreach($months as $key => $month) {
                        /* get income */
                        $sql = new DbQuery();
                        $sql->from("order");

                        if($employee->is_writer) {
                            $sql->select("COALESCE(SUM(writer_pay),0) amount");
                            $sql->select("COUNT(order_id) orders");
                            $sql->where("writer_id = " . (int) $employee->id);
                            $sql->where("is_writer_paid = 1");
                        } else {
                            $sql->select("COALESCE(SUM(editor_pay),0) amount");
                            $sql->select("COUNT(order_id) orders");
                            $sql->where("editor_id = " . (int) $employee->id);
                            $sql->where("is_editor_paid = 1");
                        }
                        
                        $sql->where("YEAR(date_add) = " . $month['y'] . " AND MONTH(date_add) = " . (int) $month['m']);
                        $result = Db::getInstance()->getRow($sql);

                        $chart['totalIncome'] += $result['amount'];
                        $chart['totalOrders'] += $result['orders'];
                        $chart['income'][$key] = $result['amount'];
                    }

                    /** performance overview */
                    $sql = new DbQuery();
                    $sql->select("COUNT(DISTINCT(oe.order_id))")->from("order_employee", "oe");
                    $sql->where("employee_id = " . (int) $employee->id );

                    $revision_orders_query = clone $sql;
                    $dispute_orders_query = clone $sql;
                    $refund_orders_query = clone $sql;

                    $revision_orders_query->where('is_revised = 1');
                    $insights['total_orders_revisions'] = Db::getInstance()->getValue($revision_orders_query);

                    $dispute_orders_query->where('is_disputed = 1');
                    $insights['total_orders_disputes'] = Db::getInstance()->getValue($dispute_orders_query);

                    $refund_orders_query->where('is_refunded = 1');
                    $insights['total_orders_refunds'] = Db::getInstance()->getValue($refund_orders_query);
                }

                $mapMonths = ArrayUtils::groupByKey($months, 'month');

                $mapMonths = array_reverse(array_keys($mapMonths));

                $chart['mapMonths'] = $mapMonths;
                $chart['chartLabels'] = '"' . implode('","', $mapMonths) . '"';

                $sql = new DbQuery();
                $sql->select('session.*');
                $sql->from('user_session', 'session');
                $sql->where('session.user_id = ' . (int) $employee->id );
                $sql->orderBy('session.session_id DESC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $total_sessions = count($result);
                $sessions = array();
                if( $total_sessions > 0) {
                    $params['total_items'] = $total_sessions;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = '/employees/edit/'.$employee->id.'?_tab=sessions&page=%s';

                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    foreach($result as $session) {
                        $session['login_time'] = DateUtils::convertUtcDateTimeToTimeZone($session['date_add'], $user->timezone);
                        $sessions[] = $session;
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
                
                $sql = new DbQuery();
                $sql->select('ef.employee_file_id');
                $sql->from('employee_file', 'ef');
                $sql->where('ef.employee_id = ' . (int) $employee->id );
                $sql->orderBy('ef.employee_file_id DESC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                $objectPresenter = new ObjectPresenter();

                $employee_files = array();
                foreach($result as $file) {
                    $file = new EmployeeFile( (int) $file['employee_file_id'] );
                    if(Validate::isLoadedObject($file)) {
                        $presentedFile = $objectPresenter->present($file);
                        $presentedFile['download_link'] = $app->base_uri . '/api/employee-files/'.$file->id.'/download';
                        $presentedFile['size_formatted'] = formatBytes($file->size);
                        $employee_files[] = $presentedFile;
                    }
                }

                # employee files
                $employee_files = ArrayUtils::groupByKey($employee_files, 'handle');

                # user disciplines
                $user_disciplines = array();
                foreach($employee->proficient_disciplines_array as $discipline_id) {
                    $discipline = new Discipline((int) $discipline_id);
                    if(Validate::isLoadedObject($discipline)) {
                        $user_disciplines[] = [
                            'id' => $discipline->id,
                            'title' => $discipline->title
                        ];
                    }
                }

                # user software & programming
                $user_softwares = array();
                if(!empty($employee->software_array)) {
                    $user_softwares = Db::getInstance()->executeS('SELECT * FROM ' . DB::prefix('software_programming') . ' WHERE software_programming_id IN ('.implode(',', $employee->software_array).')');
                }

                # softwares & programming
                $sql = new DbQuery();
                $sql->select('sp.*');
                $sql->from('software_programming', 'sp');
                $sql->orderBy('sp.software_programming_id DESC');
                $software_programming = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                // languages
                $languages = languages();

                $employeeAbout = Tools::getStringClean($employee->about);
                $employeeAbout = Tools::nl2br($employeeAbout);

                $smarty->assign([
                    'employee' => (array) $employee,
                    'employeeAbout' => $employeeAbout,
                    'user_country' => $user_country,
                    'insights' => $insights,
                    'chart' => $chart,
                    'languages' => $languages,
                    'outstanding_payment' => $employee->getOutstanding(),
                    'total_sessions' => $total_sessions,
                    'employee_files' => $employee_files,
                    'sessions' => $sessions,
                    'user_disciplines' => $user_disciplines,
                    'user_softwares' => $user_softwares,
                    'software_programming' => $software_programming
                ]);

                break;
            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        // languages
        $languages = languages();

        // disciplines
        $proficientDisciplinesIds = explode(',', $user->proficient_disciplines);
        
        $disciplineGroups = Discipline::getDisciplinesByGroup();

        $currencyFormat = $app->writer_currency;

        $academicLevels = AcademicLevel::getAcademicLevels();
        
        $smarty->assign([
            'query' => $query,
            'view' => 'users',
            'sub_view' => $sub_view,
            '_tab' => $_tab,
            'languages' => $languages,
            'disciplineGroups' => $disciplineGroups,
            'academicDegrees' => $academicLevels,
            'countries' => $countries,
            'currencyFormat' => $currencyFormat
        ]);

        $this->setTemplate('dashboard/users/employees');
        $this->display(); 
    }
} 